#!/usr/bin/env node

/**
 * Generate WordPress .asset.php files from Vite manifest.json
 *
 * This script reads the Vite build manifest and creates WordPress-compatible
 * .asset.php files for each entry point, allowing WordPress to properly
 * enqueue the bundled assets with correct dependencies.
 */

const fs = require('fs');
const path = require('path');

const MANIFEST_PATH = path.join(__dirname, '../build/manifest.json');
const BUILD_DIR = path.join(__dirname, '../build');

// Entry points defined in vite.config.mjs
const ENTRY_POINTS = [
  'src/admin/index.jsx',
  'src/editor/index.js',
  'src/admin/onboarding/index.jsx',
  'src/editor/meta-box/index.jsx',
  'src/editor/sidebar/index.jsx',
];

// Map entry source paths to output names
const ENTRY_NAMES = {
  'src/admin/index.jsx': 'admin',
  'src/editor/index.js': 'editor',
  'src/admin/onboarding/index.jsx': 'onboarding',
  'src/editor/meta-box/index.jsx': 'meta-box',
  'src/editor/sidebar/index.jsx': 'sidebar',
};

// WordPress dependencies that should be loaded before our scripts
const WP_DEPENDENCIES = [
  'wp-element',
  'wp-i18n',
  'wp-api-fetch',
  'wp-data',
  'wp-hooks',
  'wp-components',
  'wp-block-editor',
  'wp-blocks',
  'wp-dom-ready',
  'react',
  'react-dom',
];

function generateAssetPHP() {
  console.log('ProRank SEO - Asset PHP Generator');
  console.log('==================================\n');

  // Read manifest
  if (!fs.existsSync(MANIFEST_PATH)) {
    console.error(`❌ Manifest not found: ${MANIFEST_PATH}`);
    process.exit(1);
  }

  const manifest = JSON.parse(fs.readFileSync(MANIFEST_PATH, 'utf8'));
  let generated = 0;

  // Generate .asset.php for each entry point
  ENTRY_POINTS.forEach((entrySource) => {
    const entryName = ENTRY_NAMES[entrySource];

    if (!entryName) {
      console.warn(`⚠️  No entry name defined for: ${entrySource}`);
      return;
    }

    const manifestEntry = manifest[entrySource];

    if (!manifestEntry) {
      console.warn(`⚠️  Entry not found in manifest: ${entrySource}`);
      return;
    }

    const jsFile = manifestEntry.file;

    if (!jsFile) {
      console.warn(`⚠️  No JS file in manifest for: ${entrySource}`);
      return;
    }

    // Get file hash from the output filename for versioning, or use timestamp
    const parts = jsFile.replace(/\.(js|css)$/, '').split('-');
    const fileHash = parts.length > 1 ? parts.pop() : Date.now().toString(36);

    // Collect all imports to determine dependencies
    const imports = manifestEntry.imports || [];
    const dependencies = [...WP_DEPENDENCIES];

    // Generate PHP content with ABSPATH check
    const phpContent = `<?php
defined( 'ABSPATH' ) || exit;
return array(
  'dependencies' => array(${dependencies.map(dep => `'${dep}'`).join(', ')}),
  'version' => '${fileHash}'
);
`;

    // Write .asset.php file
    const assetPhpPath = path.join(BUILD_DIR, `${entryName}.asset.php`);
    fs.writeFileSync(assetPhpPath, phpContent);

    console.log(`✓ Generated: ${entryName}.asset.php`);
    generated++;
  });

  console.log(`\n✅ Generated ${generated} asset PHP files successfully!`);
}

// Run the generator
try {
  generateAssetPHP();
} catch (error) {
  console.error('❌ Error generating asset PHP files:', error.message);
  process.exit(1);
}
