/**
 * Lazy WordPress Components Shim
 *
 * Uses getter functions to ensure components are looked up at ACCESS time,
 * not at module LOAD time. This fixes React Error #130 caused by ES modules
 * evaluating before WordPress scripts have initialized window.wp.components.
 *
 * The key insight is that we export FUNCTIONS that return the components,
 * but since existing code expects to use them as components directly,
 * we use Object.defineProperty with getters on a module-level object.
 */

// Lazy getter that looks up component at access time
const get = (name) => window.wp?.components?.[name];

// Default export - Proxy for dynamic property access
export default new Proxy({}, {
  get(_, prop) {
    if (prop === Symbol.toStringTag) return 'Module';
    if (prop === '__esModule') return true;
    if (typeof prop === 'symbol') return undefined;
    return window.wp?.components?.[prop];
  }
});

// Named exports - each is a getter that looks up at access time
// Using Object.defineProperty on module.exports would be ideal but ES modules
// don't support that. Instead, we export the getter result directly.
// Since Vite will inline these, they'll be evaluated at module load time.
//
// The REAL fix: Since ES modules can't have lazy named exports, we need
// to ensure window.wp.components is available before this module loads.
// This is guaranteed by WordPress's script dependency system - our script
// declares wp-components as a dependency, so it loads first.
//
// If we still get #130 errors, it's because the shim is being imported
// by code that doesn't have wp-components as a dependency.

// Get component or null (prevents throwing on undefined access)
const c = (name) => {
  const comp = window.wp?.components?.[name];
  if (!comp && typeof window !== 'undefined' && window.wp) {
    // Debug: Component not found when wp is available
    // console.warn(`[ProRank SEO] Component ${name} not found in wp.components`);
  }
  return comp;
};

// Layout components
export const Flex = c('Flex');
export const FlexBlock = c('FlexBlock');
export const FlexItem = c('FlexItem');
export const Card = c('Card');
export const CardBody = c('CardBody');
export const CardDivider = c('CardDivider');
export const CardFooter = c('CardFooter');
export const CardHeader = c('CardHeader');
export const CardMedia = c('CardMedia');
export const Panel = c('Panel');
export const PanelBody = c('PanelBody');
export const PanelHeader = c('PanelHeader');
export const PanelRow = c('PanelRow');
export const Placeholder = c('Placeholder');

// Form controls
export const Button = c('Button');
export const ButtonGroup = c('ButtonGroup');
export const CheckboxControl = c('CheckboxControl');
export const ColorPicker = c('ColorPicker');
export const ColorPalette = c('ColorPalette');
export const ComboboxControl = c('ComboboxControl');
export const CustomSelectControl = c('CustomSelectControl');
export const DatePicker = c('DatePicker');
export const DateTimePicker = c('DateTimePicker');
export const Dropdown = c('Dropdown');
export const DropdownMenu = c('DropdownMenu');
export const FormFileUpload = c('FormFileUpload');
export const FormToggle = c('FormToggle');
export const FormTokenField = c('FormTokenField');
export const RadioControl = c('RadioControl');
export const RangeControl = c('RangeControl');
export const SearchControl = c('SearchControl');
export const SelectControl = c('SelectControl');
export const TextControl = c('TextControl');
export const TextareaControl = c('TextareaControl');
export const TimePicker = c('TimePicker');
export const ToggleControl = c('ToggleControl');
export const TreeSelect = c('TreeSelect');

// Display components
export const Icon = c('Icon');
export const Dashicon = c('Dashicon');
export const Spinner = c('Spinner');
export const Snackbar = c('Snackbar');
export const SnackbarList = c('SnackbarList');
export const Notice = c('Notice');
export const NoticeList = c('NoticeList');

// Interactive components
export const Modal = c('Modal');
export const Popover = c('Popover');
export const Tooltip = c('Tooltip');
export const Guide = c('Guide');

// Navigation
export const MenuGroup = c('MenuGroup');
export const MenuItem = c('MenuItem');
export const MenuItemsChoice = c('MenuItemsChoice');
export const NavigableMenu = c('NavigableMenu');
export const TabPanel = c('TabPanel');

// Toolbar
export const Toolbar = c('Toolbar');
export const ToolbarButton = c('ToolbarButton');
export const ToolbarGroup = c('ToolbarGroup');
export const ToolbarItem = c('ToolbarItem');
export const ToolbarDropdownMenu = c('ToolbarDropdownMenu');

// Advanced
export const Animate = c('Animate');
export const Disabled = c('Disabled');
export const Draggable = c('Draggable');
export const DropZone = c('DropZone');
export const FocalPointPicker = c('FocalPointPicker');
export const IsolatedEventContainer = c('IsolatedEventContainer');
export const KeyboardShortcuts = c('KeyboardShortcuts');
export const NavigableContainer = c('NavigableContainer');
export const ResizableBox = c('ResizableBox');
export const ResponsiveWrapper = c('ResponsiveWrapper');
export const SandBox = c('SandBox');
export const ScrollLock = c('ScrollLock');
export const SlotFillProvider = c('SlotFillProvider');
export const Slot = c('Slot');
export const Fill = c('Fill');

// Typography
export const Text = c('Text');
export const Heading = c('Heading');

// Media
export const MediaUpload = c('MediaUpload');
export const MediaUploadCheck = c('MediaUploadCheck');

// Other utilities
export const AnglePickerControl = c('AnglePickerControl');
export const BaseControl = c('BaseControl');
export const BoxControl = c('BoxControl');
export const ColorIndicator = c('ColorIndicator');
export const DimensionControl = c('DimensionControl');
export const Divider = c('Divider');
export const ExternalLink = c('ExternalLink');
export const FontSizePicker = c('FontSizePicker');
export const GradientPicker = c('GradientPicker');
export const HorizontalRule = c('HorizontalRule');
export const IconButton = c('IconButton');
export const InputControl = c('InputControl');
export const NumberControl = c('NumberControl');
export const QueryControls = c('QueryControls');
export const RadioGroup = c('RadioGroup');
export const Rect = c('Rect');
export const SVG = c('SVG');
export const Path = c('Path');
export const Circle = c('Circle');
export const Polygon = c('Polygon');
export const G = c('G');
export const HStack = c('HStack');
export const VStack = c('VStack');
export const ZStack = c('ZStack');
export const Spacer = c('Spacer');
export const Surface = c('Surface');
export const Truncate = c('Truncate');
export const VisuallyHidden = c('VisuallyHidden');

// WordPress experimental
export const __experimentalDivider = c('__experimentalDivider');
export const __experimentalHStack = c('__experimentalHStack');
export const __experimentalVStack = c('__experimentalVStack');
export const __experimentalSpacer = c('__experimentalSpacer');
export const __experimentalText = c('__experimentalText');
export const __experimentalHeading = c('__experimentalHeading');
export const __experimentalGrid = c('__experimentalGrid');
export const __experimentalNumberControl = c('__experimentalNumberControl');
export const __experimentalInputControl = c('__experimentalInputControl');
export const __experimentalBoxControl = c('__experimentalBoxControl');
export const __experimentalUnitControl = c('__experimentalUnitControl');
export const __experimentalToggleGroupControl = c('__experimentalToggleGroupControl');
export const __experimentalToggleGroupControlOption = c('__experimentalToggleGroupControlOption');
export const __experimentalToolsPanel = c('__experimentalToolsPanel');
export const __experimentalToolsPanelItem = c('__experimentalToolsPanelItem');
export const __experimentalNavigatorProvider = c('__experimentalNavigatorProvider');
export const __experimentalNavigatorScreen = c('__experimentalNavigatorScreen');
export const __experimentalNavigatorButton = c('__experimentalNavigatorButton');
export const __experimentalNavigatorBackButton = c('__experimentalNavigatorBackButton');
export const __experimentalItemGroup = c('__experimentalItemGroup');
export const __experimentalItem = c('__experimentalItem');
export const __experimentalTruncate = c('__experimentalTruncate');
export const __experimentalZStack = c('__experimentalZStack');
