/**
 * Readability Panel Component
 *
 * Displays Flesch-Kincaid readability score in the editor sidebar.
 *
 * @package
 */

import { __ } from '@wordpress/i18n';
import { useSelect } from '@wordpress/data';
import { useEffect, useState, useCallback, useMemo } from '@wordpress/element';
import { Spinner, Notice, Button, Card, CardBody, Modal, Flex } from '@wordpress/components';
import { Icon, starFilled, arrowRight, check, copy } from '@wordpress/icons';
import apiFetch from '@wordpress/api-fetch';
import { debounce } from 'lodash';

const ReadabilityPanel = ({ postId }) => {
  // State management
  const [score, setScore] = useState(null);
  const [level, setLevel] = useState(null);
  const [grade, setGrade] = useState(null);
  const [stats, setStats] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);

  // AI rewrite states
  const [showAISuggestions, setShowAISuggestions] = useState(false);
  const [aiSuggestions, setAiSuggestions] = useState([]);
  const [isLoadingAI, setIsLoadingAI] = useState(false);
  const [aiError, setAiError] = useState(null);
  const [copiedIndex, setCopiedIndex] = useState(null);

  // Get current post content from editor
  const postContent = useSelect((select) => {
    const { getEditedPostContent } = select('core/editor');
    return getEditedPostContent();
  }, []);

  // Fetch readability score from API
  const fetchReadabilityScore = useCallback(async (currentPostId) => {
    if (!currentPostId) return;

    setIsLoading(true);
    setError(null);

    try {
      const response = await apiFetch({
        path: `/prorank-seo/v1/readability-score/${currentPostId}`,
        method: 'GET',
      });

      if (response.success && response.data) {
        setScore(response.data.score);
        setLevel(response.data.level);
        setGrade(response.data.grade);
        setStats(response.data.stats);
      } else {
        throw new Error(response.message || __('Error fetching readability score', 'prorank-seo'));
      }
    } catch (err) {
      setError(err.message);
    } finally {
      setIsLoading(false);
    }
  }, []);

  // Debounced version of fetch function
  const debouncedFetch = useMemo(
    () =>
      debounce((currentPostId) => {
        fetchReadabilityScore(currentPostId);
      }, 800),
    [fetchReadabilityScore]
  );

  // Effect to fetch score when content changes
  useEffect(() => {
    if (postId && postContent) {
      debouncedFetch(postId);
    }

    // Cleanup
    return () => {
      debouncedFetch.cancel();
    };
  }, [postId, postContent, debouncedFetch]);

  // Get score color based on value
  const getScoreColor = (scoreValue) => {
    if (scoreValue >= 60) return 'success';
    if (scoreValue >= 30) return 'warning';
    return 'error';
  };

  // Get score indicator style
  const getScoreStyle = (scoreValue) => {
    const color = getScoreColor(scoreValue);
    const colorMap = {
      success: 'var(--pr-color-success)',
      warning: 'var(--pr-color-warning)',
      error: 'var(--pr-color-error)',
    };

    return {
      color: colorMap[color],
      fontWeight: 'var(--pr-font-weight-semibold)',
    };
  };

  // Get help text based on score
  const getHelpText = (scoreValue) => {
    if (scoreValue >= 60) {
      return __('Your content is easy to read. Great job!', 'prorank-seo');
    }
    if (scoreValue >= 30) {
      return __(
        'Your content readability could be improved. Consider using shorter sentences and simpler words.',
        'prorank-seo'
      );
    }
    return __(
      'Your content is difficult to read. Try breaking up long sentences and using simpler vocabulary.',
      'prorank-seo'
    );
  };

  // Fetch AI rewrite suggestions
  const fetchAISuggestions = useCallback(async () => {
    if (!postId) return;

    setIsLoadingAI(true);
    setAiError(null);
    setAiSuggestions([]);

    try {
      const response = await apiFetch({
        path: `/prorank-seo/v1/ai/suggest-readability-rewrites/${postId}`,
        method: 'POST',
      });

      if (response.success && response.data?.suggestions) {
        // Handle both array and object formats
        const suggestions = response.data.suggestions;
        if (Array.isArray(suggestions)) {
          setAiSuggestions(suggestions);
        } else if (typeof suggestions === 'object') {
          // Convert object to array format
          const suggestionsArray = Object.entries(suggestions).map(([key, value]) => {
            if (typeof value === 'object' && value.original && value.rewritten) {
              return value;
            }
            return { original: key, rewritten: value };
          });
          setAiSuggestions(suggestionsArray);
        }
      } else {
        setAiSuggestions([]);
      }
    } catch (err) {
      setAiError(err.message || __('Failed to get AI suggestions', 'prorank-seo'));
    } finally {
      setIsLoadingAI(false);
    }
  }, [postId]);

  // Copy suggestion to clipboard
  const copySuggestion = useCallback((text, index) => {
    navigator.clipboard.writeText(text).then(() => {
      setCopiedIndex(index);
      setTimeout(() => setCopiedIndex(null), 2000);
    });
  }, []);

  // Render content
  return (
    <div
      className="prorank-readability-panel"
      role="region"
      aria-label={__('Readability Analysis', 'prorank-seo')}
    >
      {isLoading && (
        <div className="prorank-readability-loading">
          <Spinner />
          <span className="screen-reader-text">
            {__('Calculating readability score…', 'prorank-seo')}
          </span>
        </div>
      )}

      {error && !isLoading && (
        <Notice status="error" isDismissible={false} className="prorank-readability-error">
          {error}
        </Notice>
      )}

      {!isLoading && !error && score !== null && (
        <div className="prorank-readability-results">
          {/* Compact Score Card */}
          <div className="prorank-readability-score-container">
            <div
              className={`prorank-readability-score-badge prorank-readability-score-badge--${getScoreColor(score)}`}
              aria-live="polite"
              aria-atomic="true"
            >
              {Math.round(score)}
            </div>
            <div className="prorank-readability-score-info">
              <p className="prorank-readability-score-title">
                {__('Readability', 'prorank-seo')}
              </p>
              <p className="prorank-readability-level-text">{level}</p>
              <p className="prorank-readability-grade-text">{grade}</p>
            </div>
          </div>

          {/* Stats Row */}
          {stats && (
            <div className="prorank-readability-stats-row">
              <span className="prorank-readability-stat">
                <span className="prorank-readability-stat-label">⏱</span>
                <span className="prorank-readability-stat-value">
                  {stats.estimated_reading_time} min read
                </span>
              </span>
            </div>
          )}

          {/* Help Box */}
          <div className={`prorank-readability-help ${score >= 60 ? 'prorank-readability-help--success' : ''}`}>
            <div className="prorank-readability-help-content">
              <p className="prorank-readability-help-text">{getHelpText(score)}</p>
              {score < 60 && (
                <Button
                  variant="secondary"
                  size="small"
                  icon={starFilled}
                  onClick={() => {
                    setShowAISuggestions(true);
                    fetchAISuggestions();
                  }}
                  className="prorank-ai-suggestions-button"
                >
                  {__('Get AI Suggestions', 'prorank-seo')}
                </Button>
              )}
            </div>
          </div>
        </div>
      )}

      {!isLoading && !error && score === null && postContent && (
        <Notice status="info" isDismissible={false} className="prorank-readability-no-content">
          {__('Add more content to see readability analysis.', 'prorank-seo')}
        </Notice>
      )}

      {!postContent && (
        <div className="prorank-readability-empty">
          <p>{__('Start writing to see your readability score.', 'prorank-seo')}</p>
        </div>
      )}

      {showAISuggestions && (
        <Modal
          title={__('AI Readability Suggestions', 'prorank-seo')}
          onRequestClose={() => setShowAISuggestions(false)}
          className="prorank-ai-readability-modal"
        >
          <div className="prorank-ai-modal-content">
            {isLoadingAI && (
              <div className="prorank-ai-loading">
                <Spinner />
                <p>{__('Analyzing your content for readability improvements…', 'prorank-seo')}</p>
              </div>
            )}

            {aiError && !isLoadingAI && (
              <Notice status="error" isDismissible={false}>
                {aiError}
              </Notice>
            )}

            {!isLoadingAI && !aiError && aiSuggestions.length > 0 && (
              <Flex direction="column" gap={4}>
                <p className="prorank-ai-description">
                  {__(
                    'Here are some AI-powered suggestions to improve your content readability:',
                    'prorank-seo'
                  )}
                </p>

                {aiSuggestions.map((suggestion, index) => (
                  <Card key={index} className="prorank-ai-suggestion-card">
                    <CardBody>
                      <Flex direction="column" gap={3}>
                        <div className="prorank-ai-original">
                          <strong>{__('Original:', 'prorank-seo')}</strong>
                          <p className="prorank-ai-text-original">{suggestion.original}</p>
                        </div>

                        <Icon icon={arrowRight} className="prorank-ai-arrow" />

                        <div className="prorank-ai-rewritten">
                          <strong>{__('Suggested:', 'prorank-seo')}</strong>
                          <p className="prorank-ai-text-suggested">{suggestion.rewritten}</p>
                        </div>

                        <Flex justify="flex-end">
                          <Button
                            variant="tertiary"
                            size="small"
                            icon={copiedIndex === index ? check : copy}
                            onClick={() => copySuggestion(suggestion.rewritten, index)}
                          >
                            {copiedIndex === index
                              ? __('Copied!', 'prorank-seo')
                              : __('Copy', 'prorank-seo')}
                          </Button>
                        </Flex>
                      </Flex>
                    </CardBody>
                  </Card>
                ))}
              </Flex>
            )}

            {!isLoadingAI && !aiError && aiSuggestions.length === 0 && (
              <Notice status="info" isDismissible={false}>
                {__(
                  'No complex sentences found that need simplification. Your content readability is already good!',
                  'prorank-seo'
                )}
              </Notice>
            )}
          </div>
        </Modal>
      )}
    </div>
  );
};

export default ReadabilityPanel;
