/* global sprintf, wp */
/**
 * Linking Suggestions Panel Component
 *
 * Displays internal linking suggestions in the editor sidebar.
 * Uses window.wp.* directly to avoid ES module shim timing issues.
 *
 * @package
 */

import { debounce } from 'lodash';

// Import styles
import './linking-suggestions-panel.css';

const LinkingSuggestionsPanel = ({ postId }) => {
  // Access WordPress APIs at render time to avoid shim timing issues
  const { useState, useEffect, useCallback, useMemo, useRef } = window.wp.element;
  const { __, sprintf } = window.wp.i18n;
  const { useSelect, useDispatch } = window.wp.data;
  const apiFetch = window.wp.apiFetch;
  const {
    Button,
    Card,
    CardBody,
    Spinner,
    Modal,
    TextControl,
  } = window.wp.components;
  // State for suggestions, loading, and error
  const [suggestions, setSuggestions] = useState([]);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const [linkingSettings, setLinkingSettings] = useState(null);
  const [insertedUrls, setInsertedUrls] = useState({});

  // State for anchor text editing modal
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [editingAnchorText, setEditingAnchorText] = useState('');
  const [pendingSuggestion, setPendingSuggestion] = useState(null);
  const lastSelectionRef = useRef({
    clientId: null,
    selectionStart: null,
    selectionEnd: null,
  });

  // Get current post content and selected block
  const { postContent, selectedBlockClientId, selectionStart, selectionEnd } = useSelect(
    (select) => {
      const { getEditedPostContent } = select('core/editor');
      const { getSelectedBlockClientId, getSelectionStart, getSelectionEnd } =
        select('core/block-editor');
      return {
        postContent: getEditedPostContent(),
        selectedBlockClientId: getSelectedBlockClientId(),
        selectionStart: getSelectionStart(),
        selectionEnd: getSelectionEnd(),
      };
    },
    []
  );

  // Get block editor dispatch functions
  const { insertBlocks, updateBlockAttributes } = useDispatch('core/block-editor');

  const getUrlVariants = useCallback((url) => {
    const raw = String(url || '').trim();
    if (!raw) {
      return [];
    }

    const variants = new Set();
    variants.add(raw);

    const withoutTrailing = raw.length > 1 ? raw.replace(/\/$/, '') : raw;
    variants.add(withoutTrailing);

    const origin = typeof window !== 'undefined' ? window.location.origin || '' : '';
    if (origin) {
      if (raw.startsWith(origin)) {
        const path = raw.slice(origin.length) || '/';
        variants.add(path);
        if (path.length > 1) {
          variants.add(path.replace(/\/$/, ''));
        }
      } else if (raw.startsWith('/')) {
        variants.add(`${origin}${raw}`);
      }
    }

    return Array.from(variants);
  }, []);

  const extractLinkedUrls = useCallback((content) => {
    if (!content) {
      return [];
    }
    const urls = new Set();
    const regex = /href=["']([^"']+)["']/gi;
    let match = null;
    while ((match = regex.exec(content)) !== null) {
      if (match[1]) {
        getUrlVariants(match[1]).forEach((variant) => {
          if (variant) {
            urls.add(variant);
          }
        });
      }
    }
    return Array.from(urls);
  }, [getUrlVariants]);

  useEffect(() => {
    let isMounted = true;
    apiFetch({ path: '/prorank-seo/v1/settings/internal-linking' })
      .then((response) => {
        if (!isMounted) {
          return;
        }
        if (response && response.success && response.data) {
          setLinkingSettings(response.data);
        }
      })
      .catch(() => {
        // Ignore settings fetch errors; fallback to defaults.
      });
    return () => {
      isMounted = false;
    };
  }, []);

  const findFirstEditableBlock = useCallback(() => {
    const { getBlocks } = wp.data.select('core/block-editor');
    const blocks = getBlocks() || [];
    for (const block of blocks) {
      const attributes = block?.attributes || {};
      for (const key of ['content', 'value', 'values']) {
        if (typeof attributes[key] === 'string') {
          return {
            clientId: block.clientId,
            contentKey: key,
            content: attributes[key],
          };
        }
      }
    }
    return null;
  }, []);

  const getContentAttributeKey = useCallback((block) => {
    const candidates = ['content', 'value', 'values'];
    for (const key of candidates) {
      if (typeof block.attributes?.[key] === 'string') {
        return key;
      }
    }
    return null;
  }, []);

  const buildAnchorRegex = useCallback((value) => {
    let decoded = value || '';
    // Decode HTML entities to match what is in the DOM text nodes
    if (typeof document !== 'undefined') {
      try {
        const txt = document.createElement('textarea');
        txt.innerHTML = decoded;
        decoded = txt.value;
      } catch (e) {
        // Fallback if DOM manipulation fails
      }
    }

    const trimmed = String(decoded).trim();
    if (!trimmed) {
      return null;
    }
    
    const tokens = trimmed
      .split(/\s+/)
      .filter(Boolean)
      .map((token) => token.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'));
    if (!tokens.length) {
      return null;
    }

    // Allow whitespace, punctuation, or inline tags between words.
    const joiner = "(?:[\\s\\u00A0]+|[\\-–—'\"’“”.,;:!?()]+|<[^>]+>)+";
    const pattern = tokens.join(joiner);

    return new RegExp(pattern, 'i');
  }, []);

  const insertLinkInHtml = useCallback((html, anchorText, url) => {
    if (!html || !anchorText) {
      return { html, inserted: false };
    }

    const matcher = buildAnchorRegex(anchorText);
    if (!matcher) {
      return { html, inserted: false };
    }

    const parser = typeof DOMParser !== 'undefined' ? new DOMParser() : null;
    if (!parser || !window.NodeFilter) {
      const match = matcher.exec(html);
      if (!match) {
        return { html, inserted: false };
      }
      const index = match.index;
      const matchText = match[0];
      const before = html.slice(0, index);
      const after = html.slice(index + matchText.length);
      return {
        html: `${before}<a href="${url}">${matchText}</a>${after}`,
        inserted: true,
      };
    }

    const doc = parser.parseFromString(`<div>${html}</div>`, 'text/html');
    const container = doc.body?.firstChild;
    if (!container) {
      return { html, inserted: false };
    }

    const textNodes = [];
    const walker = doc.createTreeWalker(container, window.NodeFilter.SHOW_TEXT);

    while (walker.nextNode()) {
      const node = walker.currentNode;
      const parent = node?.parentNode;
      if (!node?.nodeValue || !parent) {
        continue;
      }
      if (parent.nodeName && parent.nodeName.toLowerCase() === 'a') {
        continue;
      }
      textNodes.push(node);
    }

    if (textNodes.length) {
      let fullText = '';
      const map = [];
      for (const node of textNodes) {
        const start = fullText.length;
        const value = node.nodeValue;
        fullText += value;
        map.push({ node, start, end: start + value.length });
      }

      const match = matcher.exec(fullText);
      if (match) {
        const matchStart = match.index;
        const matchEnd = matchStart + match[0].length;
        let startNode = null;
        let endNode = null;
        let startOffset = 0;
        let endOffset = 0;

        for (const entry of map) {
          if (!startNode && matchStart >= entry.start && matchStart < entry.end) {
            startNode = entry.node;
            startOffset = matchStart - entry.start;
          }
          if (!endNode && matchEnd > entry.start && matchEnd <= entry.end) {
            endNode = entry.node;
            endOffset = matchEnd - entry.start;
            break;
          }
        }

        if (startNode && endNode) {
          const range = doc.createRange();
          range.setStart(startNode, startOffset);
          range.setEnd(endNode, endOffset);

          const fragment = range.cloneContents();
          if (!fragment.querySelector || !fragment.querySelector('a')) {
            const linkNode = doc.createElement('a');
            linkNode.setAttribute('href', url);
            const contents = range.extractContents();
            linkNode.appendChild(contents);
            range.insertNode(linkNode);
            return { html: container.innerHTML, inserted: true };
          }
        }
      }
    }

    let inserted = false;
    for (const node of textNodes) {
      if (inserted) {
        break;
      }
      const parent = node?.parentNode;
      if (!node?.nodeValue || !parent) {
        continue;
      }
      const text = node.nodeValue;
      const match = matcher.exec(text);
      if (!match) {
        continue;
      }

      const before = text.slice(0, match.index);
      const matchText = match[0];
      const after = text.slice(match.index + matchText.length);

      const linkNode = doc.createElement('a');
      linkNode.setAttribute('href', url);
      linkNode.textContent = matchText;

      const fragment = doc.createDocumentFragment();
      if (before) {
        fragment.appendChild(doc.createTextNode(before));
      }
      fragment.appendChild(linkNode);
      if (after) {
        fragment.appendChild(doc.createTextNode(after));
      }

      parent.replaceChild(fragment, node);
      inserted = true;
    }

    return { html: container.innerHTML, inserted };
  }, [buildAnchorRegex]);

  // Fetch suggestions from API
  const fetchSuggestions = useCallback(async () => {
    if (!postId) {
      return;
    }

    setIsLoading(true);
    setError(null);

    try {
      const response = await apiFetch({
        path: `/prorank-seo/v1/linking/basic-suggestions/${postId}`,
        method: 'GET',
      });

      if (response.success && response.data) {
        setSuggestions(response.data.suggestions || []);

        // Show message if no suggestions found
        if (!response.data.suggestions || response.data.suggestions.length === 0) {
          setError(
            response.data.message ||
              __('No internal linking suggestions found at this time.', 'prorank-seo')
          );
        }
      } else {
        setError(response.message || __('Failed to fetch suggestions.', 'prorank-seo'));
      }
    } catch (err) {
      setError(
        __('An error occurred while fetching suggestions. Please try again.', 'prorank-seo')
      );
    } finally {
      setIsLoading(false);
    }
  }, [postId]);

  // Debounced content change handler
  const debouncedFetch = useMemo(
    () =>
      debounce(() => {
        fetchSuggestions();
      }, 1000),
    [fetchSuggestions]
  );

  // Initial load
  useEffect(() => {
    fetchSuggestions();
  }, [postId, fetchSuggestions]);

  // Re-fetch when content changes (debounced)
  useEffect(() => {
    if (postContent) {
      debouncedFetch();
    }
  }, [postContent, debouncedFetch]);

  // Track already inserted links based on current content
  useEffect(() => {
    if (!postContent || !suggestions.length) {
      return;
    }
    const linked = extractLinkedUrls(postContent);
    if (!linked.length) {
      return;
    }
    setInsertedUrls((prev) => {
      const next = { ...prev };
      linked.forEach((url) => {
        next[url] = true;
      });
      return next;
    });
  }, [postContent, suggestions, extractLinkedUrls]);

  const isSuggestionInserted = useCallback(
    (url) => {
      const variants = getUrlVariants(url);
      if (!variants.length) {
        return false;
      }
      return variants.some((variant) => insertedUrls[variant]);
    },
    [getUrlVariants, insertedUrls]
  );

  // Track the last valid selection before focus moves to the sidebar/modal
  useEffect(() => {
    if (selectedBlockClientId) {
      lastSelectionRef.current = {
        clientId: selectedBlockClientId,
        selectionStart,
        selectionEnd,
      };
    }
  }, [selectedBlockClientId, selectionStart, selectionEnd]);

  // Handle insert link button click - opens modal
  const handleInsertLink = useCallback((suggestion) => {
    const suggestedAnchor =
      suggestion?.suggested_anchor || suggestion?.anchor_text || suggestion?.anchor || '';

    // Set the pending suggestion and anchor text
    setPendingSuggestion(suggestion);
    setEditingAnchorText(suggestedAnchor);

    // Open the modal
    setIsModalOpen(true);
  }, []);

  // Handle the actual link insertion after modal confirmation
  const handleConfirmInsert = useCallback(() => {
    if (!pendingSuggestion || !editingAnchorText.trim()) {
      return;
    }

    try {
      const blockEditor = wp.data.select('core/block-editor');

      const anchorTextRaw = String(editingAnchorText || '').trim();
      const anchorTextEscaped = anchorTextRaw
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');

      // Sanitize URL to prevent XSS - remove dangerous protocols
      const sanitizedUrl = String(pendingSuggestion.url || '')
        .replace(/javascript:/gi, '')
        .replace(/data:/gi, '')
        .replace(/vbscript:/gi, '')
        .replace(/"/g, '&quot;')
        .trim();

      // Validate URL is a proper http/https URL or relative path
      if (!sanitizedUrl || (!sanitizedUrl.startsWith('http') && !sanitizedUrl.startsWith('/'))) {
        console.warn('Invalid URL rejected:', pendingSuggestion.url);
        return;
      }

      const settings = linkingSettings || {};
      const siteOrigin = window.location.origin || '';
      const isExternal =
        sanitizedUrl.startsWith('http') && (!siteOrigin || !sanitizedUrl.startsWith(siteOrigin));
      const openInNewTab = isExternal
        ? settings.open_external_new_tab || settings.open_in_new_tab
        : settings.open_internal_new_tab || settings.open_in_new_tab;

      const relParts = [];
      if (openInNewTab) {
        relParts.push('noopener', 'noreferrer');
      }
      if (settings.add_nofollow && isExternal) {
        relParts.push('nofollow');
      }
      const relAttr = relParts.length ? ` rel="${Array.from(new Set(relParts)).join(' ')}"` : '';
      const targetAttr = openInNewTab ? ' target="_blank"' : '';

      let titleAttr = '';
      if (settings.add_title_attribute) {
        const titleValue = String(pendingSuggestion?.title || anchorTextRaw).replace(/"/g, '&quot;');
        titleAttr = ` title="${titleValue}"`;
      }

      // Create the link HTML with sanitized values
      const linkHTML = `<a href="${sanitizedUrl}"${targetAttr}${relAttr}${titleAttr}>${anchorTextEscaped}</a>`;

      const applyBlockUpdate = (block, key, html) => {
        updateBlockAttributes(block.clientId, { [key]: html });
        const updatedBlock = wp.data.select('core/block-editor').getBlock(block.clientId);
        if (updatedBlock?.attributes?.[key] !== html) {
          try {
            const { replaceBlock } = wp.data.dispatch('core/block-editor');
            const newAttributes = { ...block.attributes, [key]: html };
            const newBlock = wp.blocks.createBlock(block.name, newAttributes, block.innerBlocks);
            replaceBlock(block.clientId, newBlock);
          } catch (error) {
            // ignore fallback failures
          }
        }
      };

      const insertIntoBlock = (block) => {
        if (!block) {
          return false;
        }
        const attributes = block.attributes || {};
        const matcher = buildAnchorRegex(anchorTextRaw);
        if (matcher) {
          for (const [key, value] of Object.entries(attributes)) {
            if (typeof value !== 'string' || value.length < anchorTextRaw.length) {
              continue;
            }
            if (!matcher.test(value)) {
              continue;
            }
            const result = insertLinkInHtml(value, anchorTextRaw, sanitizedUrl);
            if (result.inserted) {
              applyBlockUpdate(block, key, result.html);
              return true;
            }
          }
        }
        const innerBlocks = block.innerBlocks || [];
        for (const inner of innerBlocks) {
          if (insertIntoBlock(inner)) {
            return true;
          }
        }
        return false;
      };

      const blocks = blockEditor.getBlocks() || [];
      let inserted = false;
      for (const block of blocks) {
        if (insertIntoBlock(block)) {
          inserted = true;
          break;
        }
      }

      if (!inserted && postContent) {
        const { editPost } = wp.data.dispatch('core/editor');
        const result = insertLinkInHtml(postContent, anchorTextRaw, sanitizedUrl);
        if (result.inserted) {
          editPost({ content: result.html });
          inserted = true;
        }
      }

      if (!inserted && (!postContent || !postContent.trim())) {
        const newBlock = wp.blocks.createBlock('core/paragraph', {
          content: linkHTML,
        });
        insertBlocks([newBlock]);
        inserted = true;
      }

      if (!inserted) {
        wp.data.dispatch('core/notices').createErrorNotice(
          __('Anchor text not found in the current content. Edit the anchor text to match your content and try again.', 'prorank-seo'),
          {
            type: 'snackbar',
            isDismissible: true,
          }
        );
        return;
      }

      const insertedTitle = pendingSuggestion?.title || '';
      const insertedUrl = sanitizedUrl;
      const insertedVariants = getUrlVariants(insertedUrl);

      // Close the modal
      setIsModalOpen(false);
      setPendingSuggestion(null);
      setEditingAnchorText('');

      // Verify the link landed in content before reporting success
      setTimeout(() => {
        const updatedContent = wp.data.select('core/editor').getEditedPostContent() || '';
        const found = insertedVariants.some((variant) => variant && updatedContent.includes(variant));
        if (!found) {
          wp.data
            .dispatch('core/notices')
            .createErrorNotice(__('Link was not inserted. Try editing the anchor text to match your content.', 'prorank-seo'), {
              type: 'snackbar',
              isDismissible: true,
            });
          return;
        }

        setInsertedUrls((prev) => {
          const next = { ...prev };
          insertedVariants.forEach((variant) => {
            if (variant) {
              next[variant] = true;
            }
          });
          return next;
        });

        wp.data.dispatch('core/notices').createSuccessNotice(
          sprintf(
            /* translators: %s: post title */
            __('Link to "%s" inserted successfully.', 'prorank-seo'),
            insertedTitle
          ),
          {
            type: 'snackbar',
            isDismissible: true,
          }
        );
      }, 350);
    } catch (err) {
      // Show error notification
      wp.data
        .dispatch('core/notices')
        .createErrorNotice(__('Failed to insert link. Please try again.', 'prorank-seo'), {
          type: 'snackbar',
          isDismissible: true,
        });
    }
  }, [
    selectedBlockClientId,
    insertBlocks,
    pendingSuggestion,
    editingAnchorText,
    findFirstEditableBlock,
    getContentAttributeKey,
    buildAnchorRegex,
    insertLinkInHtml,
    updateBlockAttributes,
    postContent,
    getUrlVariants,
  ]);

  // Handle modal cancel
  const handleCancelInsert = useCallback(() => {
    setIsModalOpen(false);
    setPendingSuggestion(null);
    setEditingAnchorText('');
  }, []);

  // Render loading state
  if (isLoading) {
    return (
      <div className="prorank-linking-suggestions__loading">
        <Spinner />
        <p>{__('Loading suggestions…', 'prorank-seo')}</p>
      </div>
    );
  }

  // Render error state
  if (error && suggestions.length === 0) {
    return (
      <div className="prorank-linking-suggestions__message">
        <p>{error}</p>
        <Button
          variant="secondary"
          onClick={fetchSuggestions}
          className="prorank-linking-suggestions__refresh"
        >
          {__('Refresh Suggestions', 'prorank-seo')}
        </Button>
      </div>
    );
  }

  // Render suggestions
  return (
    <div className="prorank-linking-suggestions">
      <div className="prorank-linking-suggestions__header">
        <p className="prorank-linking-suggestions__description">
          {__('Suggested internal links based on your content:', 'prorank-seo')}
        </p>
        <Button
          variant="tertiary"
          onClick={fetchSuggestions}
          disabled={isLoading === true}
          className="prorank-linking-suggestions__refresh"
          aria-label={__('Refresh internal linking suggestions', 'prorank-seo')}
        >
          {__('Refresh', 'prorank-seo')}
        </Button>
      </div>

      {suggestions.length > 0 ? (
        <ul className="prorank-linking-suggestions__list">
          {suggestions.map((suggestion, index) => {
            const anchorText =
              suggestion?.suggested_anchor || suggestion?.anchor_text || suggestion?.anchor || '';
            const isInserted = isSuggestionInserted(suggestion?.url);
            return (
            <li key={index} className="prorank-linking-suggestions__item">
              <Card size="small" className="prorank-linking-suggestion-card">
                <CardBody>
                  <div className="prorank-linking-suggestion__content">
                    <h4 className="prorank-linking-suggestion__title">{suggestion.title}</h4>
                    <p className="prorank-linking-suggestion__url">{suggestion.url}</p>
                    <div className="prorank-linking-suggestion__anchor">
                      <span className="prorank-linking-suggestion__anchor-label">
                        {__('Suggested anchor:', 'prorank-seo')}
                      </span>
                      <span className="prorank-linking-suggestion__anchor-text">
                        {anchorText ? `"${anchorText}"` : __('(No anchor found)', 'prorank-seo')}
                      </span>
                    </div>
                  </div>
                  <Button
                    variant={isInserted ? 'secondary' : 'primary'}
                    size="small"
                    onClick={() => handleInsertLink(suggestion)}
                    className={`prorank-linking-suggestion__insert${isInserted ? ' is-inserted' : ''}`}
                    disabled={isInserted}
                    aria-label={sprintf(
                      /* translators: %s: post title */
                      isInserted ? __('Link to %s inserted', 'prorank-seo') : __('Insert link to %s', 'prorank-seo'),
                      suggestion.title
                    )}
                  >
                    {isInserted ? __('Inserted', 'prorank-seo') : __('Insert Link', 'prorank-seo')}
                  </Button>
                </CardBody>
              </Card>
            </li>
          );
          })}
        </ul>
      ) : (
        <div className="prorank-linking-suggestions__message">
          <p>
            {__(
              'No suggestions available. Try adding more content or ensure you have other published posts.',
              'prorank-seo'
            )}
          </p>
        </div>
      )}

      {/* Link density stats if available */}
      {suggestions.length > 0 && (
        <div className="prorank-linking-suggestions__stats">
          <p className="prorank-linking-suggestions__stat">
            {sprintf(
              /* translators: %d: number of suggestions */
              __('%d suggestions found', 'prorank-seo'),
              suggestions.length
            )}
          </p>
        </div>
      )}

      {/* Anchor text editing modal */}
      {isModalOpen && pendingSuggestion && (
        <Modal
          title={__('Edit Anchor Text', 'prorank-seo')}
          onRequestClose={handleCancelInsert}
          className="prorank-linking-suggestion__modal"
          overlayClassName="prorank-linking-suggestion__modal-overlay"
          shouldCloseOnClickOutside={true}
          shouldCloseOnEsc={true}
          aria={{
            labelledby: 'prorank-anchor-edit-modal-title',
            describedby: 'prorank-anchor-edit-modal-description',
          }}
        >
          <div
            id="prorank-anchor-edit-modal-description"
            className="prorank-linking-suggestion__modal-description"
          >
            <p>
              {sprintf(
                /* translators: %s: post title */
                __(
                  'You are about to insert a link to "%s". You can edit the anchor text below:',
                  'prorank-seo'
                ),
                pendingSuggestion.title
              )}
            </p>
          </div>

          <TextControl
            label={__('Anchor Text', 'prorank-seo')}
            value={editingAnchorText}
            onChange={setEditingAnchorText}
            help={__('This is the text that will be displayed as the link.', 'prorank-seo')}
            className="prorank-linking-suggestion__anchor-input"
          />

          <div className="prorank-linking-suggestion__modal-actions">
            <Button
              variant="tertiary"
              onClick={handleCancelInsert}
              className="prorank-linking-suggestion__cancel-button"
            >
              {__('Cancel', 'prorank-seo')}
            </Button>
            <Button
              variant="primary"
              onClick={handleConfirmInsert}
              disabled={!editingAnchorText.trim()}
              className="prorank-linking-suggestion__confirm-button"
            >
              {__('Insert Link', 'prorank-seo')}
            </Button>
          </div>
        </Modal>
      )}
    </div>
  );
};

export default LinkingSuggestionsPanel;
