/**
 * Headline Analyzer Panel
 *
 * Google-aligned headline analysis based on Search Central guidance.
 * Analyzes rewrite risk, content match, uniqueness, stuffing, and schema alignment.
 *
 * @see https://developers.google.com/search/docs/appearance/title-link
 */

import { debounce } from 'lodash';
import './headline-analyzer-panel.css';

const HeadlineAnalyzerPanel = ({ postId }) => {
  const { useState, useEffect, useCallback, useMemo } = window.wp.element;
  const { __ } = window.wp.i18n;
  const apiFetch = window.wp.apiFetch;
  const { useSelect } = window.wp.data;
  const { Spinner, Button, Icon } = window.wp.components;

  const [analysis, setAnalysis] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState(null);
  const [expandedSections, setExpandedSections] = useState({
    rewrite: true,
    content: false,
    uniqueness: false,
    stuffing: false,
    schema: false,
  });

  const { title, seoTitle } = useSelect((select) => {
    const editor = select('core/editor');
    const postMeta = editor.getEditedPostAttribute('meta') || {};
    return {
      title: editor.getEditedPostAttribute('title') || '',
      seoTitle: postMeta._prorank_title || '',
    };
  }, []);

  const effectiveTitle = seoTitle || title;

  const analyzeHeadline = useCallback(
    debounce(async (titleToAnalyze, currentPostId) => {
      if (!titleToAnalyze || titleToAnalyze.length < 3) {
        setAnalysis(null);
        return;
      }

      const cacheKey = `${currentPostId || 0}:${titleToAnalyze}`;
      const cache = window.prorankHeadlineCache || (window.prorankHeadlineCache = {});
      if (cache[cacheKey]) {
        setAnalysis(cache[cacheKey]);
        setError(null);
        return;
      }

      setIsLoading(true);
      setError(null);

      try {
        const response = await apiFetch({
          path: '/prorank-seo/v1/headline-analyzer/analyze',
          method: 'POST',
          data: {
            title: titleToAnalyze,
            post_id: currentPostId || 0,
          },
        });

        if (response?.success && response?.data) {
          setAnalysis(response.data);
          cache[cacheKey] = response.data;
        } else {
          setError(__('Failed to analyze headline', 'prorank-seo'));
        }
      } catch (err) {
        console.error('Headline analysis error:', err);
        setError(err?.message || __('Analysis failed', 'prorank-seo'));
      } finally {
        setIsLoading(false);
      }
    }, 500),
    []
  );

  useEffect(() => {
    analyzeHeadline(effectiveTitle, postId);
  }, [effectiveTitle, postId, analyzeHeadline]);

  const toggleSection = (section) => {
    setExpandedSections((prev) => ({
      ...prev,
      [section]: !prev[section],
    }));
  };

  const getScoreColor = (score) => {
    if (score >= 80) return '#22c55e'; // Green
    if (score >= 60) return '#f59e0b'; // Orange
    return '#ef4444'; // Red
  };

  const getRiskLabel = (level) => {
    switch (level) {
      case 'low':
        return { text: __('Low Risk', 'prorank-seo'), color: '#22c55e' };
      case 'medium':
        return { text: __('Medium Risk', 'prorank-seo'), color: '#f59e0b' };
      case 'high':
        return { text: __('High Risk', 'prorank-seo'), color: '#ef4444' };
      default:
        return { text: __('Unknown', 'prorank-seo'), color: '#6b7280' };
    }
  };

  const getSeverityIcon = (severity) => {
    switch (severity) {
      case 'high':
        return '!';
      case 'medium':
        return '!';
      case 'low':
        return 'i';
      default:
        return '?';
    }
  };

  const getSeverityColor = (severity) => {
    switch (severity) {
      case 'high':
        return '#ef4444';
      case 'medium':
        return '#f59e0b';
      case 'low':
        return '#3b82f6';
      default:
        return '#6b7280';
    }
  };

  const ScoreCircle = ({ score, size = 80, label }) => {
    const radius = (size - 8) / 2;
    const circumference = 2 * Math.PI * radius;
    const progress = (score / 100) * circumference;
    const color = getScoreColor(score);

    return (
      <div className="headline-score-circle" style={{ width: size, height: size }}>
        <svg width={size} height={size} viewBox={`0 0 ${size} ${size}`}>
          <circle
            cx={size / 2}
            cy={size / 2}
            r={radius}
            fill="none"
            stroke="#e5e7eb"
            strokeWidth="6"
          />
          <circle
            cx={size / 2}
            cy={size / 2}
            r={radius}
            fill="none"
            stroke={color}
            strokeWidth="6"
            strokeDasharray={circumference}
            strokeDashoffset={circumference - progress}
            strokeLinecap="round"
            transform={`rotate(-90 ${size / 2} ${size / 2})`}
          />
        </svg>
        <div className="headline-score-value" style={{ color }}>
          {score}
        </div>
        {label && <div className="headline-score-label">{label}</div>}
      </div>
    );
  };

  const CheckSection = ({ title, score, level, issues, expanded, onToggle, children }) => {
    const riskInfo = level ? getRiskLabel(level) : null;

    return (
      <div className="headline-check-section">
        <button
          className="headline-check-header"
          onClick={onToggle}
          type="button"
        >
          <div className="headline-check-title">
            <span className="headline-check-arrow">{expanded ? '▼' : '▶'}</span>
            <span>{title}</span>
          </div>
          <div className="headline-check-meta">
            {riskInfo && (
              <span
                className="headline-risk-badge"
                style={{ backgroundColor: `${riskInfo.color}20`, color: riskInfo.color }}
              >
                {riskInfo.text}
              </span>
            )}
            {!riskInfo && score !== undefined && (
              <span
                className="headline-score-badge"
                style={{ backgroundColor: `${getScoreColor(score)}20`, color: getScoreColor(score) }}
              >
                {score}
              </span>
            )}
          </div>
        </button>
        {expanded && (
          <div className="headline-check-content">
            {issues && issues.length > 0 && (
              <ul className="headline-issues-list">
                {issues.map((issue, index) => (
                  <li
                    key={index}
                    className="headline-issue-item"
                    style={{ borderLeftColor: getSeverityColor(issue.severity) }}
                  >
                    <span
                      className="headline-issue-icon"
                      style={{ backgroundColor: getSeverityColor(issue.severity) }}
                    >
                      {getSeverityIcon(issue.severity)}
                    </span>
                    <span className="headline-issue-message">{issue.message}</span>
                  </li>
                ))}
              </ul>
            )}
            {(!issues || issues.length === 0) && (
              <p className="headline-no-issues">
                {__('No issues detected', 'prorank-seo')}
              </p>
            )}
            {children}
          </div>
        )}
      </div>
    );
  };

  if (!effectiveTitle || effectiveTitle.length < 3) {
    return (
      <div className="headline-analyzer-panel">
        <div className="headline-empty-state">
          <p>{__('Enter a title to analyze', 'prorank-seo')}</p>
        </div>
      </div>
    );
  }

  if (isLoading && !analysis) {
    return (
      <div className="headline-analyzer-panel">
        <div className="headline-loading">
          <Spinner />
          <p>{__('Analyzing headline...', 'prorank-seo')}</p>
        </div>
      </div>
    );
  }

  if (error && !analysis) {
    return (
      <div className="headline-analyzer-panel">
        <div className="headline-error">
          <p>{error}</p>
          <Button
            isSecondary
            onClick={() => analyzeHeadline(effectiveTitle, postId)}
          >
            {__('Retry', 'prorank-seo')}
          </Button>
        </div>
      </div>
    );
  }

  if (!analysis) {
    return null;
  }

  return (
    <div className="headline-analyzer-panel">
      {/* Overall Score */}
      <div className="headline-overall">
        <ScoreCircle
          score={analysis.overall_score}
          size={70}
          label={__('Overall', 'prorank-seo')}
        />
        <div className="headline-overall-info">
          <h4>{__('Google Alignment', 'prorank-seo')}</h4>
          <p className="headline-overall-desc">
            {analysis.overall_score >= 80
              ? __('Low rewrite risk', 'prorank-seo')
              : analysis.overall_score >= 60
              ? __('Some issues to address', 'prorank-seo')
              : __('High rewrite risk', 'prorank-seo')}
          </p>
        </div>
        {isLoading && <Spinner className="headline-refresh-spinner" />}
      </div>

      {/* Language Warning */}
      {analysis.language_warning && (
        <div className="headline-language-warning">
          <span className="headline-warning-icon">!</span>
          <span>
            {__('Non-Latin script detected. Analysis uses English tokenization only.', 'prorank-seo')}
          </span>
        </div>
      )}

      {/* Length Info */}
      <div className="headline-length-info">
        <div className="headline-length-item">
          <span className="headline-length-label">{__('Characters', 'prorank-seo')}</span>
          <span className="headline-length-value">{analysis.length?.chars || 0}</span>
        </div>
        <div className="headline-length-item">
          <span className="headline-length-label">{__('Pixel Width', 'prorank-seo')}</span>
          <span
            className="headline-length-value"
            style={{ color: analysis.length?.truncated ? '#ef4444' : 'inherit' }}
          >
            {analysis.length?.pixel_width || 0}px
            {analysis.length?.truncated && ' ⚠'}
          </span>
        </div>
      </div>

      {/* Check Sections */}
      <div className="headline-checks">
        <CheckSection
          title={__('Rewrite Risk', 'prorank-seo')}
          level={analysis.rewrite_risk?.level}
          issues={analysis.rewrite_risk?.issues}
          expanded={expandedSections.rewrite}
          onToggle={() => toggleSection('rewrite')}
        />

        <CheckSection
          title={__('Content Match', 'prorank-seo')}
          score={analysis.content_match?.score}
          issues={analysis.content_match?.issues}
          expanded={expandedSections.content}
          onToggle={() => toggleSection('content')}
        >
          {analysis.content_match && (
            <div className="headline-content-stats">
              <div className="headline-stat">
                <span>{__('Precision', 'prorank-seo')}</span>
                <span>{Math.round(analysis.content_match.precision * 100)}%</span>
              </div>
              <div className="headline-stat">
                <span>{__('Entity Recall', 'prorank-seo')}</span>
                <span>{Math.round(analysis.content_match.recall * 100)}%</span>
              </div>
            </div>
          )}
        </CheckSection>

        <CheckSection
          title={__('Uniqueness', 'prorank-seo')}
          score={analysis.uniqueness?.score}
          issues={analysis.uniqueness?.issues}
          expanded={expandedSections.uniqueness}
          onToggle={() => toggleSection('uniqueness')}
        >
          {analysis.uniqueness?.duplicates?.length > 0 && (
            <div className="headline-duplicates">
              <p className="headline-duplicates-label">
                {__('Duplicate titles found:', 'prorank-seo')}
              </p>
              <ul className="headline-duplicates-list">
                {analysis.uniqueness.duplicates.slice(0, 3).map((dup, i) => (
                  <li key={i}>
                    <a href={dup.url} target="_blank" rel="noopener noreferrer">
                      {dup.title}
                    </a>
                  </li>
                ))}
              </ul>
            </div>
          )}
        </CheckSection>

        <CheckSection
          title={__('Stuffing Check', 'prorank-seo')}
          score={analysis.stuffing?.score}
          issues={analysis.stuffing?.issues}
          expanded={expandedSections.stuffing}
          onToggle={() => toggleSection('stuffing')}
        />

        {analysis.schema?.applicable && (
          <CheckSection
            title={__('Schema Alignment', 'prorank-seo')}
            score={analysis.schema?.score}
            issues={analysis.schema?.issues}
            expanded={expandedSections.schema}
            onToggle={() => toggleSection('schema')}
          >
            {analysis.schema?.schema_headline && (
              <div className="headline-schema-info">
                <span className="headline-schema-label">
                  {__('Schema headline:', 'prorank-seo')}
                </span>
                <span className="headline-schema-value">
                  {analysis.schema.schema_headline}
                </span>
              </div>
            )}
          </CheckSection>
        )}
      </div>

      {/* Suggestions */}
      {analysis.suggestions && analysis.suggestions.length > 0 && (
        <div className="headline-suggestions">
          <h4>{__('Suggestions', 'prorank-seo')}</h4>
          <ul>
            {analysis.suggestions.map((suggestion, index) => (
              <li
                key={index}
                className={`headline-suggestion headline-suggestion-${suggestion.priority}`}
              >
                {suggestion.message}
              </li>
            ))}
          </ul>
        </div>
      )}

      {/* Info Footer */}
      <div className="headline-info-footer">
        <a
          href="https://developers.google.com/search/docs/appearance/title-link"
          target="_blank"
          rel="noopener noreferrer"
        >
          {__('Learn about Google title links', 'prorank-seo')} →
        </a>
      </div>
    </div>
  );
};

export default HeadlineAnalyzerPanel;
