/**
 * Content Analysis Panel (Free)
 * Modern layout with local analysis only (no cloud calls).
 */

import { debounce } from 'lodash';

const stripHtml = (html) => {
  if (!html) return '';
  return html
    .replace(/<script[^>]*>[\s\S]*?<\/script>/gi, ' ')
    .replace(/<style[^>]*>[\s\S]*?<\/style>/gi, ' ')
    .replace(/\[[^\]]+\]/g, ' ')
    .replace(/<[^>]+>/g, ' ')
    .replace(/\s+/g, ' ')
    .trim();
};

const escapeRegExp = (value) => value.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

const countWords = (text) => {
  const matches = text.match(/\b[\p{L}\p{N}'-]+\b/gu);
  return matches ? matches.length : 0;
};

const countSentences = (text) => {
  const matches = text.split(/[.!?]+/).filter((part) => part.trim().length > 0);
  return Math.max(1, matches.length);
};

const countSyllablesInWord = (word) => {
  if (!word) return 0;
  let cleaned = word.toLowerCase().replace(/e$/, '').replace(/[^aeiouy]/g, ' ');
  const syllables = cleaned.trim().split(/\s+/).filter(Boolean);
  return syllables.length > 0 ? syllables.length : 1;
};

const countSyllables = (text) => {
  const cleaned = text.toLowerCase().replace(/[^a-z]/g, ' ');
  const words = cleaned.trim().split(/\s+/).filter(Boolean);
  return words.reduce((total, word) => total + countSyllablesInWord(word), 0);
};

const calculateReadabilityScore = (text) => {
  const cleaned = text.trim();
  if (!cleaned) return null;
  const words = countWords(cleaned);
  const sentences = countSentences(cleaned);
  const syllables = countSyllables(cleaned);
  if (!words || !sentences) return null;
  const score = 206.835 - (1.015 * (words / sentences)) - (84.6 * (syllables / words));
  return Math.max(0, Math.min(100, Math.round(score * 10) / 10));
};

const ContentAnalysisPanel = ({ postId }) => {
  const { useState, useEffect, useCallback, useMemo, useRef } = window.wp.element;
  const { __ } = window.wp.i18n;
  const apiFetch = window.wp.apiFetch;
  const { useSelect } = window.wp.data;
  const { Spinner } = window.wp.components;

  const [readabilityScore, setReadabilityScore] = useState(null);
  const [readabilityLevel, setReadabilityLevel] = useState(null);
  const [readabilityGrade, setReadabilityGrade] = useState(null);
  const [readabilityStats, setReadabilityStats] = useState(null);
  const [isLoadingReadability, setIsLoadingReadability] = useState(false);
  const [isAnalyzing, setIsAnalyzing] = useState(false);
  const [focusKeyword, setFocusKeyword] = useState('');
  const [seoScore, setSeoScore] = useState(0);
  const [analysisSettings, setAnalysisSettings] = useState({
    enable_realtime: true,
    min_content_length: 300,
    target_readability_score: 60,
    auto_analyze_delay: 2000,
    show_keyword_density: true,
    show_entity_detection: true,
    show_intent_classification: true,
  });
  const [manualSnapshot, setManualSnapshot] = useState({
    content: '',
    title: '',
  });
  const [hasManualSnapshot, setHasManualSnapshot] = useState(false);
  const hasLoadedKeyword = useRef(false);

  const { content, title } = useSelect((select) => {
    const editor = select('core/editor');
    return {
      content: editor.getEditedPostContent() || '',
      title: editor.getEditedPostAttribute('title') || '',
    };
  }, []);

  useEffect(() => {
    apiFetch({ path: '/prorank-seo/v1/settings/content-analysis' })
      .then((response) => {
        if (response?.data) {
          setAnalysisSettings((prev) => ({ ...prev, ...response.data }));
        }
      })
      .catch(() => {});
  }, [apiFetch]);

  const realtimeText = useMemo(() => stripHtml(content), [content]);
  const hasRealtimeContent = realtimeText.length > 0;
  const useRealtime = analysisSettings.enable_realtime && hasRealtimeContent;

  const analysisContent = useRealtime ? content : manualSnapshot.content;
  const analysisTitle = useRealtime ? title : manualSnapshot.title;
  const analysisKeyword = focusKeyword;
  const analysisReady = useRealtime || hasManualSnapshot;

  const textContent = useMemo(() => stripHtml(analysisContent), [analysisContent]);
  const normalizedKeyword = useMemo(() => analysisKeyword.trim().toLowerCase(), [analysisKeyword]);
  const wordList = useMemo(() => textContent.toLowerCase().match(/\b[\p{L}\p{N}'-]+\b/gu) || [], [textContent]);
  const wordCount = wordList.length;

  const keywordRegex = useMemo(() => {
    if (!normalizedKeyword) return null;
    return new RegExp(`\\b${escapeRegExp(normalizedKeyword)}\\b`, 'gi');
  }, [normalizedKeyword]);

  const keywordMatches = useMemo(() => {
    if (!keywordRegex) return [];
    return textContent.toLowerCase().match(keywordRegex) || [];
  }, [textContent, keywordRegex]);

  const keywordCount = keywordMatches.length;
  const density = wordCount ? (keywordCount / wordCount) * 100 : 0;

  const headingMatches = useMemo(() => {
    if (!normalizedKeyword) return [];
    return analysisContent.match(/<h[1-6][^>]*>[\s\S]*?<\/h[1-6]>/gi) || [];
  }, [analysisContent, normalizedKeyword]);

  const inHeadings = useMemo(() => {
    if (!normalizedKeyword) return 0;
    return headingMatches.filter((heading) =>
      stripHtml(heading).toLowerCase().includes(normalizedKeyword)
    ).length;
  }, [headingMatches, normalizedKeyword]);

  const firstParagraphText = useMemo(() => {
    const match = analysisContent.match(/<p[^>]*>([\s\S]*?)<\/p>/i);
    const paragraph = match ? match[1] : textContent;
    return stripHtml(paragraph).toLowerCase();
  }, [analysisContent, textContent]);

  const inFirstParagraph = normalizedKeyword
    ? firstParagraphText.includes(normalizedKeyword)
    : false;

  const inTitle = normalizedKeyword
    ? analysisTitle.toLowerCase().includes(normalizedKeyword)
    : false;

  const keywordOptimization = useMemo(() => {
    if (!normalizedKeyword) return null;
    return {
      density,
      optimal_density: density >= 0.5 && density <= 2.5,
      exact_matches: keywordCount,
      in_headings: inHeadings,
      in_first_paragraph: inFirstParagraph,
      in_title: inTitle,
    };
  }, [normalizedKeyword, density, keywordCount, inHeadings, inFirstParagraph, inTitle]);

  const minContentLength = Number(analysisSettings.min_content_length || 0);
  const readabilityTarget = Number(analysisSettings.target_readability_score || 60);
  const hasReadabilityScore = Number.isFinite(readabilityScore);
  const shouldRunReadability = analysisReady && wordCount > 0;

  const issues = useMemo(() => {
    if (!analysisReady) {
      return [];
    }
    const list = [];
    if (minContentLength > 0 && wordCount < minContentLength) {
      list.push(
        __('Add more content (aim for {words}+ words).', 'prorank-seo').replace('{words}', String(minContentLength))
      );
    }
    if (!normalizedKeyword) {
      list.push(__('Set a focus keyword to get on-page recommendations.', 'prorank-seo'));
      return list;
    }
    if (!inTitle) {
      list.push(__('Include the focus keyword in the title.', 'prorank-seo'));
    }
    if (!inFirstParagraph) {
      list.push(__('Use the focus keyword in the first paragraph.', 'prorank-seo'));
    }
    if (inHeadings === 0) {
      list.push(__('Add the focus keyword to at least one heading.', 'prorank-seo'));
    }
    if (analysisSettings.show_keyword_density) {
      if (density < 0.5) {
        list.push(__('Keyword density is low; add it naturally.', 'prorank-seo'));
      } else if (density > 3) {
        list.push(__('Keyword density is high; reduce repetition.', 'prorank-seo'));
      }
    }
    if (readabilityScore !== null && readabilityScore < readabilityTarget) {
      list.push(__('Improve readability for better engagement.', 'prorank-seo'));
    }
    return list;
  }, [
    analysisSettings.show_keyword_density,
    analysisReady,
    minContentLength,
    normalizedKeyword,
    inTitle,
    inFirstParagraph,
    inHeadings,
    density,
    readabilityScore,
    readabilityTarget,
    wordCount
  ]);

  const structureScore = useMemo(() => {
    let score = 100 - issues.length * 10;
    if (minContentLength > 0 && wordCount < minContentLength) {
      score = Math.min(score, 60);
    }
    return Math.max(20, score);
  }, [issues, minContentLength, wordCount]);

  const analysis = useMemo(() => ({
    structure_score: {
      score: structureScore,
      issues,
    },
    keyword_optimization: keywordOptimization,
  }), [structureScore, issues, keywordOptimization]);

  const calculateSeoScore = useCallback(() => {
    let score = 0;
    if (readabilityScore !== null) {
      score += Math.min(100, Math.max(0, readabilityScore)) * 0.35;
    }
    if (analysis?.structure_score) {
      score += analysis.structure_score.score * 0.35;
    }
    if (keywordOptimization && normalizedKeyword) {
      let keywordScore = 50;
      if (keywordOptimization.optimal_density) keywordScore += 20;
      if (keywordOptimization.in_first_paragraph) keywordScore += 15;
      if (keywordOptimization.in_headings > 0) keywordScore += 15;
      if (keywordOptimization.in_title) keywordScore += 10;
      score += keywordScore * 0.2;
    }
    const lengthTarget = minContentLength > 0 ? minContentLength : 300;
    const lengthScore =
      wordCount >= lengthTarget * 3 ? 100 :
        wordCount >= lengthTarget * 2 ? 75 :
          wordCount >= lengthTarget ? 50 : 25;
    score += lengthScore * 0.1;
    setSeoScore(Math.round(score));
  }, [analysis, keywordOptimization, minContentLength, normalizedKeyword, readabilityScore, wordCount]);

  useEffect(() => {
    calculateSeoScore();
  }, [calculateSeoScore]);

  const getReadabilityLevel = useCallback(
    (score) => {
      if (score >= 90) return __('Very Easy', 'prorank-seo');
      if (score >= 80) return __('Easy', 'prorank-seo');
      if (score >= 70) return __('Fairly Easy', 'prorank-seo');
      if (score >= 60) return __('Standard', 'prorank-seo');
      if (score >= 50) return __('Fairly Difficult', 'prorank-seo');
      if (score >= 30) return __('Difficult', 'prorank-seo');
      return __('Very Difficult', 'prorank-seo');
    },
    [__]
  );

  const getReadabilityGrade = useCallback(
    (score) => {
      if (score >= 90) return __('5th grade', 'prorank-seo');
      if (score >= 80) return __('6th grade', 'prorank-seo');
      if (score >= 70) return __('7th grade', 'prorank-seo');
      if (score >= 60) return __('8th-9th grade', 'prorank-seo');
      if (score >= 50) return __('10th-12th grade', 'prorank-seo');
      if (score >= 30) return __('College', 'prorank-seo');
      return __('College graduate', 'prorank-seo');
    },
    [__]
  );

  const calculateReadability = useCallback(() => {
    if (!shouldRunReadability) {
      setReadabilityScore(null);
      setReadabilityLevel(null);
      setReadabilityGrade(null);
      setReadabilityStats(null);
      setIsLoadingReadability(false);
      return;
    }

    const score = calculateReadabilityScore(textContent);
    if (score === null) {
      setReadabilityScore(null);
      setReadabilityLevel(null);
      setReadabilityGrade(null);
      setReadabilityStats(null);
      setIsLoadingReadability(false);
      return;
    }

    const sentenceCount = countSentences(textContent);
    const paragraphMatches = analysisContent.match(/<p[\s>]/gi);
    const paragraphCount = paragraphMatches ? paragraphMatches.length : 1;
    const readingTime = Math.max(1, Math.ceil(wordCount / 225));

    setReadabilityScore(score);
    setReadabilityLevel(getReadabilityLevel(score));
    setReadabilityGrade(getReadabilityGrade(score));
    setReadabilityStats({
      word_count: wordCount,
      sentence_count: sentenceCount,
      paragraph_count: Math.max(1, paragraphCount),
      estimated_reading_time: readingTime,
    });
    setIsLoadingReadability(false);
  }, [
    analysisContent,
    getReadabilityGrade,
    getReadabilityLevel,
    shouldRunReadability,
    textContent,
    wordCount,
  ]);

  const debouncedReadabilityFetch = useMemo(
    () => debounce(calculateReadability, Math.max(300, Number(analysisSettings.auto_analyze_delay || 800))),
    [analysisSettings.auto_analyze_delay, calculateReadability]
  );

  const saveFocusKeyword = useCallback(
    async (value) => {
      if (!postId) return;
      try {
        await apiFetch({
          path: `/prorank-seo/v1/meta/${postId}`,
          method: 'POST',
          data: {
            focus_keyword: value,
          },
        });
      } catch (err) {
        console.error('Focus keyword save failed:', err);
      }
    },
    [postId]
  );

  const debouncedSaveFocusKeyword = useMemo(
    () => debounce((value) => saveFocusKeyword(value), 600),
    [saveFocusKeyword]
  );

  useEffect(() => {
    return () => debouncedSaveFocusKeyword.cancel();
  }, [debouncedSaveFocusKeyword]);

  useEffect(() => {
    if (!postId || hasLoadedKeyword.current) {
      return;
    }
    apiFetch({
      path: `/prorank-seo/v1/meta/${postId}`,
      method: 'GET',
    })
      .then((response) => {
        if (response?.data?.focus_keyword !== undefined) {
          setFocusKeyword(response.data.focus_keyword || '');
        }
      })
      .catch((err) => {
        console.error('Focus keyword load failed:', err);
      })
      .finally(() => {
        hasLoadedKeyword.current = true;
      });
  }, [postId, apiFetch]);

  useEffect(() => {
    if (hasRealtimeContent || !postId || hasManualSnapshot) {
      return;
    }
    let isMounted = true;
    apiFetch({ path: `/prorank-seo/v1/meta/${postId}/analysis-content` })
      .then((response) => {
        if (!isMounted) {
          return;
        }
        if (response?.success && response?.data?.content) {
          setManualSnapshot({
            content: response.data.content,
            title: response.data.title || title,
          });
          setHasManualSnapshot(true);
        }
      })
      .catch(() => {});

    return () => {
      isMounted = false;
    };
  }, [apiFetch, hasManualSnapshot, hasRealtimeContent, postId, title]);

  useEffect(() => {
    if (!analysisReady) {
      setReadabilityScore(null);
      setReadabilityLevel(null);
      setReadabilityGrade(null);
      setReadabilityStats(null);
      setIsLoadingReadability(false);
      return () => debouncedReadabilityFetch.cancel();
    }

    if (!analysisContent) {
      setReadabilityScore(null);
      setReadabilityLevel(null);
      setReadabilityGrade(null);
      setReadabilityStats(null);
      setIsLoadingReadability(false);
      return () => debouncedReadabilityFetch.cancel();
    }

    setIsLoadingReadability(true);
    debouncedReadabilityFetch();
    return () => debouncedReadabilityFetch.cancel();
  }, [
    analysisReady,
    analysisContent,
    debouncedReadabilityFetch,
    minContentLength,
    wordCount,
  ]);

  const handleAnalyze = async () => {
    setIsAnalyzing(true);
    const value = focusKeyword.trim();
    saveFocusKeyword(value);
    let snapshotContent = content;
    let snapshotTitle = title;

    if (!hasRealtimeContent && postId) {
      try {
        const response = await apiFetch({
          path: `/prorank-seo/v1/meta/${postId}/analysis-content`,
        });
        if (response?.success && response?.data?.content) {
          snapshotContent = response.data.content;
          snapshotTitle = response.data.title || snapshotTitle;
        }
      } catch (error) {
        // ignore
      }
    }

    if (!analysisSettings.enable_realtime || !hasRealtimeContent) {
      if (snapshotContent) {
        setManualSnapshot({
          content: snapshotContent,
          title: snapshotTitle,
        });
        setHasManualSnapshot(true);
      }
    }

    setTimeout(() => setIsAnalyzing(false), 250);
  };

  const getScoreColor = (score) => {
    if (score >= 70) return '#3b82f6';
    if (score >= 40) return '#f59e0b';
    return '#ef4444';
  };

  const getScoreLabel = (score) => {
    if (score >= 80) return 'Excellent';
    if (score >= 60) return 'Good';
    if (score >= 40) return 'Fair';
    return 'Needs Work';
  };

  return (
    <div className="pr-sidebar">
      <style>{`
        .pr-sidebar {
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
          font-size: 13px;
          color: #374151;
        }
        .pr-score-section {
          padding: 22px 20px;
          background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
          border-bottom: 1px solid #e2e8f0;
        }
        .pr-score-row {
          display: flex;
          flex-direction: column;
          align-items: center;
          gap: 16px;
        }
        .pr-score-circle {
          position: relative;
          width: 80px;
          height: 80px;
          flex-shrink: 0;
          margin: 0 auto;
        }
        .pr-score-circle svg {
          transform: rotate(-90deg);
        }
        .pr-score-value {
          position: absolute;
          top: 50%;
          left: 50%;
          transform: translate(-50%, -50%);
          text-align: center;
        }
        .pr-score-num {
          font-size: 22px;
          font-weight: 700;
          color: #111827;
          line-height: 1;
        }
        .pr-score-label {
          font-size: 9px;
          color: #6b7280;
          text-transform: uppercase;
          letter-spacing: 0.05em;
          margin-top: 2px;
        }
        .pr-metrics {
          width: 100%;
          max-width: 260px;
          display: flex;
          flex-direction: column;
          gap: 8px;
        }
        .pr-metric {
          display: flex;
          justify-content: space-between;
          align-items: center;
          padding: 10px 12px;
          background: white;
          border-radius: 6px;
          font-size: 12px;
          border: 1px solid #e5e7eb;
        }
        .pr-metric-name { color: #6b7280; }
        .pr-metric-value { font-weight: 600; }
        .pr-metric-good { color: #3b82f6; }
        .pr-metric-warn { color: #f59e0b; }
        .pr-metric-bad { color: #ef4444; }
        .pr-section {
          padding: 16px;
          border-bottom: 1px solid #e2e8f0;
        }
        .pr-section-title {
          font-size: 11px;
          font-weight: 600;
          color: #3b82f6;
          text-transform: uppercase;
          letter-spacing: 0.5px;
          margin: 0 0 12px 0;
        }
        .pr-keyword-row {
          display: flex;
          flex-direction: column;
          gap: 10px;
        }
        .pr-keyword-input {
          width: 100%;
          padding: 10px 12px;
          border: 2px solid #e5e7eb;
          border-radius: 8px;
          font-size: 13px;
          transition: border-color 0.2s;
          box-sizing: border-box;
        }
        .pr-keyword-input:focus {
          outline: none;
          border-color: #3b82f6;
        }
        .pr-analyze-btn {
          width: 100%;
          padding: 10px 16px;
          background: linear-gradient(135deg, #3b82f6 0%, #2563eb 100%);
          color: white;
          border: none;
          border-radius: 8px;
          font-size: 13px;
          font-weight: 600;
          cursor: pointer;
          transition: all 0.2s;
          display: flex;
          align-items: center;
          justify-content: center;
          gap: 6px;
        }
        .pr-analyze-btn:hover:not(:disabled) {
          background: linear-gradient(135deg, #2563eb 0%, #1d4ed8 100%);
        }
        .pr-analyze-btn:disabled {
          opacity: 0.6;
          cursor: not-allowed;
        }
        .pr-keyword-stats {
          display: grid;
          grid-template-columns: repeat(3, 1fr);
          gap: 8px;
          margin-top: 12px;
        }
        .pr-kw-stat {
          text-align: center;
          padding: 10px 6px;
          background: white;
          border-radius: 8px;
          border: 1px solid #e5e7eb;
        }
        .pr-kw-stat-label {
          font-size: 9px;
          color: #6b7280;
          text-transform: uppercase;
          letter-spacing: 0.3px;
        }
        .pr-kw-stat-value {
          font-size: 18px;
          font-weight: 700;
          margin-top: 2px;
        }
        .pr-readability {
          display: flex;
          align-items: center;
          gap: 12px;
          padding: 12px;
          background: white;
          border-radius: 8px;
          border: 1px solid #e5e7eb;
        }
        .pr-read-text {
          display: flex;
          flex-direction: column;
          gap: 4px;
          min-width: 0;
          flex: 1;
        }
        .pr-read-score {
          width: 48px;
          height: 48px;
          border-radius: 50%;
          display: flex;
          align-items: center;
          justify-content: center;
          font-size: 18px;
          font-weight: 700;
          color: white;
          flex-shrink: 0;
          line-height: 1;
        }
        .pr-read-level {
          font-size: 14px;
          font-weight: 600;
          color: #111827;
          line-height: 1.2;
        }
        .pr-read-grade {
          font-size: 12px;
          color: #6b7280;
          line-height: 1.4;
        }
        .pr-issues { margin-top: 12px; }
        .pr-issue {
          display: flex;
          align-items: center;
          gap: 8px;
          padding: 8px 0;
          font-size: 12px;
          color: #4b5563;
          border-bottom: 1px solid #f3f4f6;
        }
        .pr-issue:last-child { border-bottom: none; }
        .pr-issue-dot {
          width: 6px;
          height: 6px;
          border-radius: 50%;
          background: #f59e0b;
          flex-shrink: 0;
        }
        .pr-issue-text {
          flex: 1;
          min-width: 0;
          line-height: 1.4;
          word-break: break-word;
        }
        .pr-issue-count {
          flex-shrink: 0;
          font-size: 10px;
          font-weight: 600;
          color: #6b7280;
          background: #f3f4f6;
          border-radius: 999px;
          padding: 2px 8px;
        }
        .pr-loading {
          display: flex;
          align-items: center;
          justify-content: center;
          padding: 24px;
        }
        .pr-empty {
          padding: 20px;
          text-align: center;
          color: #9ca3af;
          font-size: 13px;
        }
      `}</style>

      <div className="pr-score-section">
        <div className="pr-score-row">
          <div className="pr-score-circle">
            <svg width="80" height="80">
              <circle cx="40" cy="40" r="34" fill="none" stroke="#e5e7eb" strokeWidth="6" />
              <circle
                cx="40" cy="40" r="34"
                fill="none"
                stroke={getScoreColor(seoScore)}
                strokeWidth="6"
                strokeLinecap="round"
                strokeDasharray={`${(seoScore / 100) * 213} 213`}
                style={{ transition: 'stroke-dasharray 0.5s ease' }}
              />
            </svg>
            <div className="pr-score-value">
              <div className="pr-score-num">{seoScore}</div>
              <div className="pr-score-label">{getScoreLabel(seoScore)}</div>
            </div>
          </div>

          <div className="pr-metrics">
            <div className="pr-metric">
              <span className="pr-metric-name">Readability</span>
              <span className={`pr-metric-value ${hasReadabilityScore ? (readabilityScore >= readabilityTarget ? 'pr-metric-good' : readabilityScore >= 40 ? 'pr-metric-warn' : 'pr-metric-bad') : ''}`}>
                {hasReadabilityScore ? `${Math.round(readabilityScore)}` : '—'}
              </span>
            </div>
            <div className="pr-metric">
              <span className="pr-metric-name">Words</span>
              <span className={`pr-metric-value ${minContentLength > 0 ? (wordCount >= minContentLength ? 'pr-metric-good' : 'pr-metric-warn') : 'pr-metric-good'}`}>
                {wordCount}
              </span>
            </div>
            {readabilityStats?.estimated_reading_time && (
              <div className="pr-metric">
                <span className="pr-metric-name">Read Time</span>
                <span className="pr-metric-value">{readabilityStats.estimated_reading_time} min</span>
              </div>
            )}
          </div>
        </div>
      </div>

      <div className="pr-section">
        <h4 className="pr-section-title">{__('Focus Keyword', 'prorank-seo')}</h4>
        <div className="pr-keyword-row">
          <input
            type="text"
            className="pr-keyword-input"
            value={focusKeyword}
            onChange={(e) => {
              const value = e.target.value;
              setFocusKeyword(value);
              debouncedSaveFocusKeyword(value);
            }}
            placeholder={__('Enter keyword...', 'prorank-seo')}
          />
          <button
            className="pr-analyze-btn"
            onClick={handleAnalyze}
            disabled={isAnalyzing || !postId}
          >
            {isAnalyzing ? <Spinner /> : __('Analyze', 'prorank-seo')}
          </button>
        </div>

        {analysisSettings.show_keyword_density && keywordOptimization && normalizedKeyword && (
          <div className="pr-keyword-stats">
            <div className="pr-kw-stat">
              <div className="pr-kw-stat-label">Density</div>
              <div className={`pr-kw-stat-value ${keywordOptimization.optimal_density ? 'pr-metric-good' : 'pr-metric-bad'}`}>
                {density.toFixed(1)}%
              </div>
            </div>
            <div className="pr-kw-stat">
              <div className="pr-kw-stat-label">Count</div>
              <div className="pr-kw-stat-value">{keywordCount}</div>
            </div>
            <div className="pr-kw-stat">
              <div className="pr-kw-stat-label">Headings</div>
              <div className={`pr-kw-stat-value ${inHeadings > 0 ? 'pr-metric-good' : 'pr-metric-bad'}`}>
                {inHeadings}
              </div>
            </div>
          </div>
        )}
      </div>

      {!analysisReady && (
        <div className="pr-section">
          <div className="pr-empty">{__('Click Analyze to run content analysis.', 'prorank-seo')}</div>
        </div>
      )}

      <div className="pr-section">
        <h4 className="pr-section-title">{__('Readability', 'prorank-seo')}</h4>
        {isLoadingReadability ? (
          <div className="pr-loading"><Spinner /></div>
        ) : readabilityScore !== null ? (
          <div className="pr-readability">
          <div
            className="pr-read-score"
            style={{
              background: readabilityScore >= readabilityTarget
                  ? 'linear-gradient(135deg, #3b82f6, #2563eb)'
                  : readabilityScore >= 40
                    ? 'linear-gradient(135deg, #f59e0b, #d97706)'
                    : 'linear-gradient(135deg, #ef4444, #dc2626)'
              }}
            >
              {Math.round(readabilityScore)}
            </div>
            <div className="pr-read-text">
              <div className="pr-read-level">{readabilityLevel}</div>
              <div className="pr-read-grade">{readabilityGrade}</div>
            </div>
          </div>
        ) : (
          <div className="pr-empty">
            {!analysisReady
              ? __('Click Analyze to run content analysis.', 'prorank-seo')
              : wordCount === 0
                ? __('Add content to analyze', 'prorank-seo')
                : __('Readability will appear after analysis.', 'prorank-seo')}
          </div>
        )}
      </div>

      {analysisSettings.show_entity_detection && analysisReady && wordCount >= minContentLength && (
        <div className="pr-section">
          <h4 className="pr-section-title">{__('Entities', 'prorank-seo')}</h4>
          {(() => {
            const matches = textContent.match(/\b[A-Z][a-z]+(?:\s+[A-Z][a-z]+){0,2}\b/g) || [];
            const blacklist = new Set(['The', 'A', 'An', 'And', 'But', 'Or', 'For', 'With', 'From', 'When', 'How', 'Why', 'What', 'Who']);
            const counts = new Map();
            matches.forEach((value) => {
              if (blacklist.has(value)) return;
              const key = value.trim();
              if (!key) return;
              counts.set(key, (counts.get(key) || 0) + 1);
            });
            const entities = Array.from(counts.entries())
              .sort((a, b) => b[1] - a[1])
              .slice(0, 6);
            if (!entities.length) {
              return <div className="pr-empty">{__('No entities detected.', 'prorank-seo')}</div>;
            }
            return (
              <div className="pr-issues">
                {entities.map(([name, count]) => (
                  <div key={name} className="pr-issue pr-issue--entity">
                    <span className="pr-issue-dot"></span>
                    <span className="pr-issue-text">{name}</span>
                    <span className="pr-issue-count">{count}</span>
                  </div>
                ))}
              </div>
            );
          })()}
        </div>
      )}

      {analysisSettings.show_intent_classification && analysisReady && wordCount >= minContentLength && (
        <div className="pr-section">
          <h4 className="pr-section-title">{__('Search Intent', 'prorank-seo')}</h4>
          {(() => {
            const corpus = `${analysisTitle} ${textContent}`.toLowerCase();
            const hasAny = (terms) => terms.some((term) => corpus.includes(term));
            let intent = 'Informational';
            if (hasAny(['buy', 'price', 'pricing', 'order', 'deal', 'discount', 'coupon', 'subscribe'])) {
              intent = 'Transactional';
            } else if (hasAny(['best', 'review', 'compare', 'comparison', 'vs', 'top'])) {
              intent = 'Commercial';
            } else if (hasAny(['login', 'sign in', 'contact', 'address', 'about us'])) {
              intent = 'Navigational';
            } else if (hasAny(['near me', 'nearby', 'location', 'hours', 'phone'])) {
              intent = 'Local';
            }
            return (
              <div className="pr-readability">
                <div className="pr-read-score" style={{ background: 'linear-gradient(135deg, #10b981, #059669)' }}>
                  {intent.slice(0, 2).toUpperCase()}
                </div>
                <div className="pr-read-text">
                  <div className="pr-read-level">{intent}</div>
                  <div className="pr-read-grade">{__('Detected intent for this content.', 'prorank-seo')}</div>
                </div>
              </div>
            );
          })()}
        </div>
      )}

      {analysis?.structure_score?.issues?.length > 0 && (
        <div className="pr-section">
          <h4 className="pr-section-title">{__('Issues', 'prorank-seo')} ({analysis.structure_score.issues.length})</h4>
          <div className="pr-issues">
            {analysis.structure_score.issues.map((issue, idx) => (
              <div key={idx} className="pr-issue">
                <span className="pr-issue-dot"></span>
                <span className="pr-issue-text">{issue}</span>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  );
};

export default ContentAnalysisPanel;
