/**
 * Editor Sidebar Entry Point
 *
 * Registers the ProRank SEO sidebar plugin for the Block Editor.
 * Uses window.wp.* directly to avoid ES module timing issues.
 *
 * @package
 */

// Import styles (no dependencies)
import './styles.css';
import { debounce } from 'lodash';
import fallbackLogoUrl from '../../../assets/images/prorank-icon.svg';

const resolvePluginUrl = () => {
  const candidates = [
    window?.proRankSidebar?.pluginUrl,
    window?.prorankSeo?.pluginUrl,
    window?.prorankSeoAdmin?.pluginUrl,
    window?.prorankSeoEditor?.pluginUrl,
    window?.proRankSEO?.pluginUrl,
    window?.proRankSeoAdmin?.pluginUrl,
  ];

  for (const candidate of candidates) {
    if (typeof candidate === 'string' && candidate.length > 0) {
      return candidate;
    }
  }

  if (typeof document === 'undefined') {
    return '';
  }

  const scripts = document.getElementsByTagName('script');
  for (let index = 0; index < scripts.length; index += 1) {
    const src = scripts[index]?.src || '';
    if (!src) {
      continue;
    }
    if (src.includes('/wp-content/plugins/') && src.includes('/build/') && src.includes('/prorank')) {
      return src.replace(/\?.*$/, '').replace(/\/build\/[^/]+$/, '/');
    }
  }

  return '';
};

const pluginUrl = resolvePluginUrl();
const proRankLogoUrl = pluginUrl
  ? `${pluginUrl.replace(/\/+$/, '')}/assets/prorank.svg`
  : fallbackLogoUrl;

// Prevent duplicate registration
let sidebarRegistered = false;
let toolbarButtonInjected = false;

/**
 * Wait for WordPress to be fully initialized
 */
const waitForWordPress = (callback, maxAttempts = 50) => {
  let attempts = 0;

  const check = () => {
    attempts++;

    const hasPlugins = window.wp?.plugins?.registerPlugin;
    const hasEditor = window.wp?.editor?.PluginSidebar || window.wp?.editPost?.PluginSidebar;
    const hasI18n = window.wp?.i18n?.__;
    const hasElement = window.wp?.element?.createElement;

    if (hasPlugins && hasEditor && hasI18n && hasElement) {
      callback();
      return;
    }

    if (attempts < maxAttempts) {
      setTimeout(check, 100);
    } else {
      console.warn('[ProRank SEO] WordPress sidebar dependencies not available after timeout');
    }
  };

  check();
};

/**
 * Inject toolbar button directly into DOM (WordPress 6.6+ fallback)
 */
const injectToolbarButton = () => {
  if (toolbarButtonInjected) return;

  // Find the editor header settings area
  const headerSettings = document.querySelector('.editor-header__settings') ||
                         document.querySelector('.edit-post-header__settings') ||
                         document.querySelector('.interface-pinned-items');

  if (!headerSettings) {
    // Retry after a short delay
    setTimeout(injectToolbarButton, 500);
    return;
  }

  // Check if already injected
  if (document.querySelector('.prorank-toolbar-badge-score')) {
    toolbarButtonInjected = true;
    return;
  }

  const findExistingButton = () => {
    const containers = [
      headerSettings,
      document.querySelector('.edit-post-pinned-plugins'),
      document.querySelector('.interface-pinned-items'),
    ].filter(Boolean);

    for (const container of containers) {
      const buttons = container.querySelectorAll('button');
      for (const button of buttons) {
        const label = (
          (button.getAttribute('aria-label') || '') +
          ' ' +
          (button.getAttribute('title') || '') +
          ' ' +
          (button.textContent || '')
        ).toLowerCase();
        if (label.includes('prorank')) {
          return button;
        }
      }
    }

    return null;
  };

  const existingButton = findExistingButton();

  // Use existing ProRank button if available, otherwise create one
  const badge = existingButton || document.createElement('button');
  badge.classList.add('prorank-toolbar-badge');
  badge.type = badge.type || 'button';
  badge.setAttribute('aria-label', 'ProRank SEO Score');
  badge.innerHTML = `
    <img class="prorank-toolbar-badge-icon" src="${proRankLogoUrl}" width="22" height="22" alt="ProRank" />
    <span class="prorank-toolbar-badge-score">—</span>
  `;

  if (!existingButton) {
    // Click handler to toggle sidebar
    badge.addEventListener('click', () => {
      const dispatch = window.wp?.data?.dispatch?.('core/edit-post');
      const select = window.wp?.data?.select?.('core/edit-post');
      if (!dispatch || !select) return;

      const activeSidebar = select.getActiveGeneralSidebarName?.();
      if (activeSidebar === 'prorank-seo-sidebar/prorank-seo-sidebar') {
        dispatch.closeGeneralSidebar?.();
        badge.classList.remove('is-active');
      } else {
        dispatch.openGeneralSidebar?.('prorank-seo-sidebar/prorank-seo-sidebar');
        badge.classList.add('is-active');
      }
    });
  }

  if (!existingButton) {
    // Insert before the Save/Update button
    const saveButton = headerSettings.querySelector('.editor-post-publish-button__button') ||
                       headerSettings.querySelector('.editor-post-save-draft') ||
                       headerSettings.firstChild;

    if (saveButton) {
      headerSettings.insertBefore(badge, saveButton);
    } else {
      headerSettings.prepend(badge);
    }
  }

  // Hide any duplicate ProRank buttons without a score badge
  document.querySelectorAll('button').forEach((button) => {
    if (button === badge) return;
    const label = (
      (button.getAttribute('aria-label') || '') +
      ' ' +
      (button.getAttribute('title') || '') +
      ' ' +
      (button.textContent || '')
    ).toLowerCase();
    if (label.includes('prorank') && !button.querySelector('.prorank-toolbar-badge-score')) {
      button.classList.add('prorank-toolbar-hidden');
    }
  });

  toolbarButtonInjected = true;

  // Start score updates
  updateToolbarScore();
};

/**
 * Update the toolbar badge score
 */
const updateToolbarScore = debounce(async () => {
  const badge = document.querySelector('.prorank-toolbar-badge');
  if (!badge) return;

  const scoreEl = badge.querySelector('.prorank-toolbar-badge-score');
  if (!scoreEl) return;

  try {
    const select = window.wp?.data?.select?.('core/editor');
    if (!select) return;

    const postId = select.getCurrentPostId?.() || 0;
    const title = select.getEditedPostAttribute?.('title') || '';
    const meta = select.getEditedPostAttribute?.('meta') || {};
    const seoTitle = meta._prorank_title || '';
    const effectiveTitle = seoTitle || title;

    if (!effectiveTitle || effectiveTitle.length < 3) {
      scoreEl.textContent = '—';
      badge.classList.remove('score-good', 'score-ok', 'score-poor');
      badge.classList.add('prorank-toolbar-badge');
      return;
    }

    // Check cache first
    const cacheKey = `${postId}:${effectiveTitle}`;
    const cache = window.prorankHeadlineCache || (window.prorankHeadlineCache = {});

    if (cache[cacheKey]?.overall_score !== undefined) {
      const score = cache[cacheKey].overall_score;
      scoreEl.textContent = `${score}/100`;
      badge.classList.remove('score-good', 'score-ok', 'score-poor');
      badge.classList.add('prorank-toolbar-badge', score >= 80 ? 'score-good' : score >= 60 ? 'score-ok' : 'score-poor');
      return;
    }

    // Fetch score from API
    scoreEl.textContent = '...';
    const response = await window.wp.apiFetch({
      path: '/prorank-seo/v1/headline-analyzer/analyze',
      method: 'POST',
      data: { title: effectiveTitle, post_id: postId },
    });

    if (response?.success && response?.data?.overall_score !== undefined) {
      const score = response.data.overall_score;
      cache[cacheKey] = response.data;
      scoreEl.textContent = `${score}/100`;
      badge.classList.remove('score-good', 'score-ok', 'score-poor');
      badge.classList.add('prorank-toolbar-badge', score >= 80 ? 'score-good' : score >= 60 ? 'score-ok' : 'score-poor');
    } else {
      scoreEl.textContent = '—';
      badge.classList.remove('score-good', 'score-ok', 'score-poor');
      badge.classList.add('prorank-toolbar-badge');
    }
  } catch (err) {
    console.error('[ProRank] Score update error:', err);
    scoreEl.textContent = '—';
    badge.classList.remove('score-good', 'score-ok', 'score-poor');
    badge.classList.add('prorank-toolbar-badge');
  }
}, 500);

// Subscribe to editor changes to update score
const subscribeToEditorChanges = () => {
  const subscribe = window.wp?.data?.subscribe;
  if (!subscribe) return;

  let lastTitle = '';
  subscribe(() => {
    const select = window.wp?.data?.select?.('core/editor');
    if (!select) return;

    const title = select.getEditedPostAttribute?.('title') || '';
    const meta = select.getEditedPostAttribute?.('meta') || {};
    const seoTitle = meta._prorank_title || '';
    const effectiveTitle = seoTitle || title;

    if (effectiveTitle !== lastTitle) {
      lastTitle = effectiveTitle;
      updateToolbarScore();
    }
  });
};

/**
 * Initialize the sidebar
 */
const initSidebar = () => {
  // Prevent duplicate registration
  if (sidebarRegistered) {
    return;
  }

  const { createElement, lazy, Suspense, useState, useEffect, useMemo, Fragment } = window.wp.element;
  const FragmentEl = Fragment || window.wp.element.Fragment || 'div';
  const { registerPlugin, getPlugin } = window.wp.plugins;
  const { __, sprintf } = window.wp.i18n;
  const { Button, Spinner, Tooltip } = window.wp.components;
  const { useSelect, useDispatch } = window.wp.data;
  const apiFetch = window.wp.apiFetch;

  // Check if already registered
  if (getPlugin && getPlugin('prorank-seo-sidebar')) {
    sidebarRegistered = true;
    return;
  }

  // Get PluginSidebar from editor (WP 6.6+) or editPost (older)
  const editor = window.wp.editor || {};
  const editPost = window.wp.editPost || {};
  const PluginSidebar = editor.PluginSidebar || editPost.PluginSidebar;

  if (!PluginSidebar) {
    console.warn('[ProRank SEO] PluginSidebar components not available');
    return;
  }

  // ProRank SEO Icon - Custom logo from SVG file
  const ProRankIcon = ({ size = 20 }) => createElement(
    'img',
    {
      src: proRankLogoUrl,
      width: size,
      height: size,
      alt: 'ProRank SEO',
      style: { display: 'block' },
    }
  );

  // Toolbar button component that toggles the sidebar
  const ProRankToolbarButton = ({ usePluginToolbar = false } = {}) => {
    const { openGeneralSidebar, closeGeneralSidebar } = useDispatch('core/edit-post') || {};
    const [headlineScore, setHeadlineScore] = useState(null);
    const [isScoreLoading, setIsScoreLoading] = useState(false);

    const isOpen = useSelect((select) => {
      const editPostSelect = select('core/edit-post');
      if (!editPostSelect) return false;
      const activePanel = editPostSelect.getActiveGeneralSidebarName?.();
      return activePanel === 'prorank-seo-sidebar/prorank-seo-sidebar';
    }, []);

    const { postId, title, seoTitle } = useSelect((select) => {
      const editor = select('core/editor');
      const postMeta = editor?.getEditedPostAttribute('meta') || {};
      return {
        postId: editor?.getCurrentPostId?.() || 0,
        title: editor?.getEditedPostAttribute?.('title') || '',
        seoTitle: postMeta._prorank_title || '',
      };
    }, []);

    const effectiveTitle = seoTitle || title;

    const debouncedScoreFetch = useMemo(
      () => debounce(async (titleToAnalyze, currentPostId) => {
        if (!titleToAnalyze || titleToAnalyze.length < 3) {
          setHeadlineScore(null);
          setIsScoreLoading(false);
          return;
        }

        const cacheKey = `${currentPostId || 0}:${titleToAnalyze}`;
        const cache = window.prorankHeadlineCache || (window.prorankHeadlineCache = {});
        if (cache[cacheKey]?.overall_score !== undefined) {
          setHeadlineScore(cache[cacheKey].overall_score);
          setIsScoreLoading(false);
          return;
        }

        setIsScoreLoading(true);
        try {
          const response = await apiFetch({
            path: '/prorank-seo/v1/headline-analyzer/analyze',
            method: 'POST',
            data: {
              title: titleToAnalyze,
              post_id: currentPostId || 0,
            },
          });

          if (response?.success && response?.data?.overall_score !== undefined) {
            setHeadlineScore(response.data.overall_score);
            cache[cacheKey] = response.data;
          } else {
            setHeadlineScore(null);
          }
        } catch (err) {
          console.error('Headline score toolbar error:', err);
          setHeadlineScore(null);
        } finally {
          setIsScoreLoading(false);
        }
      }, 500),
      []
    );

    useEffect(() => {
      debouncedScoreFetch(effectiveTitle, postId);
      return () => debouncedScoreFetch.cancel();
    }, [effectiveTitle, postId, debouncedScoreFetch]);

    const toggleSidebar = () => {
      if (isOpen) {
        closeGeneralSidebar?.();
      } else {
        openGeneralSidebar?.('prorank-seo-sidebar/prorank-seo-sidebar');
      }
    };

    const scoreClass = headlineScore === null
      ? 'empty'
      : headlineScore >= 80
      ? 'good'
      : headlineScore >= 60
      ? 'ok'
      : 'poor';

    const scoreLabel = headlineScore === null ? '—' : headlineScore;
    const scoreTitle = headlineScore === null
      ? __('Headline score not analyzed', 'prorank-seo')
      : sprintf(__('Headline score: %d/100', 'prorank-seo'), headlineScore);

    const iconWithScore = createElement(
      'span',
      { className: 'prorank-toolbar-icon-wrap' },
      createElement('span', { className: 'prorank-toolbar-icon' }, createElement(ProRankIcon, { size: 20 })),
      createElement(
        'span',
        {
          className: `prorank-toolbar-score ${scoreClass}${isScoreLoading ? ' is-loading' : ''}`,
          title: scoreTitle,
          'aria-label': scoreTitle,
        },
        scoreLabel
      )
    );

    const buttonProps = {
      className: 'prorank-toolbar-button' + (isOpen ? ' is-pressed' : ''),
      onClick: toggleSidebar,
      'aria-pressed': isOpen,
      'aria-label': __('Toggle ProRank SEO sidebar', 'prorank-seo'),
    };

    if (usePluginToolbar) {
      const PluginToolbarButton = window.wp?.editPost?.PluginToolbarButton || window.wp?.editor?.PluginToolbarButton;
      if (PluginToolbarButton) {
        return createElement(PluginToolbarButton, {
          ...buttonProps,
          icon: iconWithScore,
          label: __('ProRank SEO', 'prorank-seo'),
          title: __('ProRank SEO', 'prorank-seo'),
        });
      }
    }

    return createElement(
      Tooltip,
      { text: __('ProRank SEO', 'prorank-seo') },
      createElement(Button, buttonProps, iconWithScore)
    );
  };

  // Lazy load the EditorSidebar component
  const EditorSidebar = lazy(() =>
    import(/* webpackMode: "eager" */ './EditorSidebar').catch((err) => {
      console.error('[ProRank SEO] Failed to load EditorSidebar:', err);
      return {
        default: () =>
          createElement(
            'div',
            { style: { padding: '16px', color: '#856404', background: '#fff3cd' } },
            __('Failed to load sidebar. Please refresh.', 'prorank-seo')
          ),
      };
    })
  );

  // Loading fallback
  const LoadingFallback = createElement(
    'div',
    {
      style: {
        display: 'flex',
        alignItems: 'center',
        justifyContent: 'center',
        padding: '20px',
        gap: '10px',
      },
    },
    Spinner && createElement(Spinner, null),
    createElement('span', null, __('Loading...', 'prorank-seo'))
  );

  // Get header toolbar fill slot
  const { createSlotFill } = window.wp.components;
  const PluginHeaderFill = createSlotFill ? createSlotFill('PinnedPlugins').Fill : null;

  const toolbarFill = PluginHeaderFill
    ? createElement(PluginHeaderFill, null, createElement(ProRankToolbarButton, { usePluginToolbar: true }))
    : null;

  // Register the plugin (toolbar via SlotFill; avoids DOM injection into React trees)
  registerPlugin('prorank-seo-sidebar', {
    icon: createElement(ProRankIcon, { size: 20 }),
    render: () =>
      createElement(
        FragmentEl,
        null,
        toolbarFill,
        createElement(
          PluginSidebar,
          {
            name: 'prorank-seo-sidebar',
            title: __('ProRank SEO', 'prorank-seo'),
            icon: createElement(ProRankIcon, { size: 20 }),
            className: 'prorank-plugin-sidebar',
          },
          createElement(Suspense, { fallback: LoadingFallback }, createElement(EditorSidebar, null))
        )
      ),
  });

  sidebarRegistered = true;
};

// Start initialization when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    waitForWordPress(initSidebar);
  });
} else {
  waitForWordPress(initSidebar);
}
