/**
 * Premium Score Circle Component
 * 
 * Animated SVG circular progress indicator
 * 
 * @package ProRank\SEO
 */

import { useState, useEffect, useRef } from "@wordpress/element";

/**
 * Get score status based on value
 */
const getScoreStatus = (score) => {
  if (score >= 60) return "success";
  if (score >= 30) return "warning";
  return "error";
};

/**
 * ScoreCircle Component
 * 
 * @param {Object} props
 * @param {number} props.score - Score value 0-100
 * @param {string} props.size - "small" (44px) | "large" (80px)
 * @param {string} props.label - Optional label text
 * @param {boolean} props.animate - Animate on mount (default true)
 * @param {boolean} props.glow - Show glow effect
 */
const ScoreCircle = ({ 
  score = 0, 
  size = "large", 
  label = "", 
  animate = true,
  glow = true 
}) => {
  const [animatedScore, setAnimatedScore] = useState(animate ? 0 : score);
  const prevScoreRef = useRef(score);
  
  // Dimensions based on size
  const dimensions = size === "large" ? 80 : 44;
  const strokeWidth = size === "large" ? 6 : 4;
  const radius = (dimensions - strokeWidth) / 2;
  const circumference = 2 * Math.PI * radius;
  
  // Calculate stroke dashoffset based on score
  const strokeDashoffset = circumference - (animatedScore / 100) * circumference;
  
  // Get status for coloring
  const status = getScoreStatus(score);
  
  // Animate score on mount and when score changes
  useEffect(() => {
    if (!animate) {
      setAnimatedScore(score);
      return;
    }
    
    const startScore = prevScoreRef.current;
    const endScore = score;
    const duration = 1200;
    const startTime = performance.now();
    
    const animateValue = (currentTime) => {
      const elapsed = currentTime - startTime;
      const progress = Math.min(elapsed / duration, 1);
      
      // Easing function (ease-out-cubic)
      const eased = 1 - Math.pow(1 - progress, 3);
      
      const currentValue = startScore + (endScore - startScore) * eased;
      setAnimatedScore(Math.round(currentValue));
      
      if (progress < 1) {
        requestAnimationFrame(animateValue);
      }
    };
    
    requestAnimationFrame(animateValue);
    prevScoreRef.current = score;
  }, [score, animate]);
  
  // Status class names
  const statusClass = `pr-score-circle--${status}`;
  const sizeClass = `pr-score-circle--${size}`;
  const glowClass = glow ? "pr-score-circle--glow" : "";
  
  return (
    <div className={`pr-score-circle ${sizeClass} ${statusClass} ${glowClass}`}>
      <svg 
        width={dimensions} 
        height={dimensions} 
        viewBox={`0 0 ${dimensions} ${dimensions}`}
      >
        {/* Background track */}
        <circle
          className="pr-score-circle__track"
          cx={dimensions / 2}
          cy={dimensions / 2}
          r={radius}
          strokeWidth={strokeWidth}
        />
        
        {/* Progress arc */}
        <circle
          className={`pr-score-circle__progress pr-score-circle__progress--${status}`}
          cx={dimensions / 2}
          cy={dimensions / 2}
          r={radius}
          strokeWidth={strokeWidth}
          strokeDasharray={circumference}
          strokeDashoffset={strokeDashoffset}
          style={{
            transition: animate ? "stroke-dashoffset 1.2s cubic-bezier(0.4, 0, 0.2, 1)" : "none"
          }}
        />
      </svg>
      
      {/* Center content */}
      <div className="pr-score-circle__content">
        <span className="pr-score-circle__value">{Math.round(animatedScore)}</span>
        {label && size === "large" && (
          <span className="pr-score-circle__label">{label}</span>
        )}
      </div>
    </div>
  );
};

export default ScoreCircle;
