/**
 * ProRank SEO - MultiRingScore Component
 *
 * Concentric animated progress rings showing multiple SEO dimensions
 * - Outer ring: Overall SEO Score
 * - Middle ring: Content Quality
 * - Inner ring: Technical SEO
 *
 * @package ProRank\SEO
 */

import { useState, useEffect } from '@wordpress/element';
import ProgressRing, { getScoreColor } from './ProgressRing.jsx';

/**
 * MultiRingScore Component
 */
const MultiRingScore = ({
  overallScore = 0,
  contentScore = 0,
  technicalScore = 0,
  size = 160,
  animate = true,
  showLabels = true,
  onRingClick,
  className = '',
}) => {
  const [hoveredRing, setHoveredRing] = useState(null);

  // Ring configuration
  const rings = [
    {
      id: 'overall',
      label: 'SEO',
      value: overallScore,
      strokeWidth: 10,
      delay: 0,
    },
    {
      id: 'content',
      label: 'Content',
      value: contentScore,
      strokeWidth: 8,
      delay: 200,
    },
    {
      id: 'technical',
      label: 'Technical',
      value: technicalScore,
      strokeWidth: 6,
      delay: 400,
    },
  ];

  // Calculate sizes for concentric rings
  const gap = 16;
  const outerSize = size;
  const middleSize = size - (rings[0].strokeWidth * 2 + gap);
  const innerSize = middleSize - (rings[1].strokeWidth * 2 + gap);

  const handleRingClick = (ringId) => {
    if (onRingClick) {
      onRingClick(ringId);
    }
  };

  const containerStyle = {
    position: 'relative',
    width: size,
    height: size,
    display: 'flex',
    alignItems: 'center',
    justifyContent: 'center',
  };

  const ringWrapperStyle = (ringSize) => ({
    position: 'absolute',
    top: '50%',
    left: '50%',
    transform: 'translate(-50%, -50%)',
    transition: 'transform 200ms ease, filter 200ms ease',
  });

  const centerLabelStyle = {
    position: 'absolute',
    top: '50%',
    left: '50%',
    transform: 'translate(-50%, -50%)',
    textAlign: 'center',
    zIndex: 10,
    pointerEvents: 'none',
  };

  const scoreValueStyle = {
    fontSize: size * 0.16,
    fontWeight: 700,
    color: getScoreColor(overallScore),
    lineHeight: 1,
    fontVariantNumeric: 'tabular-nums',
  };

  const scoreLabelStyle = {
    fontSize: size * 0.07,
    fontWeight: 500,
    color: 'var(--pr-text-muted, #9CA3AF)',
    marginTop: 4,
  };

  const tooltipStyle = {
    position: 'absolute',
    bottom: '100%',
    left: '50%',
    transform: 'translateX(-50%) translateY(-8px)',
    background: 'var(--pr-text-primary, #1F2937)',
    color: 'white',
    padding: '6px 10px',
    borderRadius: '6px',
    fontSize: '12px',
    fontWeight: 500,
    whiteSpace: 'nowrap',
    opacity: hoveredRing ? 1 : 0,
    visibility: hoveredRing ? 'visible' : 'hidden',
    transition: 'opacity 150ms ease, visibility 150ms ease',
    zIndex: 100,
    boxShadow: '0 4px 12px rgba(0, 0, 0, 0.15)',
  };

  const getHoveredInfo = () => {
    if (!hoveredRing) return null;
    const ring = rings.find(r => r.id === hoveredRing);
    if (!ring) return null;
    return ring.label + ': ' + ring.value + '/100';
  };

  return (
    <div className={'pr-multi-ring ' + className} style={containerStyle}>
      {/* Outer Ring - Overall SEO */}
      <div
        style={{
          ...ringWrapperStyle(outerSize),
          transform: hoveredRing === 'overall'
            ? 'translate(-50%, -50%) scale(1.02)'
            : 'translate(-50%, -50%)',
          filter: hoveredRing && hoveredRing !== 'overall' ? 'opacity(0.6)' : 'none',
        }}
        onMouseEnter={() => setHoveredRing('overall')}
        onMouseLeave={() => setHoveredRing(null)}
        onClick={() => handleRingClick('overall')}
      >
        <ProgressRing
          value={overallScore}
          size={outerSize}
          strokeWidth={rings[0].strokeWidth}
          animate={animate}
          animationDelay={rings[0].delay}
          showValue={false}
          showGlow={true}
        />
      </div>

      {/* Middle Ring - Content Quality */}
      <div
        style={{
          ...ringWrapperStyle(middleSize),
          transform: hoveredRing === 'content'
            ? 'translate(-50%, -50%) scale(1.03)'
            : 'translate(-50%, -50%)',
          filter: hoveredRing && hoveredRing !== 'content' ? 'opacity(0.6)' : 'none',
        }}
        onMouseEnter={() => setHoveredRing('content')}
        onMouseLeave={() => setHoveredRing(null)}
        onClick={() => handleRingClick('content')}
      >
        <ProgressRing
          value={contentScore}
          size={middleSize}
          strokeWidth={rings[1].strokeWidth}
          animate={animate}
          animationDelay={rings[1].delay}
          showValue={false}
          showGlow={false}
        />
      </div>

      {/* Inner Ring - Technical SEO */}
      <div
        style={{
          ...ringWrapperStyle(innerSize),
          transform: hoveredRing === 'technical'
            ? 'translate(-50%, -50%) scale(1.04)'
            : 'translate(-50%, -50%)',
          filter: hoveredRing && hoveredRing !== 'technical' ? 'opacity(0.6)' : 'none',
        }}
        onMouseEnter={() => setHoveredRing('technical')}
        onMouseLeave={() => setHoveredRing(null)}
        onClick={() => handleRingClick('technical')}
      >
        <ProgressRing
          value={technicalScore}
          size={innerSize}
          strokeWidth={rings[2].strokeWidth}
          animate={animate}
          animationDelay={rings[2].delay}
          showValue={false}
          showGlow={false}
        />
      </div>

      {/* Center Label */}
      {showLabels && (
        <div style={centerLabelStyle}>
          <div style={scoreValueStyle}>
            {Math.round(overallScore)}
          </div>
          <div style={scoreLabelStyle}>
            SEO Score
          </div>
        </div>
      )}

      {/* Hover Tooltip */}
      <div style={tooltipStyle}>
        {getHoveredInfo()}
      </div>
    </div>
  );
};

/**
 * Legend Component for MultiRingScore
 */
const MultiRingLegend = ({
  overallScore,
  contentScore,
  technicalScore,
  onItemClick,
  className = '',
}) => {
  const items = [
    { id: 'overall', label: 'Overall SEO', value: overallScore },
    { id: 'content', label: 'Content Quality', value: contentScore },
    { id: 'technical', label: 'Technical SEO', value: technicalScore },
  ];

  const legendStyle = {
    display: 'flex',
    flexDirection: 'column',
    gap: '8px',
  };

  const itemStyle = {
    display: 'flex',
    alignItems: 'center',
    gap: '8px',
    padding: '6px 10px',
    borderRadius: '6px',
    cursor: onItemClick ? 'pointer' : 'default',
    transition: 'background-color 150ms ease',
  };

  const dotStyle = (score) => ({
    width: '10px',
    height: '10px',
    borderRadius: '50%',
    backgroundColor: getScoreColor(score),
  });

  const labelStyle = {
    flex: 1,
    fontSize: '13px',
    color: 'var(--pr-text-secondary, #6B7280)',
  };

  const valueStyle = (score) => ({
    fontSize: '13px',
    fontWeight: 600,
    color: getScoreColor(score),
    fontVariantNumeric: 'tabular-nums',
  });

  return (
    <div className={'pr-multi-ring-legend ' + className} style={legendStyle}>
      {items.map(item => (
        <div
          key={item.id}
          style={itemStyle}
          onClick={() => onItemClick && onItemClick(item.id)}
        >
          <span style={dotStyle(item.value)} />
          <span style={labelStyle}>{item.label}</span>
          <span style={valueStyle(item.value)}>{Math.round(item.value || 0)}</span>
        </div>
      ))}
    </div>
  );
};

export default MultiRingScore;
export { MultiRingLegend };
