/**
 * Action Card Component
 * 
 * Premium styled action card for AI Assistant actions
 * 
 * @package ProRank\SEO
 */

import { Spinner } from "@wordpress/components";

/**
 * ActionCard Component
 * 
 * @param {Object} props
 * @param {React.ReactNode} props.icon - Icon component
 * @param {string} props.title - Action title
 * @param {string} props.description - Action description
 * @param {Function} props.onClick - Click handler
 * @param {boolean} props.loading - Loading state
 * @param {boolean} props.disabled - Disabled state
 */
const ActionCard = ({ 
  icon = null,
  title = "",
  description = "",
  onClick = () => {},
  loading = false,
  disabled = false
}) => {
  const handleClick = () => {
    if (!loading && !disabled) {
      onClick();
    }
  };
  
  return (
    <button 
      type="button"
      className={`pr-action-card ${loading ? "pr-action-card--loading" : ""} ${disabled ? "pr-action-card--disabled" : ""}`}
      onClick={handleClick}
      disabled={disabled || loading}
    >
      {icon && (
        <div className="pr-action-card__icon">
          {loading ? <Spinner /> : icon}
        </div>
      )}
      
      <div className="pr-action-card__content">
        <p className="pr-action-card__title">{title}</p>
        {description && (
          <p className="pr-action-card__description">{description}</p>
        )}
      </div>
      
      <div className="pr-action-card__arrow">
        <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path d="M9 6L15 12L9 18" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round"/>
        </svg>
      </div>
      
      <style>{`
        .pr-action-card {
          display: flex;
          align-items: center;
          gap: 12px;
          width: 100%;
          padding: 12px 16px;
          background: var(--pr-background, #fff);
          border: 1px solid var(--pr-border, #E5E7EB);
          border-radius: var(--pr-radius-lg, 8px);
          cursor: pointer;
          text-align: left;
          transition: all 0.2s ease;
          position: relative;
          overflow: hidden;
        }
        
        .pr-action-card::before {
          content: "";
          position: absolute;
          left: 0;
          top: 0;
          bottom: 0;
          width: 3px;
          background: linear-gradient(180deg, var(--pr-accent, #3b82f6) 0%, var(--pr-accent-dark, #27894A) 100%);
          opacity: 0;
          transition: opacity 0.15s ease;
        }
        
        .pr-action-card:hover:not(:disabled) {
          border-color: var(--pr-accent, #3b82f6);
          box-shadow: var(--pr-shadow-md, 0 4px 6px rgba(0,0,0,0.07));
          transform: translateX(2px);
        }
        
        .pr-action-card:hover:not(:disabled)::before {
          opacity: 1;
        }
        
        .pr-action-card--disabled {
          opacity: 0.6;
          cursor: not-allowed;
        }
        
        .pr-action-card--loading {
          pointer-events: none;
        }
        
        .pr-action-card--loading::after {
          content: "";
          position: absolute;
          inset: 0;
          background: linear-gradient(90deg, transparent, rgba(255,255,255,0.6), transparent);
          animation: pr-shimmer 1.5s infinite;
        }
        
        @keyframes pr-shimmer {
          0% { transform: translateX(-100%); }
          100% { transform: translateX(100%); }
        }
        
        .pr-action-card__icon {
          flex-shrink: 0;
          width: 36px;
          height: 36px;
          display: flex;
          align-items: center;
          justify-content: center;
          background: rgba(49, 162, 76, 0.08);
          border-radius: var(--pr-radius-md, 6px);
          color: var(--pr-accent, #3b82f6);
          transition: background 0.15s ease;
        }
        
        .pr-action-card:hover:not(:disabled) .pr-action-card__icon {
          background: var(--pr-accent-light, #D1FAE5);
        }
        
        .pr-action-card__content {
          flex: 1;
          min-width: 0;
        }
        
        .pr-action-card__title {
          font-size: 13px;
          font-weight: 600;
          color: var(--pr-text-primary, #1F2937);
          margin: 0;
          line-height: 1.3;
        }
        
        .pr-action-card__description {
          font-size: 12px;
          color: var(--pr-text-secondary, #6B7280);
          margin: 2px 0 0 0;
          line-height: 1.4;
        }
        
        .pr-action-card__arrow {
          flex-shrink: 0;
          color: var(--pr-text-muted, #9CA3AF);
          transition: transform 0.15s ease, color 0.15s ease;
        }
        
        .pr-action-card:hover:not(:disabled) .pr-action-card__arrow {
          color: var(--pr-accent, #3b82f6);
          transform: translateX(4px);
        }
      `}</style>
    </button>
  );
};

export default ActionCard;
