/**
 * Social Preview Component
 *
 * Shows a preview of how content will appear on social media platforms.
 *
 * @package
 */

import { __ } from '@wordpress/i18n';

const platformLimits = {
  facebook: {
    title: 80,
    description: 200,
  },
  twitter: {
    title: 70,
    description: (cardType) => ('summary_large_image' === cardType ? 200 : 125),
  },
};

const getCharacterLimit = (platform, type, cardType) => {
  const limits = platformLimits[platform];
  if (!limits) {
    return 'title' === type ? 70 : 160;
  }

  const limit = limits[type];
  return 'function' === typeof limit ? limit(cardType) : limit;
};

const SocialPreview = ({ platform, cardType = 'summary', title, description, imageUrl }) => {
  // Truncate text based on platform limits
  const truncateText = (text, maxLength) => {
    if (!text || text.length <= maxLength) {
      return text;
    }
    return text.substring(0, maxLength - 3) + '…';
  };

  const displayTitle = truncateText(
    title || __('Page Title', 'prorank-seo'),
    getCharacterLimit(platform, 'title', cardType)
  );

  const displayDescription = truncateText(
    description ||
      __('Add a description to see how your content will appear on social media.', 'prorank-seo'),
    getCharacterLimit(platform, 'description', cardType)
  );

  // Facebook Preview
  if ('facebook' === platform) {
    return (
      <div className="prorank-social-preview prorank-social-preview--facebook">
        <div className="prorank-social-preview__container">
          {imageUrl && (
            <div className="prorank-social-preview__image">
              <img src={imageUrl} alt={__('Facebook preview', 'prorank-seo')} />
            </div>
          )}
          <div className="prorank-social-preview__content">
            <div className="prorank-social-preview__domain">
              {window.location.hostname.toUpperCase()}
            </div>
            <h3 className="prorank-social-preview__title">{displayTitle}</h3>
            <p className="prorank-social-preview__description">{displayDescription}</p>
          </div>
        </div>
      </div>
    );
  }

  // Twitter/X Preview
  if ('twitter' === platform) {
    return (
      <div
        className={`prorank-social-preview prorank-social-preview--twitter prorank-social-preview--twitter-${cardType}`}
      >
        <div className="prorank-social-preview__container">
          {imageUrl && (
            <div className="prorank-social-preview__image">
              <img src={imageUrl} alt={__('X preview', 'prorank-seo')} />
            </div>
          )}
          <div className="prorank-social-preview__content">
            <h3 className="prorank-social-preview__title">{displayTitle}</h3>
            <p className="prorank-social-preview__description">{displayDescription}</p>
            <div className="prorank-social-preview__domain">
              <span className="prorank-social-preview__icon">🔗</span>
              {window.location.hostname}
            </div>
          </div>
        </div>
      </div>
    );
  }

  return null;
};

export default SocialPreview;
