/* global wp */
/**
 * Social Tab Content Component
 *
 * @package
 */

import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import {
  TextControl,
  TextareaControl,
  SelectControl,
  Button,
  Spinner,
} from '@wordpress/components';
/* eslint-disable @wordpress/no-unsafe-wp-apis */
import { MediaUpload, MediaUploadCheck } from '@wordpress/block-editor';
import { useState, useEffect, useCallback, useMemo } from '@wordpress/element';

// Access lodash from WordPress global
const { debounce } = window.lodash || { debounce: (fn) => fn };

// Import components
import SocialPreview from './components/SocialPreview';

const SocialTabContent = ({ postId }) => {
  // State for Open Graph fields
  const [ogTitle, setOgTitle] = useState('');
  const [ogDescription, setOgDescription] = useState('');
  const [ogImageId, setOgImageId] = useState(0);
  const [ogImageUrl, setOgImageUrl] = useState('');

  // State for Twitter fields
  const [twitterTitle, setTwitterTitle] = useState('');
  const [twitterDescription, setTwitterDescription] = useState('');
  const [twitterImageId, setTwitterImageId] = useState(0);
  const [twitterImageUrl, setTwitterImageUrl] = useState('');
  const [twitterCardType, setTwitterCardType] = useState('summary');

  // State for UI
  const [isLoading, setIsLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);
  const [saveError, setSaveError] = useState(null);
  const [defaults, setDefaults] = useState({ title: '', description: '', imageUrl: '' });

  // Fetch existing social data on mount
  useEffect(() => {
    if (!postId) {
      setIsLoading(false);
      return;
    }

    setIsLoading(true);

    apiFetch({
      path: `/prorank-seo/v1/meta/${postId}`,
      method: 'GET',
    })
      .then((response) => {
        if (response.success && response.data) {
          // Set OG fields
          setOgTitle(response.data.og_title || '');
          setOgDescription(response.data.og_description || '');
          setOgImageId(response.data.og_image_id || 0);
          setOgImageUrl(response.data.og_image_url || '');

          // Set Twitter fields
          setTwitterTitle(response.data.twitter_title || '');
          setTwitterDescription(response.data.twitter_description || '');
          setTwitterImageId(response.data.twitter_image_id || 0);
          setTwitterImageUrl(response.data.twitter_image_url || '');
          setTwitterCardType(response.data.twitter_card_type || 'summary');

          // Set defaults from meta tab
          setDefaults({
            title: response.data.seoTitle || response.data.defaults?.title || '',
            description: response.data.seo_description || '',
            imageUrl: response.data.featured_image_url || '',
          });
        }
      })
      .catch(() => {
        setSaveError(__('Failed to load social data.', 'prorank-seo'));
      })
      .finally(() => {
        setIsLoading(false);
      });
  }, [postId]);

  // Create save function
  const saveSocialDataImpl = useCallback((data) => {
    if (!postId) return;

    setIsSaving(true);
    setSaveError(null);

    apiFetch({
      path: `/prorank-seo/v1/meta/${postId}`,
      method: 'POST',
      data,
    })
      .then((response) => {
        if (!response.success) {
          throw new Error(response.message || __('Save failed', 'prorank-seo'));
        }
      })
      .catch((error) => {
        setSaveError(error.message || __('Failed to save changes.', 'prorank-seo'));
      })
      .finally(() => {
        setIsSaving(false);
      });
  }, [postId]);

  // Debounced save function
  const saveSocialData = useMemo(() => debounce(saveSocialDataImpl, 500), [saveSocialDataImpl]);

  // Handle field changes
  const handleOgTitleChange = (value) => {
    setOgTitle(value);
    saveSocialData({ og_title: value });
  };

  const handleOgDescriptionChange = (value) => {
    setOgDescription(value);
    saveSocialData({ og_description: value });
  };

  const handleOgImageSelect = (media) => {
    setOgImageId(media.id);
    setOgImageUrl(media.url);
    saveSocialData({ og_image_id: media.id });
  };

  const handleOgImageRemove = () => {
    setOgImageId(0);
    setOgImageUrl('');
    saveSocialData({ og_image_id: 0 });
  };

  const handleTwitterTitleChange = (value) => {
    setTwitterTitle(value);
    saveSocialData({ twitter_title: value });
  };

  const handleTwitterDescriptionChange = (value) => {
    setTwitterDescription(value);
    saveSocialData({ twitter_description: value });
  };

  const handleTwitterImageSelect = (media) => {
    setTwitterImageId(media.id);
    setTwitterImageUrl(media.url);
    saveSocialData({ twitter_image_id: media.id });
  };

  const handleTwitterImageRemove = () => {
    setTwitterImageId(0);
    setTwitterImageUrl('');
    saveSocialData({ twitter_image_id: 0 });
  };

  const handleTwitterCardTypeChange = (value) => {
    setTwitterCardType(value);
    saveSocialData({ twitter_card_type: value });
  };

  if (isLoading) {
    return (
      <div className="prorank-social-tab-loading">
        <Spinner />
        <span>{__('Loading social data…', 'prorank-seo')}</span>
      </div>
    );
  }

  return (
    <div className="prorank-social-tab-content">
      {/* Open Graph Section */}
      <div className="prorank-social-section">
        <h3>{__('Open Graph (Facebook)', 'prorank-seo')}</h3>
        <p className="description">
          {__(
            'Customize how your content appears when shared on Facebook and other platforms that support Open Graph.',
            'prorank-seo'
          )}
        </p>

        <div className="prorank-field-group">
          <TextControl
            label={__('Facebook Title', 'prorank-seo')}
            value={ogTitle}
            onChange={handleOgTitleChange}
            placeholder={defaults.title || __('Enter Facebook title…', 'prorank-seo')}
            help={__('Leave empty to use SEO title.', 'prorank-seo')}
            __nextHasNoMarginBottom={true}
            __next40pxDefaultSize={true}
          />
        </div>

        <div className="prorank-field-group">
          <TextareaControl
            label={__('Facebook Description', 'prorank-seo')}
            value={ogDescription}
            onChange={handleOgDescriptionChange}
            placeholder={defaults.description || __('Enter Facebook description…', 'prorank-seo')}
            help={__('Leave empty to use meta description.', 'prorank-seo')}
            rows={3}
            __nextHasNoMarginBottom={true}
          />
        </div>

        <div className="prorank-field-group">
          {/* eslint-disable-next-line jsx-a11y/label-has-associated-control */}
          <label className="components-base-control__label" id="og-image-label">
            {__('Facebook Image', 'prorank-seo')}
          </label>
          <MediaUploadCheck>
            <div className="prorank-media-upload" aria-labelledby="og-image-label">
              {ogImageUrl ? (
                <div className="prorank-media-preview">
                  <img src={ogImageUrl} alt={__('Facebook image preview', 'prorank-seo')} />
                  <Button isDestructive variant="secondary" onClick={handleOgImageRemove}>
                    {__('Remove Image', 'prorank-seo')}
                  </Button>
                </div>
              ) : (
                <MediaUpload
                  onSelect={handleOgImageSelect}
                  allowedTypes={['image']}
                  value={ogImageId}
                  render={({ open }) => (
                    <Button variant="secondary" onClick={open}>
                      {__('Select Image', 'prorank-seo')}
                    </Button>
                  )}
                />
              )}
            </div>
          </MediaUploadCheck>
          <p className="description">
            {__(
              'Recommended size: 1200 x 630 pixels. Leave empty to use featured image.',
              'prorank-seo'
            )}
          </p>
        </div>

        {/* Facebook Preview */}
        <div className="prorank-field-group">
          <h4>{__('Facebook Preview', 'prorank-seo')}</h4>
          <SocialPreview
            platform="facebook"
            title={ogTitle || defaults.title}
            description={ogDescription || defaults.description}
            imageUrl={ogImageUrl || defaults.imageUrl}
          />
        </div>
      </div>

      <hr className="prorank-section-divider" />

      {/* Twitter/X Section */}
      <div className="prorank-social-section">
        <h3>{__('X (Twitter) Card', 'prorank-seo')}</h3>
        <p className="description">
          {__(
            'Customize how your content appears when shared on X (formerly Twitter).',
            'prorank-seo'
          )}
        </p>

        <div className="prorank-field-group">
          <SelectControl
            label={__('Card Type', 'prorank-seo')}
            value={twitterCardType}
            onChange={handleTwitterCardTypeChange}
            options={[
              { label: __('Summary', 'prorank-seo'), value: 'summary' },
              {
                label: __('Summary with Large Image', 'prorank-seo'),
                value: 'summary_large_image',
              },
              { label: __('App', 'prorank-seo'), value: 'app' },
              { label: __('Player', 'prorank-seo'), value: 'player' },
            ]}
            __nextHasNoMarginBottom={true}
            __next40pxDefaultSize={true}
          />
        </div>

        <div className="prorank-field-group">
          <TextControl
            label={__('X Title', 'prorank-seo')}
            value={twitterTitle}
            onChange={handleTwitterTitleChange}
            placeholder={ogTitle || defaults.title || __('Enter X title…', 'prorank-seo')}
            help={__('Leave empty to use Facebook title or SEO title.', 'prorank-seo')}
            __nextHasNoMarginBottom={true}
            __next40pxDefaultSize={true}
          />
        </div>

        <div className="prorank-field-group">
          <TextareaControl
            label={__('X Description', 'prorank-seo')}
            value={twitterDescription}
            onChange={handleTwitterDescriptionChange}
            placeholder={
              ogDescription || defaults.description || __('Enter X description…', 'prorank-seo')
            }
            help={__('Leave empty to use Facebook description or meta description.', 'prorank-seo')}
            rows={2}
            __nextHasNoMarginBottom={true}
          />
        </div>

        <div className="prorank-field-group">
          {/* eslint-disable-next-line jsx-a11y/label-has-associated-control */}
          <label className="components-base-control__label" id="twitter-image-label">
            {__('X Image', 'prorank-seo')}
          </label>
          <MediaUploadCheck>
            <div className="prorank-media-upload" aria-labelledby="twitter-image-label">
              {twitterImageUrl ? (
                <div className="prorank-media-preview">
                  <img src={twitterImageUrl} alt={__('X image preview', 'prorank-seo')} />
                  <Button isDestructive variant="secondary" onClick={handleTwitterImageRemove}>
                    {__('Remove Image', 'prorank-seo')}
                  </Button>
                </div>
              ) : (
                <MediaUpload
                  onSelect={handleTwitterImageSelect}
                  allowedTypes={['image']}
                  value={twitterImageId}
                  render={({ open }) => (
                    <Button variant="secondary" onClick={open}>
                      {__('Select Image', 'prorank-seo')}
                    </Button>
                  )}
                />
              )}
            </div>
          </MediaUploadCheck>
          <p className="description">
            {twitterCardType === 'summary_large_image'
              ? __(
                  'Recommended size: 1200 x 600 pixels. Leave empty to use Facebook image.',
                  'prorank-seo'
                )
              : __(
                  'Recommended size: 144 x 144 pixels. Leave empty to use Facebook image.',
                  'prorank-seo'
                )}
          </p>
        </div>

        {/* X Preview */}
        <div className="prorank-field-group">
          <h4>{__('X Preview', 'prorank-seo')}</h4>
          <SocialPreview
            platform="twitter"
            cardType={twitterCardType}
            title={twitterTitle || ogTitle || defaults.title}
            description={twitterDescription || ogDescription || defaults.description}
            imageUrl={twitterImageUrl || ogImageUrl || defaults.imageUrl}
          />
        </div>
      </div>

      {/* Save Status */}
      {isSaving === true && (
        <div className="prorank-save-status prorank-save-status--saving">
          <Spinner />
          <span>{__('Saving…', 'prorank-seo')}</span>
        </div>
      )}

      {saveError && (
        <div className="prorank-save-status prorank-save-status--error">
          <span className="dashicons dashicons-warning"></span>
          <span>{saveError}</span>
        </div>
      )}
    </div>
  );
};

export default SocialTabContent;
