/**
 * Meta Tab Content Component (Free)
 *
 * @package
 */

import { __ } from '@wordpress/i18n';
import {
  TextControl,
  TextareaControl,
  CheckboxControl,
  Spinner,
  Notice,
  Card,
  CardBody,
} from '@wordpress/components';
import { useState, useEffect, useCallback, useMemo } from '@wordpress/element';
import apiFetch from '@wordpress/api-fetch';

// Access lodash from WordPress global
const { debounce } = window.lodash || { debounce: (fn) => fn };

// Import components
import SERPPreview from './components/SERPPreview';
import CharacterCounter from './components/CharacterCounter';

const MetaTabContent = ({ postId }) => {
  // State for form fields
  const [seoTitle, setSeoTitle] = useState('');
  const [seoDescription, setSeoDescription] = useState('');
  const [focusKeyword, setFocusKeyword] = useState('');
  const [canonicalUrl, setCanonicalUrl] = useState('');
  const [metaRobots, setMetaRobots] = useState([]);

  // State for UI
  const [isLoading, setIsLoading] = useState(true);
  const [isSaving, setIsSaving] = useState(false);
  const [saveError, setSaveError] = useState(null);
  const [defaults, setDefaults] = useState({ title: '', url: '' });

  // Fetch existing meta data on mount
  useEffect(() => {
    if (!postId) {
      setIsLoading(false);
      return;
    }

    setIsLoading(true);

    apiFetch({
      path: `/prorank-seo/v1/meta/${postId}`,
      method: 'GET',
    })
      .then((response) => {
        if (response.success && response.data) {
          setSeoTitle(response.data.seo_title || '');
          setSeoDescription(response.data.seo_description || '');
          setFocusKeyword(response.data.focus_keyword || '');
          setCanonicalUrl(response.data.canonical_url || '');
          setMetaRobots(response.data.meta_robots || []);
          setDefaults(response.data.defaults || { title: '', url: '' });
        }
      })
      .catch(() => {
        setSaveError(__('Failed to load meta data.', 'prorank-seo'));
      })
      .finally(() => {
        setIsLoading(false);
      });
  }, [postId]);

  // Create save function
  const saveMetaDataImpl = useCallback(
    (data) => {
      if (!postId) return;

      setIsSaving(true);
      setSaveError(null);

      apiFetch({
        path: `/prorank-seo/v1/meta/${postId}`,
        method: 'POST',
        data: {
          seo_title: seoTitle,
          seo_description: seoDescription,
          focus_keyword: focusKeyword,
          canonical_url: canonicalUrl,
          meta_robots: metaRobots,
          ...data,
        },
      })
        .then((response) => {
          if (!response.success) {
            throw new Error(response.message || __('Save failed', 'prorank-seo'));
          }
        })
        .catch((error) => {
          setSaveError(error.message || __('Failed to save changes.', 'prorank-seo'));
        })
        .finally(() => {
          setIsSaving(false);
        });
    },
    [postId, seoTitle, seoDescription, focusKeyword, canonicalUrl, metaRobots]
  );

  const saveMetaData = useMemo(() => debounce(saveMetaDataImpl, 500), [saveMetaDataImpl]);

  const handleTitleChange = (value) => {
    setSeoTitle(value);
    saveMetaData({ seo_title: value });
  };

  const handleDescriptionChange = (value) => {
    setSeoDescription(value);
    saveMetaData({ seo_description: value });
  };

  const handleKeywordChange = (value) => {
    setFocusKeyword(value);
    saveMetaData({ focus_keyword: value });
  };

  const handleCanonicalChange = (value) => {
    setCanonicalUrl(value);
    saveMetaData({ canonical_url: value });
  };

  const handleRobotsChange = useCallback(
    (directive, checked) => {
      const newRobots = checked === true
        ? [...metaRobots, directive]
        : metaRobots.filter((r) => r !== directive);

      setMetaRobots(newRobots);
      saveMetaData({ meta_robots: newRobots });
    },
    [metaRobots, saveMetaData]
  );

  if (isLoading) {
    return (
      <div className="prorank-loading">
        <Spinner />
        <span>{__('Loading meta data…', 'prorank-seo')}</span>
      </div>
    );
  }

  return (
    <div className="prorank-meta-tab">
      {saveError && (
        <Notice status="error" isDismissible={true} onRemove={() => setSaveError(null)}>
          {saveError}
        </Notice>
      )}

      <div className="prorank-field-group prorank-field-group--preview">
        <SERPPreview
          title={seoTitle || defaults.title}
          url={canonicalUrl || defaults.url}
          description={seoDescription}
        />
      </div>

      <div className="prorank-field-group">
        <TextControl
          label={__('SEO Title', 'prorank-seo')}
          value={seoTitle}
          onChange={handleTitleChange}
          help={__('The SEO title appears in search results and browser tabs.', 'prorank-seo')}
          placeholder={defaults.title || __('Post Title', 'prorank-seo')}
          __nextHasNoMarginBottom={true}
          __next40pxDefaultSize={true}
        />
        <CharacterCounter value={seoTitle} recommended={60} minimum={10} type="title" />
      </div>

      <div className="prorank-field-group">
        <TextareaControl
          label={__('Meta Description', 'prorank-seo')}
          value={seoDescription}
          onChange={handleDescriptionChange}
          help={__('Summarize your content for search engines and readers.', 'prorank-seo')}
          placeholder={__('Write a compelling meta description…', 'prorank-seo')}
          rows={3}
          __nextHasNoMarginBottom={true}
          __next40pxDefaultSize={true}
        />
        <CharacterCounter value={seoDescription} recommended={160} minimum={50} type="description" />
      </div>

      <div className="prorank-field-group">
        <TextControl
          label={__('Focus Keyword', 'prorank-seo')}
          value={focusKeyword}
          onChange={handleKeywordChange}
          help={__('Primary keyword or phrase this content targets.', 'prorank-seo')}
          placeholder={__('e.g. wordpress performance', 'prorank-seo')}
          __nextHasNoMarginBottom={true}
          __next40pxDefaultSize={true}
        />
      </div>

      <div className="prorank-field-group">
        <TextControl
          label={__('Canonical URL', 'prorank-seo')}
          type="url"
          value={canonicalUrl}
          onChange={handleCanonicalChange}
          help={__(
            'The canonical URL helps prevent duplicate content issues. Leave empty to use the default permalink.',
            'prorank-seo'
          )}
          placeholder={defaults.url || __('https://example.com/page', 'prorank-seo')}
          __nextHasNoMarginBottom={true}
          __next40pxDefaultSize={true}
        />
      </div>

      <div className="prorank-field-group">
        <h3>{__('Meta Robots', 'prorank-seo')}</h3>
        <p className="description">
          {__('Control how search engines index and display this content.', 'prorank-seo')}
        </p>

        <Card>
          <CardBody>
            <CheckboxControl
              label={__('No Index', 'prorank-seo')}
              help={__('Prevent search engines from indexing this page.', 'prorank-seo')}
              checked={metaRobots.includes('noindex')}
              onChange={(checked) => handleRobotsChange('noindex', checked)}
              __nextHasNoMarginBottom={true}
            />

            <CheckboxControl
              label={__('No Follow', 'prorank-seo')}
              help={__('Prevent search engines from following links on this page.', 'prorank-seo')}
              checked={metaRobots.includes('nofollow')}
              onChange={(checked) => handleRobotsChange('nofollow', checked)}
              __nextHasNoMarginBottom={true}
            />

            <CheckboxControl
              label={__('No Archive', 'prorank-seo')}
              help={__('Prevent search engines from showing cached version.', 'prorank-seo')}
              checked={metaRobots.includes('noarchive')}
              onChange={(checked) => handleRobotsChange('noarchive', checked)}
              __nextHasNoMarginBottom={true}
            />

            <CheckboxControl
              label={__('No Snippet', 'prorank-seo')}
              help={__('Prevent search engines from showing text snippets.', 'prorank-seo')}
              checked={metaRobots.includes('nosnippet')}
              onChange={(checked) => handleRobotsChange('nosnippet', checked)}
              __nextHasNoMarginBottom={true}
            />
          </CardBody>
        </Card>
      </div>

      {isSaving && (
        <div className="prorank-save-status prorank-save-status--saving">
          <Spinner />
          <span>{__('Saving…', 'prorank-seo')}</span>
        </div>
      )}
    </div>
  );
};

export default MetaTabContent;
