/**
 * Hreflang Tab Content Component
 *
 * Provides interface for managing manual hreflang URLs in post editor
 *
 * @package ProRank\SEO\Editor
 */

import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import {
  Button,
  TextControl,
  SelectControl,
  Notice,
  /* eslint-disable @wordpress/no-unsafe-wp-apis */
  __experimentalText as Text,
  __experimentalSpacer as Spacer,
  /* eslint-enable @wordpress/no-unsafe-wp-apis */
} from '@wordpress/components';
// Access icons from window.wp.icons at render time to avoid shim timing issues
const getWpIcon = (name) => window.wp?.icons?.[name];
const WpIcon = (props) => {
  const IconComponent = window.wp?.icons?.Icon;
  return IconComponent ? <IconComponent {...props} /> : null;
};
import { select, dispatch } from '@wordpress/data';
import apiFetch from '@wordpress/api-fetch';

/**
 * Language options for hreflang
 */
const LANGUAGE_OPTIONS = [
  { label: __('Select language', 'prorank-seo'), value: '' },
  { label: 'English (en)', value: 'en' },
  { label: 'English - US (en-US)', value: 'en-US' },
  { label: 'English - UK (en-GB)', value: 'en-GB' },
  { label: 'Spanish (es)', value: 'es' },
  { label: 'Spanish - Spain (es-ES)', value: 'es-ES' },
  { label: 'Spanish - Mexico (es-MX)', value: 'es-MX' },
  { label: 'French (fr)', value: 'fr' },
  { label: 'French - France (fr-FR)', value: 'fr-FR' },
  { label: 'French - Canada (fr-CA)', value: 'fr-CA' },
  { label: 'German (de)', value: 'de' },
  { label: 'German - Germany (de-DE)', value: 'de-DE' },
  { label: 'German - Austria (de-AT)', value: 'de-AT' },
  { label: 'Italian (it)', value: 'it' },
  { label: 'Italian - Italy (it-IT)', value: 'it-IT' },
  { label: 'Portuguese (pt)', value: 'pt' },
  { label: 'Portuguese - Brazil (pt-BR)', value: 'pt-BR' },
  { label: 'Portuguese - Portugal (pt-PT)', value: 'pt-PT' },
  { label: 'Dutch (nl)', value: 'nl' },
  { label: 'Dutch - Netherlands (nl-NL)', value: 'nl-NL' },
  { label: 'Russian (ru)', value: 'ru' },
  { label: 'Japanese (ja)', value: 'ja' },
  { label: 'Chinese - Simplified (zh-CN)', value: 'zh-CN' },
  { label: 'Chinese - Traditional (zh-TW)', value: 'zh-TW' },
  { label: 'Korean (ko)', value: 'ko' },
  { label: 'Arabic (ar)', value: 'ar' },
  { label: 'Hebrew (he)', value: 'he' },
  { label: 'Polish (pl)', value: 'pl' },
  { label: 'Czech (cs)', value: 'cs' },
  { label: 'Swedish (sv)', value: 'sv' },
  { label: 'Danish (da)', value: 'da' },
  { label: 'Norwegian (no)', value: 'no' },
  { label: 'Finnish (fi)', value: 'fi' },
  { label: 'x-default', value: 'x-default' },
];

/**
 * Hreflang Tab Content Component
 */
const HreflangTabContent = () => {
  const [hreflangUrls, setHreflangUrls] = useState({});
  const [loading, setLoading] = useState(false);
  const [saving, setSaving] = useState(false);
  const [validationResults, setValidationResults] = useState(null);
  const [previewTags, setPreviewTags] = useState([]);
  const [settings, setSettings] = useState({ generation_mode: 'automatic' });
  const [customLang, setCustomLang] = useState('');
  const [customUrl, setCustomUrl] = useState('');

  // Get current post ID
  const postId = select('core/editor').getCurrentPostId();
  const postType = select('core/editor').getCurrentPostType();

  useEffect(() => {
    loadSettings();
    loadHreflangUrls();
  }, []);

  /**
   * Load module settings
   */
  const loadSettings = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/hreflang',
      });
      // Handle both response formats (InternationalisationModule and SettingsController)
      const settingsData = response.data || response.settings;
      if (settingsData) {
        setSettings(settingsData);
      }
    } catch (error) {
      console.error('Failed to load hreflang settings:', error);
    }
  };

  /**
   * Load existing hreflang URLs
   */
  const loadHreflangUrls = async () => {
    setLoading(true);
    try {
      const post = await apiFetch({
        path: `/wp/v2/${postType}/${postId}`,
      });
      
      if (post.prorank_hreflang_urls) {
        setHreflangUrls(post.prorank_hreflang_urls);
      }
    } catch (error) {
      console.error('Failed to load hreflang URLs:', error);
    } finally {
      setLoading(false);
    }
  };

  /**
   * Save hreflang URLs
   */
  const saveHreflangUrls = async () => {
    setSaving(true);
    try {
      await apiFetch({
        path: `/wp/v2/${postType}/${postId}`,
        method: 'POST',
        data: {
          prorank_hreflang_urls: hreflangUrls,
        },
      });

      // Show success notice
      dispatch('core/notices').createSuccessNotice(
        __('Hreflang URLs saved successfully', 'prorank-seo'),
        { type: 'snackbar' }
      );

      // Trigger validation after save
      validateHreflang();
    } catch (error) {
      dispatch('core/notices').createErrorNotice(
        __('Failed to save hreflang URLs', 'prorank-seo'),
        { type: 'snackbar' }
      );
    } finally {
      setSaving(false);
    }
  };

  /**
   * Add new hreflang URL
   */
  const addHreflangUrl = () => {
    if (!customLang || !customUrl) {
      dispatch('core/notices').createErrorNotice(
        __('Please enter both language code and URL', 'prorank-seo'),
        { type: 'snackbar' }
      );
      return;
    }

    // Validate URL
    try {
      new URL(customUrl);
    } catch (e) {
      dispatch('core/notices').createErrorNotice(
        __('Please enter a valid URL', 'prorank-seo'),
        { type: 'snackbar' }
      );
      return;
    }

    setHreflangUrls({
      ...hreflangUrls,
      [customLang]: customUrl,
    });

    // Clear inputs
    setCustomLang('');
    setCustomUrl('');
  };

  /**
   * Remove hreflang URL
   */
  const removeHreflangUrl = (lang) => {
    const newUrls = { ...hreflangUrls };
    delete newUrls[lang];
    setHreflangUrls(newUrls);
  };

  /**
   * Validate hreflang implementation
   */
  const validateHreflang = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/hreflang/validate',
        method: 'POST',
      });

      if (response.success && response.data) {
        setValidationResults(response.data);
      }
    } catch (error) {
      console.error('Failed to validate hreflang:', error);
    }
  };

  /**
   * Get hreflang preview
   */
  const getPreview = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/hreflang/preview',
      });

      if (response.success && response.data) {
        setPreviewTags(response.data.tags);
      }
    } catch (error) {
      console.error('Failed to get hreflang preview:', error);
    }
  };

  // Only show in manual mode
  if (settings.generation_mode !== 'manual') {
    return (
      <div className="prorank-hreflang-tab">
        <Notice status="info" isDismissible={false}>
          <Text>
            {__(
              'Hreflang is currently set to automatic mode. Switch to manual mode in the Technical SEO > Internationalisation settings to manage hreflang URLs here.',
              'prorank-seo'
            )}
          </Text>
        </Notice>
      </div>
    );
  }

  if (loading) {
    return (
      <div className="prorank-hreflang-tab">
        <Text>{__('Loading hreflang URLs...', 'prorank-seo')}</Text>
      </div>
    );
  }

  return (
    <div className="prorank-hreflang-tab">
      <div className="prorank-hreflang-header">
        <WpIcon icon={getWpIcon('globe')} />
        <h3>{__('Manual Hreflang URLs', 'prorank-seo')}</h3>
      </div>

      <Text className="prorank-hreflang-description">
        {__(
          'Configure alternate language versions of this page for international SEO.',
          'prorank-seo'
        )}
      </Text>

      <Spacer marginTop={4} />

      {/* Existing URLs */}
      {Object.keys(hreflangUrls).length > 0 && (
        <div className="prorank-hreflang-list">
          <h4>{__('Configured URLs', 'prorank-seo')}</h4>
          {Object.entries(hreflangUrls).map(([lang, url]) => (
            <div key={lang} className="prorank-hreflang-item">
              <span className="prorank-hreflang-lang">{lang}</span>
              <span className="prorank-hreflang-url">{url}</span>
              <Button
                icon={getWpIcon('trash')}
                label={__('Remove', 'prorank-seo')}
                onClick={() => removeHreflangUrl(lang)}
                isDestructive
                isSmall
              />
            </div>
          ))}
        </div>
      )}

      {/* Add new URL */}
      <div className="prorank-hreflang-add">
        <h4>{__('Add Language Version', 'prorank-seo')}</h4>
        <div className="prorank-hreflang-add-form">
          <SelectControl
            label={__('Language', 'prorank-seo')}
            value={customLang}
            options={LANGUAGE_OPTIONS}
            onChange={setCustomLang}
          />
          <TextControl
            label={__('URL', 'prorank-seo')}
            value={customUrl}
            onChange={setCustomUrl}
            placeholder="https://example.com/page"
            type="url"
          />
          <Button
            icon={getWpIcon('plus')}
            variant="secondary"
            onClick={addHreflangUrl}
            disabled={!customLang || !customUrl}
          >
            {__('Add', 'prorank-seo')}
          </Button>
        </div>
      </div>

      {/* Validation Results */}
      {validationResults && (
        <div className="prorank-hreflang-validation">
          <h4>{__('Validation Results', 'prorank-seo')}</h4>
          {validationResults.errors.length > 0 && (
            <Notice status="error" isDismissible={false}>
              <ul>
                {validationResults.errors.map((error, index) => (
                  <li key={index}>{error}</li>
                ))}
              </ul>
            </Notice>
          )}
          {validationResults.warnings.length > 0 && (
            <Notice status="warning" isDismissible={false}>
              <ul>
                {validationResults.warnings.map((warning, index) => (
                  <li key={index}>{warning}</li>
                ))}
              </ul>
            </Notice>
          )}
          {validationResults.valid && validationResults.errors.length === 0 && validationResults.warnings.length === 0 && (
            <Notice status="success" isDismissible={false}>
              <Text>{__('Hreflang configuration is valid!', 'prorank-seo')}</Text>
            </Notice>
          )}
        </div>
      )}

      {/* Preview */}
      {previewTags.length > 0 && (
        <div className="prorank-hreflang-preview">
          <h4>{__('Generated Tags Preview', 'prorank-seo')}</h4>
          <pre className="prorank-hreflang-preview-code">
            {previewTags.join('\n')}
          </pre>
        </div>
      )}

      {/* Actions */}
      <div className="prorank-hreflang-actions">
        <Button
          variant="primary"
          onClick={saveHreflangUrls}
          isBusy={saving}
          disabled={saving || Object.keys(hreflangUrls).length === 0}
        >
          {saving ? __('Saving...', 'prorank-seo') : __('Save Hreflang URLs', 'prorank-seo')}
        </Button>
        <Button
          variant="secondary"
          onClick={validateHreflang}
          disabled={Object.keys(hreflangUrls).length === 0}
        >
          {__('Validate', 'prorank-seo')}
        </Button>
        <Button
          variant="secondary"
          onClick={getPreview}
          disabled={Object.keys(hreflangUrls).length === 0}
        >
          {__('Preview Tags', 'prorank-seo')}
        </Button>
      </div>

      {/* Help Text */}
      <div className="prorank-hreflang-help">
        <h4>{__('Best Practices', 'prorank-seo')}</h4>
        <ul>
          <li>{__('Include a self-referencing hreflang tag for the current page language', 'prorank-seo')}</li>
          <li>{__('Use proper language codes (e.g., en, en-US, fr-CA)', 'prorank-seo')}</li>
          <li>{__('Ensure all URLs return 200 status codes', 'prorank-seo')}</li>
          <li>{__('Keep hreflang tags consistent across all language versions', 'prorank-seo')}</li>
          <li>{__('Use x-default for international/fallback pages', 'prorank-seo')}</li>
        </ul>
      </div>
    </div>
  );
};

export default HreflangTabContent;