/**
 * ProRank SEO Breadcrumbs Block
 *
 * Gutenberg block for displaying breadcrumbs navigation with schema markup.
 */

import { registerBlockType } from '@wordpress/blocks';
import { InspectorControls, useBlockProps } from '@wordpress/block-editor';
import { PanelBody, TextControl, ToggleControl, SelectControl } from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import ServerSideRender from '@wordpress/server-side-render';

/**
 * Block Icon
 */
const BreadcrumbsIcon = () => (
    <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24">
        <path fill="currentColor" d="M3 6h18v2H3V6zm0 5h18v2H3v-2zm0 5h18v2H3v-2z" />
        <path fill="currentColor" d="M9 4l-2 2 2 2M15 4l2 2-2 2" opacity="0.5" />
    </svg>
);

/**
 * Edit Component
 */
const Edit = ({ attributes, setAttributes }) => {
    const blockProps = useBlockProps({
        className: 'prorank-breadcrumbs-block-editor',
    });

    const {
        separator,
        homeText,
        showCurrent,
        showHome,
        homeIcon,
        style,
        displayStyle,
    } = attributes;

    return (
        <div {...blockProps}>
            <InspectorControls>
                <PanelBody title={__('Breadcrumb Settings', 'prorank-seo')} initialOpen={true}>
                    <ToggleControl
                        label={__('Show Home', 'prorank-seo')}
                        checked={showHome}
                        onChange={(value) => setAttributes({ showHome: value })}
                        help={__('Display the home link at the start of breadcrumbs.', 'prorank-seo')}
                    />
                    {showHome && (
                        <>
                            <TextControl
                                label={__('Home Text', 'prorank-seo')}
                                value={homeText}
                                onChange={(value) => setAttributes({ homeText: value })}
                                help={__('Text to display for the home link.', 'prorank-seo')}
                            />
                            <ToggleControl
                                label={__('Show Home Icon', 'prorank-seo')}
                                checked={homeIcon}
                                onChange={(value) => setAttributes({ homeIcon: value })}
                                help={__('Display a home icon before the home text.', 'prorank-seo')}
                            />
                        </>
                    )}
                    <ToggleControl
                        label={__('Show Current Page', 'prorank-seo')}
                        checked={showCurrent}
                        onChange={(value) => setAttributes({ showCurrent: value })}
                        help={__('Display the current page title at the end.', 'prorank-seo')}
                    />
                    <TextControl
                        label={__('Separator', 'prorank-seo')}
                        value={separator}
                        onChange={(value) => setAttributes({ separator: value })}
                        help={__('Character(s) to separate breadcrumb items.', 'prorank-seo')}
                    />
                </PanelBody>
                <PanelBody title={__('Appearance', 'prorank-seo')} initialOpen={false}>
                    <SelectControl
                        label={__('Separator Style', 'prorank-seo')}
                        value={style}
                        options={[
                            { label: __('Default', 'prorank-seo'), value: 'default' },
                            { label: __('Arrow (→)', 'prorank-seo'), value: 'arrow' },
                            { label: __('Slash (/)', 'prorank-seo'), value: 'slash' },
                            { label: __('Pipe (|)', 'prorank-seo'), value: 'pipe' },
                            { label: __('Chevron (›)', 'prorank-seo'), value: 'chevron' },
                        ]}
                        onChange={(value) => setAttributes({ style: value })}
                    />
                    <SelectControl
                        label={__('Display Style', 'prorank-seo')}
                        value={displayStyle}
                        options={[
                            { label: __('Inline', 'prorank-seo'), value: 'inline' },
                            { label: __('With Background', 'prorank-seo'), value: 'background' },
                            { label: __('Compact', 'prorank-seo'), value: 'compact' },
                            { label: __('Large', 'prorank-seo'), value: 'large' },
                        ]}
                        onChange={(value) => setAttributes({ displayStyle: value })}
                    />
                </PanelBody>
            </InspectorControls>
            <ServerSideRender
                block="prorank-seo/breadcrumbs"
                attributes={attributes}
                EmptyResponsePlaceholder={() => (
                    <div className="prorank-breadcrumbs-placeholder">
                        <BreadcrumbsIcon />
                        <span>{__('Breadcrumbs will appear here on the frontend.', 'prorank-seo')}</span>
                    </div>
                )}
            />
        </div>
    );
};

/**
 * Register Block
 */
registerBlockType('prorank-seo/breadcrumbs', {
    title: __('ProRank Breadcrumbs', 'prorank-seo'),
    description: __('Display navigational breadcrumbs with schema markup.', 'prorank-seo'),
    category: 'prorank-seo',
    icon: BreadcrumbsIcon,
    keywords: [
        __('breadcrumbs', 'prorank-seo'),
        __('navigation', 'prorank-seo'),
        __('seo', 'prorank-seo'),
        __('schema', 'prorank-seo'),
    ],
    supports: {
        html: false,
        align: true,
    },
    attributes: {
        separator: {
            type: 'string',
            default: '›',
        },
        homeText: {
            type: 'string',
            default: 'Home',
        },
        showCurrent: {
            type: 'boolean',
            default: true,
        },
        showHome: {
            type: 'boolean',
            default: true,
        },
        homeIcon: {
            type: 'boolean',
            default: false,
        },
        style: {
            type: 'string',
            default: 'default',
        },
        displayStyle: {
            type: 'string',
            default: 'inline',
        },
    },
    edit: Edit,
    save: () => null, // Server-side rendered
});
