/**
 * Axe-core Integration for Accessibility Testing
 *
 * Client-side accessibility testing using axe-core library
 *
 * @package ProRank\SEO\Utils
 * @since   1.0.0
 */

class AxeIntegration {
  constructor() {
    this.axeLoaded = false;
    this.axeScript = null;
    this.results = null;
  }

  resolveAxeUrl() {
    const candidates = [
      window?.prorankSeoAdmin?.pluginUrl,
      window?.prorankSeo?.pluginUrl,
      window?.proRankSeoAdmin?.pluginUrl,
      window?.proRankSEO?.pluginUrl,
    ].filter(Boolean);

    if (candidates.length) {
      return `${String(candidates[0]).replace(/\/+$/, '')}/assets/js/axe-core/axe.min.js`;
    }

    return '';
  }

  /**
   * Load axe-core library from local plugin assets
   * @returns {Promise<boolean>}
   */
  async loadAxeCore() {
    if (this.axeLoaded && window.axe) {
      return true;
    }

    return new Promise((resolve, reject) => {
      // Check if already loaded
      if (window.axe) {
        this.axeLoaded = true;
        resolve(true);
        return;
      }

      // Create script element
      this.axeScript = document.createElement('script');
      const axeUrl = this.resolveAxeUrl();
      if (!axeUrl) {
        console.error('ProRank SEO: pluginUrl is missing for axe-core');
        reject(new Error('Missing plugin URL for axe-core'));
        return;
      }

      this.axeScript.src = axeUrl;
      this.axeScript.async = true;
      
      this.axeScript.onload = () => {
        this.axeLoaded = true;
        this.configureAxe();
        resolve(true);
      };
      
      this.axeScript.onerror = () => {
        console.error('Failed to load axe-core');
        reject(new Error('Failed to load axe-core library'));
      };
      
      document.head.appendChild(this.axeScript);
    });
  }

  /**
   * Configure axe-core with WCAG rules
   */
  configureAxe() {
    if (!window.axe) return;

    // Configure axe for WCAG 2.2 AA compliance
    window.axe.configure({
      rules: [
        // Enable WCAG 2.2 AA rules
        { id: 'area-alt', enabled: true },
        { id: 'aria-allowed-attr', enabled: true },
        { id: 'aria-required-attr', enabled: true },
        { id: 'aria-required-children', enabled: true },
        { id: 'aria-required-parent', enabled: true },
        { id: 'aria-roles', enabled: true },
        { id: 'aria-valid-attr-value', enabled: true },
        { id: 'aria-valid-attr', enabled: true },
        { id: 'audio-caption', enabled: true },
        { id: 'button-name', enabled: true },
        { id: 'bypass', enabled: true },
        { id: 'color-contrast', enabled: true },
        { id: 'definition-list', enabled: true },
        { id: 'dlitem', enabled: true },
        { id: 'document-title', enabled: true },
        { id: 'duplicate-id-active', enabled: true },
        { id: 'duplicate-id-aria', enabled: true },
        { id: 'duplicate-id', enabled: true },
        { id: 'empty-heading', enabled: true },
        { id: 'focus-order-semantics', enabled: true },
        { id: 'form-field-multiple-labels', enabled: true },
        { id: 'frame-title', enabled: true },
        { id: 'heading-order', enabled: true },
        { id: 'html-has-lang', enabled: true },
        { id: 'html-lang-valid', enabled: true },
        { id: 'image-alt', enabled: true },
        { id: 'input-image-alt', enabled: true },
        { id: 'label', enabled: true },
        { id: 'landmark-one-main', enabled: true },
        { id: 'link-name', enabled: true },
        { id: 'list', enabled: true },
        { id: 'listitem', enabled: true },
        { id: 'meta-refresh', enabled: true },
        { id: 'object-alt', enabled: true },
        { id: 'role-img-alt', enabled: true },
        { id: 'scrollable-region-focusable', enabled: true },
        { id: 'select-name', enabled: true },
        { id: 'server-side-image-map', enabled: true },
        { id: 'svg-img-alt', enabled: true },
        { id: 'td-headers-attr', enabled: true },
        { id: 'th-has-data-cells', enabled: true },
        { id: 'valid-lang', enabled: true },
        { id: 'video-caption', enabled: true },
      ],
      reporter: 'v2',
      resultTypes: ['violations', 'incomplete', 'inapplicable'],
    });
  }

  /**
   * Run accessibility scan on current page or element
   * @param {string|Element} context - Element or selector to test
   * @param {Object} options - Axe run options
   * @returns {Promise<Object>}
   */
  async runScan(context = document, options = {}) {
    // Ensure axe is loaded
    if (!this.axeLoaded) {
      await this.loadAxeCore();
    }

    const defaultOptions = {
      runOnly: {
        type: 'tag',
        values: ['wcag2a', 'wcag2aa', 'wcag21a', 'wcag21aa', 'wcag22aa'],
      },
      resultTypes: ['violations', 'incomplete'],
      elementRef: true,
    };

    const mergedOptions = { ...defaultOptions, ...options };

    try {
      const results = await window.axe.run(context, mergedOptions);
      this.results = this.processResults(results);
      return this.results;
    } catch (error) {
      console.error('Axe scan error:', error);
      throw error;
    }
  }

  /**
   * Process axe results into ProRank format
   * @param {Object} axeResults - Raw axe-core results
   * @returns {Object}
   */
  processResults(axeResults) {
    const processed = {
      violations: [],
      incomplete: [],
      passes: axeResults.passes?.length || 0,
      timestamp: new Date().toISOString(),
      url: window.location.href,
      stats: {
        total: 0,
        critical: 0,
        serious: 0,
        moderate: 0,
        minor: 0,
      },
    };

    // Process violations
    if (axeResults.violations) {
      axeResults.violations.forEach(violation => {
        violation.nodes.forEach(node => {
          const issue = {
            id: violation.id,
            impact: violation.impact,
            severity: this.mapImpactToSeverity(violation.impact),
            description: violation.description,
            help: violation.help,
            helpUrl: violation.helpUrl,
            wcag: violation.tags.filter(tag => tag.startsWith('wcag')),
            element: node.html,
            target: node.target,
            failureSummary: node.failureSummary,
            fixes: this.generateFixes(violation.id, node),
          };
          
          processed.violations.push(issue);
          processed.stats[violation.impact]++;
          processed.stats.total++;
        });
      });
    }

    // Process incomplete (needs review)
    if (axeResults.incomplete) {
      axeResults.incomplete.forEach(incomplete => {
        incomplete.nodes.forEach(node => {
          processed.incomplete.push({
            id: incomplete.id,
            description: incomplete.description,
            help: incomplete.help,
            helpUrl: incomplete.helpUrl,
            element: node.html,
            target: node.target,
            message: node.message || 'Needs manual review',
          });
        });
      });
    }

    return processed;
  }

  /**
   * Map axe impact levels to ProRank severity
   * @param {string} impact - Axe impact level
   * @returns {string}
   */
  mapImpactToSeverity(impact) {
    const mapping = {
      critical: 'error',
      serious: 'error',
      moderate: 'warning',
      minor: 'notice',
    };
    return mapping[impact] || 'warning';
  }

  /**
   * Generate fix suggestions based on violation type
   * @param {string} violationId - Violation ID
   * @param {Object} node - Node with violation
   * @returns {Array}
   */
  generateFixes(violationId, node) {
    const fixes = [];

    switch (violationId) {
      case 'image-alt':
        fixes.push({
          type: 'auto',
          description: 'Add alt attribute to image',
          code: `alt="[descriptive text]"`,
        });
        break;
        
      case 'button-name':
        fixes.push({
          type: 'manual',
          description: 'Add accessible text to button',
          code: `<button>Click here</button> or <button aria-label="Submit form">Icon</button>`,
        });
        break;
        
      case 'color-contrast':
        fixes.push({
          type: 'manual',
          description: 'Adjust foreground or background color for better contrast',
          recommendation: 'Ensure contrast ratio is at least 4.5:1 for normal text, 3:1 for large text',
        });
        break;
        
      case 'heading-order':
        fixes.push({
          type: 'manual',
          description: 'Fix heading hierarchy to not skip levels',
          recommendation: 'Use h1, then h2, then h3, etc. in sequential order',
        });
        break;
        
      case 'label':
        fixes.push({
          type: 'auto',
          description: 'Add label to form input',
          code: `<label for="input-id">Label text</label>`,
        });
        break;
        
      case 'link-name':
        fixes.push({
          type: 'manual',
          description: 'Add descriptive text to link',
          recommendation: 'Avoid generic text like "click here" or "read more"',
        });
        break;
        
      default:
        fixes.push({
          type: 'manual',
          description: 'Review and fix based on WCAG guidelines',
          helpUrl: node.helpUrl,
        });
    }

    return fixes;
  }

  /**
   * Highlight elements with violations on the page
   * @param {Array} violations - Array of violations
   */
  highlightViolations(violations) {
    // Remove existing highlights
    this.clearHighlights();

    violations.forEach((violation, index) => {
      violation.target.forEach(selector => {
        try {
          const element = document.querySelector(selector.join(' '));
          if (element) {
            element.style.outline = `3px solid ${this.getHighlightColor(violation.severity)}`;
            element.style.outlineOffset = '2px';
            element.setAttribute('data-prorank-a11y-issue', violation.id);
            element.setAttribute('data-prorank-a11y-index', index);
          }
        } catch (e) {
          console.warn('Could not highlight element:', selector, e);
        }
      });
    });
  }

  /**
   * Clear all highlights from the page
   */
  clearHighlights() {
    const highlighted = document.querySelectorAll('[data-prorank-a11y-issue]');
    highlighted.forEach(element => {
      element.style.outline = '';
      element.style.outlineOffset = '';
      element.removeAttribute('data-prorank-a11y-issue');
      element.removeAttribute('data-prorank-a11y-index');
    });
  }

  /**
   * Get highlight color based on severity
   * @param {string} severity - Issue severity
   * @returns {string}
   */
  getHighlightColor(severity) {
    const colors = {
      error: '#dc3545',
      warning: '#f0b849',
      notice: '#0073aa',
    };
    return colors[severity] || '#666';
  }

  /**
   * Generate accessibility report HTML
   * @param {Object} results - Scan results
   * @returns {string}
   */
  generateReportHTML(results) {
    let html = `
      <div class="prorank-axe-report">
        <h2>Accessibility Scan Results</h2>
        <p>URL: ${results.url}</p>
        <p>Timestamp: ${new Date(results.timestamp).toLocaleString()}</p>
        
        <div class="prorank-axe-stats">
          <h3>Summary</h3>
          <ul>
            <li>Total Issues: ${results.stats.total}</li>
            <li>Critical: ${results.stats.critical}</li>
            <li>Serious: ${results.stats.serious}</li>
            <li>Moderate: ${results.stats.moderate}</li>
            <li>Minor: ${results.stats.minor}</li>
            <li>Passed Tests: ${results.passes}</li>
            <li>Needs Review: ${results.incomplete.length}</li>
          </ul>
        </div>
        
        <div class="prorank-axe-violations">
          <h3>Violations</h3>
    `;

    results.violations.forEach(violation => {
      html += `
        <div class="prorank-axe-violation prorank-axe-${violation.severity}">
          <h4>${violation.help}</h4>
          <p>${violation.description}</p>
          <p><strong>Impact:</strong> ${violation.impact}</p>
          <p><strong>WCAG:</strong> ${violation.wcag.join(', ')}</p>
          <details>
            <summary>Affected Element</summary>
            <code>${this.escapeHtml(violation.element)}</code>
          </details>
          <details>
            <summary>How to Fix</summary>
            <p>${violation.failureSummary}</p>
            ${violation.fixes.map(fix => `
              <div class="prorank-axe-fix">
                <strong>${fix.type === 'auto' ? '🔧 Auto-fixable' : '✋ Manual fix'}</strong>
                <p>${fix.description}</p>
                ${fix.code ? `<code>${this.escapeHtml(fix.code)}</code>` : ''}
              </div>
            `).join('')}
          </details>
          <a href="${violation.helpUrl}" target="_blank">Learn more →</a>
        </div>
      `;
    });

    html += `</div></div>`;
    return html;
  }

  /**
   * Escape HTML for safe display
   * @param {string} html - HTML string
   * @returns {string}
   */
  escapeHtml(html) {
    const div = document.createElement('div');
    div.textContent = html;
    return div.innerHTML;
  }

  /**
   * Send results to server for storage
   * @param {Object} results - Scan results
   * @returns {Promise<Object>}
   */
  async sendToServer(results) {
    try {
      const response = await fetch('/wp-json/prorank-seo/v1/accessibility/client-scan', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-WP-Nonce': window.wpApiSettings?.nonce || '',
        },
        body: JSON.stringify({
          url: results.url,
          violations: results.violations,
          incomplete: results.incomplete,
          stats: results.stats,
        }),
      });

      if (!response.ok) {
        throw new Error('Failed to send results to server');
      }

      return await response.json();
    } catch (error) {
      console.error('Error sending results to server:', error);
      throw error;
    }
  }
}

// Export singleton instance
export default new AxeIntegration();
