/**
 * Advanced Settings Types and Schema
 *
 * @module SupportAccount/AdvancedSettings
 * @since 1.0.0
 */

import { z } from 'zod';

/**
 * Advanced settings validation schema
 */
export const advancedSettingsSchema = z.object({
  // Debug & Development
  debug_mode: z.boolean().default(false),
  log_level: z.enum(['error', 'warning', 'info', 'debug']).default('error'),

  // Performance
  api_timeout: z.number().int().min(5).max(120).default(30),
  cache_duration: z.number().int().min(60).max(86400).default(3600),

  // Security
  disable_xmlrpc: z.boolean().default(true),
  disable_rest_api_public: z.boolean().default(false),
  remove_version_strings: z.boolean().default(true),
  disable_file_editing: z.boolean().default(true),

  // WordPress Features
  disable_emojis: z.boolean().default(false),
  disable_gutenberg: z.boolean().default(false),
});

/**
 * TypeScript type for advanced settings
 */
export type AdvancedSettings = z.infer<typeof advancedSettingsSchema>;

/**
 * Default values for advanced settings
 */
export const advancedSettingsDefaults: AdvancedSettings = advancedSettingsSchema.parse({});

/**
 * Validate advanced settings
 */
export function validateAdvancedSettings(data: unknown): AdvancedSettings {
  return advancedSettingsSchema.parse(data);
}

/**
 * Safely validate advanced settings (returns validation result)
 */
export function safeValidateAdvancedSettings(data: unknown) {
  return advancedSettingsSchema.safeParse(data);
}

/**
 * Tab component props
 */
export interface TabComponentProps {
  addNotice: (type: 'success' | 'error' | 'warning' | 'info', message: string) => void;
}

/**
 * Import result type
 */
export interface ImportResult {
  success: boolean;
  message?: string;
  stats?: {
    posts?: number;
    pages?: number;
    terms?: number;
    redirects?: number;
    settings?: number;
    errors?: string[];
  };
}

/**
 * Export settings response
 */
export interface ExportResponse {
  version: string;
  exported_at: string;
  settings: Record<string, unknown>;
  meta?: Record<string, unknown>;
  redirects?: unknown[];
}

/**
 * Competitor source types
 */
export type CompetitorSource = 'yoast' | 'rankmath' | 'aioseo' | 'seopress';

/**
 * Log level options for UI
 */
export const logLevelOptions = [
  { label: 'Error Only', value: 'error' as const },
  { label: 'Warning', value: 'warning' as const },
  { label: 'Info', value: 'info' as const },
  { label: 'Debug', value: 'debug' as const },
] as const;

/**
 * Competitor options for UI
 */
export const competitorOptions = [
  { label: 'Yoast SEO', value: 'yoast' as const },
  { label: 'Rank Math', value: 'rankmath' as const },
  { label: 'All in One SEO', value: 'aioseo' as const },
  { label: 'SEOPress', value: 'seopress' as const },
] as const;
