# CSS Migration Guide - From Unified to Modular

## Overview
This guide helps developers migrate from the old unified CSS (`prorank-unified.css`) to the new modular CSS architecture.

## Quick Start

### For New Components
```javascript
// ✅ Use the new modular CSS
import '@admin/styles/index.css';

// ❌ Don't import the old unified CSS
import '@admin/styles/prorank-unified.css';
```

### Component Class Names
```jsx
// ✅ Use new BEM classes with modifiers
<div className="prorank-card prorank-card--metric">
  <div className="prorank-card__header">
    <h3 className="prorank-card__title">Page Views</h3>
  </div>
  <div className="prorank-card__body">
    <div className="prorank-metric-value">1,234</div>
    <div className="prorank-metric-label">Total Views</div>
  </div>
</div>

// ❌ Avoid creating new card variants
<div className="prorank-pageviews-card">...</div>
```

## Migration Steps

### Step 1: Update Your Import
Replace the unified CSS import with the modular index:

```diff
- import '@admin/styles/prorank-unified.css';
+ import '@admin/styles/index.css';
```

### Step 2: Use Design Tokens
Replace hard-coded values with CSS variables:

```diff
// ❌ Old way - hard-coded colors
- style={{ color: '#2563eb', backgroundColor: '#f3f4f6' }}

// ✅ New way - CSS variables
+ style={{ color: 'var(--prorank-primary)', backgroundColor: 'var(--prorank-bg-secondary)' }}
```

### Step 3: Update Component Classes

#### Cards
```diff
- <div className="prorank-metric-card">
+ <div className="prorank-card prorank-card--metric">

- <div className="prorank-stat-card">
+ <div className="prorank-card prorank-card--stat">

- <div className="prorank-feature-card">
+ <div className="prorank-card prorank-card--feature">
```

#### Buttons
```diff
- <button className="prorank-btn-primary">
+ <button className="prorank-button prorank-button--primary">

- <button className="prorank-btn-orange">
+ <button className="prorank-button prorank-button--accent">
```

#### Forms
```diff
- <input className="prorank-text-input" />
+ <input className="prorank-input" />

- <div className="prorank-toggle-switch">
+ <div className="prorank-toggle">
```

#### Badges
```diff
- <span className="prorank-badge-success">
+ <span className="prorank-badge prorank-badge--success">

- <span className="prorank-badge-pro">
+ <span className="prorank-badge prorank-badge--premium">
```

## Component Reference

### Cards
Base class: `prorank-card`
Modifiers:
- `prorank-card--metric` - For stats/metrics
- `prorank-card--feature` - For feature highlights
- `prorank-card--task` - For task items
- `prorank-card--campaign` - For campaigns
- `prorank-card--compact` - Reduced padding
- `prorank-card--premium` - Pro features

### Buttons
Base class: `prorank-button`
Modifiers:
- `prorank-button--primary` - Primary action (blue)
- `prorank-button--secondary` - Secondary action
- `prorank-button--accent` - Accent/CTA (orange)
- `prorank-button--danger` - Destructive action
- `prorank-button--small` - Small size
- `prorank-button--large` - Large size

### Forms
Input: `prorank-input`
Textarea: `prorank-textarea`
Select: `prorank-select`
Checkbox: `prorank-checkbox`
Radio: `prorank-radio`
Toggle: `prorank-toggle`

### Tables
Base class: `prorank-table`
Modifiers:
- `prorank-table--striped` - Alternating row colors
- `prorank-table--compact` - Reduced padding
- `prorank-table--responsive` - Mobile-friendly

### Modals
Base class: `prorank-modal`
Overlay: `prorank-modal-overlay`
Modifiers:
- `prorank-modal--small` - 400px max width
- `prorank-modal--large` - 900px max width
- `prorank-modal--confirm` - Confirmation dialog
- `prorank-modal--premium` - Pro feature modal

### Badges
Base class: `prorank-badge`
Modifiers:
- `prorank-badge--success` - Green
- `prorank-badge--warning` - Yellow
- `prorank-badge--error` - Red
- `prorank-badge--info` - Blue
- `prorank-badge--premium` - Pro/Standard

## CSS Variables Reference

### Colors
```css
--prorank-primary: #2563eb;      /* Blue */
--prorank-accent: #ff6b35;       /* Orange */
--prorank-success: #10b981;      /* Green */
--prorank-warning: #fbbf24;      /* Yellow */
--prorank-error: #ef4444;        /* Red */
```

### Spacing
```css
--prorank-space-xs: 4px;
--prorank-space-sm: 8px;
--prorank-space-md: 16px;
--prorank-space-lg: 24px;
--prorank-space-xl: 32px;
--prorank-space-2xl: 48px;
```

### Typography
```css
--prorank-font-family: -apple-system, BlinkMacSystemFont, ...;
--prorank-text: #111827;
--prorank-text-secondary: #6b7280;
--prorank-text-tertiary: #9ca3af;
```

## Utility Classes

### Spacing
- Margin: `prorank-m-{size}`, `prorank-mt-{size}`, `prorank-mb-{size}`, etc.
- Padding: `prorank-p-{size}`, `prorank-pt-{size}`, `prorank-pb-{size}`, etc.
- Sizes: `0`, `xs`, `sm`, `md`, `lg`, `xl`, `2xl`

### Display
- `prorank-hidden`
- `prorank-block`
- `prorank-flex`
- `prorank-inline-flex`
- `prorank-grid`

### Text
- Alignment: `prorank-text-left`, `prorank-text-center`, `prorank-text-right`
- Weight: `prorank-font-normal`, `prorank-font-medium`, `prorank-font-bold`
- Size: `prorank-text-xs`, `prorank-text-sm`, `prorank-text-lg`, etc.

### Flexbox
- Direction: `prorank-flex-row`, `prorank-flex-col`
- Justify: `prorank-justify-start`, `prorank-justify-center`, `prorank-justify-between`
- Align: `prorank-items-start`, `prorank-items-center`, `prorank-items-end`
- Gap: `prorank-gap-sm`, `prorank-gap-md`, `prorank-gap-lg`

## Common Patterns

### Card with Header
```jsx
<div className="prorank-card">
  <div className="prorank-card__header">
    <h3 className="prorank-card__title">Card Title</h3>
  </div>
  <div className="prorank-card__body">
    {/* Content */}
  </div>
  <div className="prorank-card__footer">
    <button className="prorank-button prorank-button--primary">
      Save Changes
    </button>
  </div>
</div>
```

### Form Group
```jsx
<div className="prorank-form-group">
  <label className="prorank-label">
    Field Label
  </label>
  <input type="text" className="prorank-input" />
  <span className="prorank-help-text">
    Help text goes here
  </span>
</div>
```

### Metric Display
```jsx
<div className="prorank-card prorank-card--metric">
  <div className="prorank-card__body">
    <div className="prorank-metric-value">1,234</div>
    <div className="prorank-metric-label">Page Views</div>
    <div className="prorank-metric-change prorank-metric-change--positive">
      ↑ 12% from last week
    </div>
  </div>
</div>
```

### Modal Dialog
```jsx
<div className="prorank-modal-overlay">
  <div className="prorank-modal">
    <div className="prorank-modal-header">
      <h2 className="prorank-modal-title">Modal Title</h2>
      <button className="prorank-modal-close">×</button>
    </div>
    <div className="prorank-modal-body">
      {/* Content */}
    </div>
    <div className="prorank-modal-footer">
      <button className="prorank-button prorank-button--secondary">
        Cancel
      </button>
      <button className="prorank-button prorank-button--primary">
        Confirm
      </button>
    </div>
  </div>
</div>
```

## Testing Your Migration

### Checklist
- [ ] Updated all CSS imports to use `index.css`
- [ ] Replaced hard-coded colors with CSS variables
- [ ] Updated component classes to use new BEM structure
- [ ] Tested responsive behavior on mobile
- [ ] Verified no visual regressions
- [ ] Build succeeds without errors
- [ ] No console warnings about missing styles

### Build Command
```bash
npm run build
```

### Common Issues

#### Issue: Styles not applying
**Solution**: Make sure you're importing `@admin/styles/index.css` and not the old unified CSS.

#### Issue: Colors look different
**Solution**: Use CSS variables from `core/_variables.css` instead of hard-coded values.

#### Issue: Component spacing is off
**Solution**: Use spacing utilities or CSS variables for consistent spacing.

#### Issue: Build fails with CSS errors
**Solution**: Check that PostCSS is configured correctly and `postcss-import` is installed.

## Gradual Migration

During the transition period, both systems work together:
1. The new `index.css` imports both modular styles AND the old unified CSS
2. You can migrate components one at a time
3. Once all components are migrated, we'll remove the unified CSS import

## Need Help?

1. Check the [DESIGN_SYSTEM.md](./DESIGN_SYSTEM.md) for detailed documentation
2. Look at existing migrated components for examples
3. Ask in the development channel
4. Create an issue if you find a bug

Remember: **Consistency is key to becoming the #1 SEO plugin!**