# How the ProRank Modular CSS System Works

## Overview
The ProRank plugin now uses a **modular CSS architecture** instead of a single unified CSS file. This makes the styles more maintainable, reusable, and performant.

## How Pages Use These Styles

### 1. **Automatic Import Chain**
```
App.tsx imports → styles/index.css → loads all modular components
```

When any ProRank admin page loads:
1. `src/admin/App.tsx` imports `@admin/styles/index.css` (line 48)
2. `index.css` automatically imports ALL modular components in the correct order
3. Webpack bundles everything into `admin.css` 
4. WordPress loads this single CSS file for all admin pages

### 2. **Component Structure**
```
styles/
├── index.css           # Main entry point (imports everything)
├── core/               # Foundation styles
│   ├── _variables.css  # Design tokens (colors, spacing, etc.)
│   ├── _reset.css      # Browser resets
│   ├── _typography.css # Font styles
│   └── _layout.css     # Page structure & signature header
├── components/         # Reusable UI components
│   ├── _buttons.css    # All button styles
│   ├── _cards.css      # Card/panel layouts
│   ├── _forms.css      # Form controls
│   ├── _tabs.css       # Tab navigation
│   └── _tables.css     # Table styles
└── features/          # Page-specific styles
    ├── _dashboard.css
    ├── _analytics.css
    └── ...
```

### 3. **How to Use in React Components**

**For Buttons:**
```jsx
// Primary orange gradient button
<button className="prorank-button prorank-button--primary">
  Save Changes
</button>

// Secondary blue button
<button className="prorank-button prorank-button--secondary">
  Cancel
</button>

// Danger/delete button
<button className="prorank-button prorank-button--danger">
  Delete
</button>
```

**For Cards:**
```jsx
<div className="prorank-card">
  <div className="prorank-card-header">
    <h3>Card Title</h3>
  </div>
  <div className="prorank-card-body">
    Content goes here
  </div>
</div>
```

**For Forms:**
```jsx
<div className="prorank-control-group">
  <label>Field Label</label>
  <input type="text" className="prorank-input" />
  <span className="prorank-form-help">Help text</span>
</div>
```

**For Tabs (with module toggles):**
```jsx
<div className="prorank-tab-navigation">
  <ul className="prorank-tab-list">
    <li className="prorank-tab-item">
      <button className="prorank-tab-button active">
        <span className="prorank-tab-name">Schema Markup</span>
        {/* Module toggle appears INSIDE the tab button */}
        <div className="prorank-tab-toggle">
          <FormToggle checked={true} />
          <span className="prorank-toggle-label">Module</span>
        </div>
      </button>
    </li>
    <li className="prorank-tab-item">
      <button className="prorank-tab-button">
        <span className="prorank-tab-name">Breadcrumbs</span>
        <div className="prorank-tab-toggle">
          <FormToggle checked={false} />
          <span className="prorank-toggle-label">Module</span>
        </div>
      </button>
    </li>
    <li className="prorank-tab-item">
      {/* Tab without toggle */}
      <button className="prorank-tab-button">
        <span className="prorank-tab-name">Site Basics</span>
      </button>
    </li>
  </ul>
</div>
<div className="prorank-tab-content">
  Tab content here (no toggles in content, they're on the tabs!)
</div>
```

### 4. **The Signature Header**
Every ProRank page automatically gets the premium header with the animated orange accent bar:

```jsx
<div className="prorank-header">
  <div className="prorank-header-content">
    <div className="prorank-header-left">
      <h1>Page Title</h1>
      <p>Page description</p>
    </div>
    <div className="prorank-header-stats">
      {/* Optional stats cards */}
    </div>
  </div>
</div>
```

### 5. **Benefits of This System**

✅ **Consistency**: All pages use the same components = unified look  
✅ **Maintainability**: Change button style in ONE place, updates everywhere  
✅ **Performance**: Only load CSS that's actually used  
✅ **Developer Experience**: Clear naming conventions (`.prorank-*`)  
✅ **No Conflicts**: Prefixed classes avoid WordPress conflicts  

### 6. **Key Design Elements from Original**

The modular system preserves all premium design elements:
- **Orange Gradient Buttons**: Primary CTAs with gradient backgrounds
- **Blue Secondary Buttons**: For secondary actions
- **Animated Header**: Signature header with moving orange accent bar
- **Modern Tabs**: Underline animation on active tabs
- **Focus States**: Purple glow on focused form inputs
- **Hover Effects**: Smooth transitions on all interactive elements
- **Card Shadows**: Subtle shadows that increase on hover

### 7. **For Developers**

When creating new pages/components:
1. **DON'T** import individual CSS files
2. **DON'T** create inline styles
3. **DO** use the predefined classes (`.prorank-button`, `.prorank-card`, etc.)
4. **DO** check existing components first before creating new styles

The system is already loaded globally - just use the classes!

## Example: Complete Page Structure

```jsx
const MyNewPage = () => {
  return (
    <div className="prorank-page">
      {/* Signature header with orange accent */}
      <div className="prorank-header">
        <div className="prorank-header-content">
          <h1>My New Feature</h1>
          <p>Feature description here</p>
        </div>
      </div>
      
      {/* Main content area */}
      <div className="prorank-content">
        {/* Tab navigation */}
        <div className="prorank-tab-navigation">
          {/* tabs here */}
        </div>
        
        {/* Cards grid */}
        <div className="prorank-card-grid-3">
          <div className="prorank-card">
            <div className="prorank-card-body">
              {/* Card content */}
            </div>
          </div>
        </div>
        
        {/* Form section */}
        <div className="prorank-card">
          <div className="prorank-card-body">
            <div className="prorank-control-group">
              <label>Setting Name</label>
              <input type="text" />
            </div>
            <button className="prorank-button prorank-button--primary">
              Save Settings
            </button>
          </div>
        </div>
      </div>
    </div>
  );
};
```

This structure ensures your page matches the ProRank design system perfectly!