# ProRank Design System Documentation

## Overview
The ProRank CSS architecture has been modularized to ensure design consistency while enabling scalable development. This document explains the structure, usage, and migration process.

## Directory Structure

```
src/admin/styles/
├── index.css                    # Main entry point - imports all modules
├── core/                        # Foundation styles (loaded first)
│   ├── _variables.css          # Design tokens - SINGLE SOURCE OF TRUTH
│   ├── _reset.css              # Browser resets & WordPress overrides
│   ├── _typography.css         # Text styles & hierarchy
│   └── _layout.css             # Page structure & signature header
├── components/                  # Reusable UI components
│   ├── _buttons.css            # Button system with modifiers
│   ├── _cards.css              # Unified card component (replaces 7 variants)
│   ├── _forms.css              # Form controls & inputs
│   ├── _tables.css             # Table styles & responsive layouts
│   ├── _modals.css             # Modal & overlay styles (coming soon)
│   ├── _badges.css             # Badges & status indicators (coming soon)
│   └── _stats.css              # Stats & metrics displays (coming soon)
├── features/                    # Feature-specific styles
│   ├── _dashboard.css          # Dashboard page styles (coming soon)
│   ├── _analytics.css          # Analytics features (coming soon)
│   └── ...                     # Other feature styles
├── wordpress/                   # WordPress compatibility
│   └── _overrides.css          # WordPress admin overrides (coming soon)
└── utilities/                   # Helper classes
    ├── _animations.css         # Animation utilities (coming soon)
    └── _helpers.css            # Utility classes (coming soon)
```

## Design Principles

### 1. Consistency First
- **ALL** colors must use CSS variables from `_variables.css`
- **NEVER** hard-code colors, spacing, or shadows
- **ALWAYS** extend base components, don't create new ones

### 2. Component Modifiers (BEM)
Use BEM naming convention for component variations:
```css
/* Base component */
.prorank-card { }

/* Modifier */
.prorank-card--metric { }
.prorank-card--feature { }

/* Sub-element */
.prorank-card__header { }
.prorank-card__body { }
```

### 3. Single Source of Truth
The `core/_variables.css` file contains ALL design tokens:
- Colors (brand, semantic, text)
- Spacing (xs, sm, md, lg, xl, 2xl)
- Typography scales
- Shadows
- Border radius
- Transitions
- Z-index scale

## Component Usage Examples

### Cards
Instead of creating new card types, use the unified card with modifiers:

```html
<!-- Basic Card -->
<div class="prorank-card">
  <div class="prorank-card__header">
    <h3 class="prorank-card__title">Card Title</h3>
  </div>
  <div class="prorank-card__body">
    Content here
  </div>
</div>

<!-- Metric Card -->
<div class="prorank-card prorank-card--metric">
  <div class="prorank-card__body">
    <div class="prorank-metric-value">1,234</div>
    <div class="prorank-metric-label">Total Views</div>
  </div>
</div>

<!-- Feature Card -->
<div class="prorank-card prorank-card--feature">
  <div class="prorank-card__body">
    <div class="prorank-feature-icon">🚀</div>
    <h3>Feature Name</h3>
    <p>Feature description</p>
  </div>
</div>
```

### Buttons
Use button modifiers for different styles:

```html
<!-- Primary Button -->
<button class="prorank-button prorank-button--primary">Save Changes</button>

<!-- Secondary Button -->
<button class="prorank-button prorank-button--secondary">Cancel</button>

<!-- Accent Button (Orange) -->
<button class="prorank-button prorank-button--accent">Get Started</button>

<!-- Danger Button -->
<button class="prorank-button prorank-button--danger">Delete</button>

<!-- Button Sizes -->
<button class="prorank-button prorank-button--primary prorank-button--small">Small</button>
<button class="prorank-button prorank-button--primary prorank-button--large">Large</button>
```

### Forms
Consistent form controls:

```html
<!-- Text Input -->
<div class="prorank-form-group">
  <label class="prorank-label">Field Label</label>
  <input type="text" class="prorank-input" placeholder="Enter text...">
  <span class="prorank-help-text">Help text goes here</span>
</div>

<!-- Toggle Switch -->
<div class="prorank-toggle">
  <input type="checkbox" class="prorank-toggle-input" id="toggle1">
  <label class="prorank-toggle-track" for="toggle1">
    <span class="prorank-toggle-thumb"></span>
  </label>
  <label class="prorank-toggle-label" for="toggle1">Enable Feature</label>
</div>

<!-- Checkbox -->
<div class="prorank-checkbox-wrapper">
  <input type="checkbox" class="prorank-checkbox" id="check1">
  <label class="prorank-checkbox-label" for="check1">
    I agree to the terms
  </label>
</div>
```

## Migration Guide

### Phase 1: Update Imports (Current)
In React components, change:
```javascript
// Old
import '@admin/styles/prorank-unified.css';

// New (during transition)
import '@admin/styles/index.css';
```

The `index.css` currently imports both the new modular styles AND the old unified CSS for backward compatibility.

### Phase 2: Use New Components
Start using the new component classes in new features:
```html
<!-- Old (multiple card types) -->
<div class="prorank-metric-card">...</div>
<div class="prorank-stat-card">...</div>
<div class="prorank-feature-card">...</div>

<!-- New (unified with modifiers) -->
<div class="prorank-card prorank-card--metric">...</div>
<div class="prorank-card prorank-card--stat">...</div>
<div class="prorank-card prorank-card--feature">...</div>
```

### Phase 3: Gradual Component Migration
As you update features, migrate them to use the new components:
1. Replace hard-coded colors with variables
2. Use unified components with modifiers
3. Follow BEM naming for new styles

### Phase 4: Remove Old CSS
Once all components are migrated:
1. Remove the `@import './prorank-unified.css';` from `index.css`
2. Delete the old unified CSS file
3. Celebrate! 🎉

## Adding New Styles

### For Core Design Changes
Edit the appropriate file in `core/`:
- Colors, spacing → `_variables.css`
- Page structure → `_layout.css`
- Text styles → `_typography.css`

### For New Components
1. Create a new file in `components/` (e.g., `_tooltips.css`)
2. Follow the existing component structure
3. Import it in `index.css`
4. Document usage here

### For Feature-Specific Styles
1. Create a file in `features/` (e.g., `_seo-simulator.css`)
2. Import core components, don't duplicate
3. Only add layout and feature-specific overrides

## Benefits of This System

1. **Enforced Consistency**: Can't accidentally use wrong colors
2. **Faster Development**: Find styles quickly in organized files
3. **Team Scalability**: Multiple devs can work without conflicts
4. **Performance**: Can lazy-load feature CSS in the future
5. **Maintainability**: Clear structure makes updates easy

## Rules to Follow

1. **NEVER** create a new color - use variables
2. **ALWAYS** check if a component exists before creating new
3. **USE** BEM naming: `.prorank-component--modifier`
4. **KEEP** specificity low - max 3 levels deep
5. **DOCUMENT** any new components or patterns

## Testing Checklist

When making CSS changes:
- [ ] All colors use CSS variables
- [ ] No duplicate component definitions
- [ ] BEM naming convention followed
- [ ] Works on mobile (responsive)
- [ ] No console errors
- [ ] Visual regression test passed
- [ ] Build succeeds

## Support

For questions or issues with the design system:
1. Check this documentation
2. Look at existing component examples
3. Ask in the development channel
4. Create an issue if you find a bug

Remember: **Consistency is key to becoming the #1 SEO plugin!**
