/**
 * Settings validation schema for Sitemaps
 * Auto-generated from baseline.json
 * 
 * @module TechnicalSEO/Sitemaps
 * @since 0.1.0
 */

import { z } from 'zod';

/**
 * Settings schema for Sitemaps
 */
export const SitemapsSchema = z.object({
  enable_xml_sitemap: z.boolean().default(true),  // Enable XML sitemap generation
  posts_per_sitemap: z.number().int().min(1).max(50000).default(1000),  // Maximum number of URLs per sitemap
  include_post_types: z.array(z.string()).default(["post","page"]),  // Post types to include in sitemap
  include_taxonomies: z.array(z.string()).default(["category","post_tag"]),  // Taxonomies to include in sitemap
  exclude_empty_terms: z.boolean().default(true),  // Exclude empty taxonomy terms
  include_images: z.boolean().default(true),  // Include images in sitemap
  include_author_archives: z.boolean().default(false),  // Include author archive pages
  exclude_noindex: z.boolean().default(true),  // Exclude noindex pages from sitemap
  ping_search_engines: z.boolean().default(true),  // Automatically ping search engines on update
  sitemap_index_url: z.string().regex(/^[a-zA-Z0-9_-]+\.xml$/).default('sitemap_index.xml'),  // Sitemap index filename
});

/**
 * TypeScript type for Sitemaps settings
 */
export type SitemapsSettings = z.infer<typeof SitemapsSchema>;

/**
 * Default values for Sitemaps settings
 */
export const SitemapsDefaults: SitemapsSettings = SitemapsSchema.parse({});

/**
 * Validate Sitemaps settings
 */
export function validateSitemapsSettings(data: unknown): SitemapsSettings {
  return SitemapsSchema.parse(data);
}

/**
 * Safely validate Sitemaps settings (returns validation result)
 */
export function safeValidateSitemapsSettings(data: unknown) {
  return SitemapsSchema.safeParse(data);
}
