import { useEffect, useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Button, Card, Checkbox, Input, Toggle } from '../../../components/ui';
import LoadingState from '../../../components/LoadingState';
import { useNotification } from '../../../contexts/NotificationContext';

const SitemapSettings = () => {
  const [settings, setSettings] = useState({
    enable_xml_sitemap: true,
    posts_per_sitemap: 1000,
    include_post_types: ['post', 'page'],
    include_taxonomies: ['category', 'post_tag'],
    include_images: true,
    exclude_noindex: true,
    ping_search_engines: true,
  });

  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const { showNotification } = useNotification();

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/sitemaps',
      });

      if (response?.settings) {
        setSettings((prev) => ({ ...prev, ...response.settings }));
      }
    } catch (error) {
      showNotification(__('Failed to load sitemap settings', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      await apiFetch({
        path: '/prorank-seo/v1/settings/sitemaps',
        method: 'POST',
        data: { settings },
      });
      showNotification(__('Sitemap settings saved', 'prorank-seo'), 'success');
    } catch (error) {
      showNotification(__('Failed to save sitemap settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  const togglePostType = (postType) => {
    setSettings((prev) => {
      const next = new Set(prev.include_post_types || []);
      if (next.has(postType)) {
        next.delete(postType);
      } else {
        next.add(postType);
      }
      return { ...prev, include_post_types: Array.from(next) };
    });
  };

  const toggleTaxonomy = (taxonomy) => {
    setSettings((prev) => {
      const next = new Set(prev.include_taxonomies || []);
      if (next.has(taxonomy)) {
        next.delete(taxonomy);
      } else {
        next.add(taxonomy);
      }
      return { ...prev, include_taxonomies: Array.from(next) };
    });
  };

  if (loading) {
    return <LoadingState message={__('Loading sitemap settings…', 'prorank-seo')} />;
  }

  return (
    <div className="pr-space-y-6 pr-max-w-4xl">
      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('XML Sitemaps', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Control which content types appear in your sitemap.', 'prorank-seo')}
            </p>
          </div>

          <Toggle
            label={__('Enable XML sitemap', 'prorank-seo')}
            checked={Boolean(settings.enable_xml_sitemap)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, enable_xml_sitemap: value }))
            }
          />

          <Input
            type="number"
            label={__('Posts per sitemap', 'prorank-seo')}
            value={String(settings.posts_per_sitemap ?? 1000)}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, posts_per_sitemap: Number(event.target.value) }))
            }
          />

          <div className="pr-space-y-2">
            <p className="pr-text-sm pr-font-medium pr-text-gray-700">
              {__('Post types', 'prorank-seo')}
            </p>
            <Checkbox
              label={__('Posts', 'prorank-seo')}
              checked={settings.include_post_types?.includes('post')}
              onChange={() => togglePostType('post')}
            />
            <Checkbox
              label={__('Pages', 'prorank-seo')}
              checked={settings.include_post_types?.includes('page')}
              onChange={() => togglePostType('page')}
            />
          </div>

          <div className="pr-space-y-2">
            <p className="pr-text-sm pr-font-medium pr-text-gray-700">
              {__('Taxonomies', 'prorank-seo')}
            </p>
            <Checkbox
              label={__('Categories', 'prorank-seo')}
              checked={settings.include_taxonomies?.includes('category')}
              onChange={() => toggleTaxonomy('category')}
            />
            <Checkbox
              label={__('Tags', 'prorank-seo')}
              checked={settings.include_taxonomies?.includes('post_tag')}
              onChange={() => toggleTaxonomy('post_tag')}
            />
          </div>

          <Toggle
            label={__('Include images', 'prorank-seo')}
            checked={Boolean(settings.include_images)}
            onChange={(value) => setSettings((prev) => ({ ...prev, include_images: value }))}
          />

          <Toggle
            label={__('Exclude noindex content', 'prorank-seo')}
            checked={Boolean(settings.exclude_noindex)}
            onChange={(value) => setSettings((prev) => ({ ...prev, exclude_noindex: value }))}
          />

          <Toggle
            label={__('Ping search engines', 'prorank-seo')}
            checked={Boolean(settings.ping_search_engines)}
            onChange={(value) => setSettings((prev) => ({ ...prev, ping_search_engines: value }))}
          />
        </div>
      </Card>

      <div className="pr-flex pr-gap-3">
        <Button variant="primary" onClick={saveSettings} loading={saving} disabled={saving}>
          {saving ? __('Saving…', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>
        <Button variant="secondary" onClick={loadSettings} disabled={saving}>
          {__('Reload', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

export default SitemapSettings;
