import { useState, useEffect } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { useNotification } from '../../../contexts/NotificationContext';
import { Toggle, Textarea, Input, Button, Alert, Spinner, Card, Tabs } from '../../../components/ui';
import { Heading, Text } from '../../../components/common/Typography';

const RobotsIndexingSettings = () => {
  const ajaxUrl = window?.prorankSeo?.ajaxUrl || window?.ajaxurl || '';
  let adminAjaxPath = '';
  if (ajaxUrl) {
    try {
      adminAjaxPath = new URL(ajaxUrl, window.location.origin).pathname;
    } catch (error) {
      adminAjaxPath = '';
    }
  }
  const defaultRobotsRules = `User-agent: *
Disallow: /wp-admin/${adminAjaxPath ? `\nAllow: ${adminAjaxPath}` : ''}`;

  const [settings, setSettings] = useState({
    enable_robots_editor: false,
    robots_txt_rules: '',
    block_ai_bots: false,
    enable_indexnow: false,
    indexnow_api_key: '',
    enable_google_indexing: false,
    google_service_account: '',
    // Content Safeguard settings
    enable_noai_meta: false,
    enable_noimageai_meta: false,
    enable_rule_noindex: false,
    rule_min_words: 300,
    rule_max_age_days: 0,
    // Global Noindex settings
    noindex_posts: false,
    noindex_pages: false,
    noindex_attachments: true,
    noindex_author: false,
    noindex_date: true,
    noindex_category: false,
    noindex_tag: false,
    noindex_search: true,
    noindex_404: true,
    noindex_paginated: false,
  });
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [validating, setValidating] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);
  const [physicalRobotsExists, setPhysicalRobotsExists] = useState(false);

  const { showNotification } = useNotification();
  const isCore = true;

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/robots_indexing',
      });

      setSettings(response.settings || {
        enable_robots_editor: false,
        robots_txt_rules: '',
        block_ai_bots: false,
        enable_indexnow: false,
        indexnow_api_key: '',
        enable_google_indexing: false,
        google_service_account: '',
        enable_noai_meta: false,
        enable_noimageai_meta: false,
        enable_rule_noindex: false,
        rule_min_words: 300,
        rule_max_age_days: 0,
        noindex_posts: false,
        noindex_pages: false,
        noindex_attachments: true,
        noindex_author: false,
        noindex_date: true,
        noindex_category: false,
        noindex_tag: false,
        noindex_search: true,
        noindex_404: true,
        noindex_paginated: false,
      });
      setPhysicalRobotsExists((response.meta && response.meta.physical_robots_exists) || false);
    } catch (error) {
      showNotification(__('Failed to load settings', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  // Load settings on mount
  useEffect(() => {
    loadSettings();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, []);

  const handleSettingChange = (key, value) => {
    setSettings((prev) => ({
      ...prev,
      [key]: value,
    }));
    setHasChanges(true);
  };

  const handleSave = async () => {
    setSaving(true);
    try {
      await apiFetch({
        path: '/prorank-seo/v1/settings/robots_indexing',
        method: 'POST',
        data: settings,
      });

      showNotification(__('Settings saved successfully', 'prorank-seo'), 'success');
      setHasChanges(false);
    } catch (error) {
      showNotification(error.message || __('Failed to save settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  const validateApiKey = async (type) => {
    setValidating(true);

    const key = type === 'indexnow' ? settings.indexnow_api_key : settings.google_service_account;

    if (!key) {
      showNotification(__('Please enter a key to validate', 'prorank-seo'), 'warning');
      setValidating(false);
      return;
    }

    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/robots_indexing/validate-key',
        method: 'POST',
        data: { type, key },
      });

      showNotification(response.message, 'success');

      if (type === 'google' && response.client_email) {
        showNotification(
          sprintf(
            /* translators: %s: service account email */
            __('Service account: %s', 'prorank-seo'),
            response.client_email
          ),
          'info'
        );
      }
    } catch (error) {
      showNotification(error.message || __('Validation failed', 'prorank-seo'), 'error');
    } finally {
      setValidating(false);
    }
  };

  const tabs = [
    {
      name: 'robots',
      title: __('Robots.txt', 'prorank-seo'),
      content: (
        <div className="prorank-settings-content pr-transition-all pr-duration-300">
          {physicalRobotsExists && (
            <Alert variant="warning" className="pr-mb-4">
              <strong>{__('Physical robots.txt detected', 'prorank-seo')}</strong>
              <p>
                {__(
                  'A physical robots.txt file exists in your site root. The virtual editor will be disabled to prevent conflicts.',
                  'prorank-seo'
                )}
              </p>
            </Alert>
          )}

          <Card>
            
              <div>
                <Toggle
                  label={__('Enable Virtual Robots.txt Editor', 'prorank-seo')}
                  helpText={__(
                    'Create and manage robots.txt rules without a physical file',
                    'prorank-seo'
                  )}
                  checked={settings.enable_robots_editor && !physicalRobotsExists}
                  onChange={(checked) => handleSettingChange('enable_robots_editor', checked)}
                  disabled={physicalRobotsExists}
                />
              </div>

              {settings.enable_robots_editor && !physicalRobotsExists && (
                <div className="pr-mt-4">
                  <Textarea
                    label={__('Robots.txt Rules', 'prorank-seo')}
                    helpText={__(
                      'Enter your custom robots.txt rules. Sitemap URL will be added automatically.',
                      'prorank-seo'
                    )}
                    value={settings.robots_txt_rules}
                    onChange={(e) => handleSettingChange('robots_txt_rules', e.target.value)}
                    rows={10}
                    placeholder={defaultRobotsRules}
                  />
                </div>
              )}

              <div className="pr-mt-5">
                <Toggle
                  label={__('Block AI/ML Training Bots via Robots.txt', 'prorank-seo')}
                  helpText={__(
                    'Prevents AI crawlers (GPTBot, Claude-Web, Gemini-Bot, etc.) from accessing your site entirely by adding User-agent rules to robots.txt',
                    'prorank-seo'
                  )}
                  checked={settings.block_ai_bots}
                  onChange={(checked) => handleSettingChange('block_ai_bots', checked)}
                  disabled={!isCore}
                />
                {settings.block_ai_bots && (
                  <Alert variant="info" className="pr-mt-2.5">
                    <strong>{__('How this works:', 'prorank-seo')}</strong>
                    <br />
                    {__(
                      'This adds Disallow rules for 50+ known AI bots to your robots.txt, completely blocking them from crawling your site. This is the strongest form of AI protection.',
                      'prorank-seo'
                    )}
                    <br /><br />
                    <strong>{__('Blocked bots include:', 'prorank-seo')}</strong> GPTBot, Claude-Web, Gemini-Bot, Bard-Bot, PerplexityBot, MidJourney-Bot, DALL-E-Bot, and 40+ others.
                  </Alert>
                )}
              </div>
            
          </Card>
        </div>
      ),
    },
    {
      name: 'indexing',
      title: __('Indexing APIs', 'prorank-seo'),
      content: (
        <div className="prorank-settings-content pr-transition-all pr-duration-300">
          

          <Card>
            
              <h3>{__('IndexNow', 'prorank-seo')}</h3>
              <Text>
                {__('Submit URLs instantly to participating search engines.', 'prorank-seo')}
              </Text>

              <div className="pr-mt-4">
                <Toggle
                  label={__('Enable IndexNow', 'prorank-seo')}
                  helpText={__(
                    'Automatically submit new and updated content to IndexNow',
                    'prorank-seo'
                  )}
                  checked={settings.enable_indexnow || false}
                  onChange={(checked) => handleSettingChange('enable_indexnow', checked)}
                  disabled={false}
                />

                {settings.enable_indexnow && (
                  <>
                    <div className="pr-mt-4">
                      <Input
                        label={__('IndexNow API Key', 'prorank-seo')}
                        helpText={
                          <>
                            {__('Get your key from ', 'prorank-seo')}
                            <a href="https://www.bing.com/indexnow" target="_blank" rel="noopener noreferrer" className="pr-text-primary-600 hover:pr-text-primary-700 pr-transition-all pr-duration-300">
                              {__('IndexNow', 'prorank-seo')}
                            </a>
                          </>
                        }
                        value={settings.indexnow_api_key}
                        onChange={(e) => handleSettingChange('indexnow_api_key', e.target.value)}
                        disabled={false}
                      />
                    </div>

                    <Button
                      variant="secondary"
                      onClick={() => validateApiKey('indexnow')}
                      disabled={!isCore || validating}
                      loading={validating}
                      className="pr-mt-2"
                    >
                      {__('Validate Key', 'prorank-seo')}
                    </Button>
                  </>
                )}
              </div>
            
          </Card>

          <Card>

              <h3>{__('Google Indexing API', 'prorank-seo')}</h3>
              <Text>
                {__('Submit URLs directly to Google for faster indexing.', 'prorank-seo')}
              </Text>

              {/* 2025 Enhancement: Google API temporarily disabled until library is added */}
              <Alert variant="info" className="pr-mt-4">
                <strong>{__('Coming Soon', 'prorank-seo')}</strong>
                <p>
                  {__(
                    'Google Indexing API integration is currently being implemented. This feature requires the Google API client library which will be added in an upcoming release. In the meantime, please use IndexNow API for instant indexing across Bing, Yandex, Seznam, and Naver.',
                    'prorank-seo'
                  )}
                </p>
              </Alert>

              <div className="pr-mt-4">
                <Toggle
                  label={__('Enable Google Indexing API', 'prorank-seo')}
                  helpText={__('Coming soon - requires Google API client library', 'prorank-seo')}
                  checked={false}
                  onChange={() => {}}
                  disabled={true}
                />
              </div>
            
          </Card>
        </div>
      ),
    },
    {
      name: 'noindex',
      title: __('Global Noindex', 'prorank-seo'),
      content: (
        <div className="prorank-settings-content pr-transition-all pr-duration-300">
          <Card>
            
              <h3>{__('Post Types', 'prorank-seo')}</h3>
              <Text>
                {__(
                  'Set noindex for entire post types to prevent them from appearing in search results.',
                  'prorank-seo'
                )}
              </Text>

              <div className="pr-mt-5 pr-space-y-4">
                <Toggle
                  label={__('Noindex Posts', 'prorank-seo')}
                  helpText={__('Prevent all blog posts from being indexed', 'prorank-seo')}
                  checked={settings.noindex_posts || false}
                  onChange={(checked) => handleSettingChange('noindex_posts', checked)}
                />

                <Toggle
                  label={__('Noindex Pages', 'prorank-seo')}
                  helpText={__('Prevent all pages from being indexed', 'prorank-seo')}
                  checked={settings.noindex_pages || false}
                  onChange={(checked) => handleSettingChange('noindex_pages', checked)}
                />

                <Toggle
                  label={__('Noindex Media Attachments', 'prorank-seo')}
                  helpText={__(
                    'Prevent attachment pages from being indexed (recommended)',
                    'prorank-seo'
                  )}
                  checked={settings.noindex_attachments || false}
                  onChange={(checked) => handleSettingChange('noindex_attachments', checked)}
                />
              </div>
            
          </Card>

          <Card className="pr-mt-5">
            
              <h3>{__('Archives', 'prorank-seo')}</h3>
              <Text>
                {__(
                  'Control indexing of archive pages to prevent duplicate content issues.',
                  'prorank-seo'
                )}
              </Text>

              <div className="pr-mt-5 pr-space-y-4">
                <Toggle
                  label={__('Noindex Author Archives', 'prorank-seo')}
                  helpText={__('Prevent author archive pages from being indexed', 'prorank-seo')}
                  checked={settings.noindex_author || false}
                  onChange={(checked) => handleSettingChange('noindex_author', checked)}
                />

                <Toggle
                  label={__('Noindex Date Archives', 'prorank-seo')}
                  helpText={__('Prevent date-based archive pages from being indexed', 'prorank-seo')}
                  checked={settings.noindex_date || false}
                  onChange={(checked) => handleSettingChange('noindex_date', checked)}
                />

                <Toggle
                  label={__('Noindex Category Archives', 'prorank-seo')}
                  helpText={__('Prevent category archive pages from being indexed', 'prorank-seo')}
                  checked={settings.noindex_category || false}
                  onChange={(checked) => handleSettingChange('noindex_category', checked)}
                />

                <Toggle
                  label={__('Noindex Tag Archives', 'prorank-seo')}
                  helpText={__('Prevent tag archive pages from being indexed', 'prorank-seo')}
                  checked={settings.noindex_tag || false}
                  onChange={(checked) => handleSettingChange('noindex_tag', checked)}
                />
              </div>
            
          </Card>

          <Card className="pr-mt-5">
            
              <h3>{__('Special Pages', 'prorank-seo')}</h3>
              <p>{__('Control indexing of special WordPress pages.', 'prorank-seo')}</p>

              <div className="pr-mt-5 pr-space-y-4">
                <Toggle
                  label={__('Noindex Search Results', 'prorank-seo')}
                  helpText={__(
                    'Prevent internal search results pages from being indexed (recommended)',
                    'prorank-seo'
                  )}
                  checked={settings.noindex_search || false}
                  onChange={(checked) => handleSettingChange('noindex_search', checked)}
                />

                <Toggle
                  label={__('Noindex 404 Pages', 'prorank-seo')}
                  helpText={__('Prevent 404 error pages from being indexed', 'prorank-seo')}
                  checked={settings.noindex_404 || false}
                  onChange={(checked) => handleSettingChange('noindex_404', checked)}
                />

                <Toggle
                  label={__('Noindex Paginated Pages', 'prorank-seo')}
                  helpText={__(
                    'Prevent paginated pages (page 2, 3, etc.) from being indexed',
                    'prorank-seo'
                  )}
                  checked={settings.noindex_paginated || false}
                  onChange={(checked) => handleSettingChange('noindex_paginated', checked)}
                />
              </div>
            
          </Card>

          <Alert variant="warning" className="pr-mt-5">
            <strong>{__('Important:', 'prorank-seo')}</strong>{' '}
            {__(
              "These settings will prevent entire sections of your site from appearing in search results. Use with caution and only noindex content that you don't want search engines to find.",
              'prorank-seo'
            )}
          </Alert>
        </div>
      ),
    },
    {
      name: 'safeguard',
      title: __('Content Safeguard', 'prorank-seo'),
      content: (
        <div className="prorank-settings-content pr-transition-all pr-duration-300">
          <Card>
            
              <h3>{__('AI Content Protection Meta Tags', 'prorank-seo')}</h3>
              <Text>
                {__(
                  'Add meta tag directives that politely request AI systems not to use your content for training (respected by ethical AI companies).',
                  'prorank-seo'
                )}
              </Text>

              <div className="pr-mt-5 pr-space-y-4">
                <Toggle
                  label={__('Add "noai" meta tag to all pages', 'prorank-seo')}
                  helpText={__(
                    'Requests AI systems not to use your text content for training purposes',
                    'prorank-seo'
                  )}
                  checked={settings.enable_noai_meta || false}
                  onChange={(checked) => handleSettingChange('enable_noai_meta', checked)}
                />

                <Toggle
                  label={__('Add "noimageai" meta tag to all pages', 'prorank-seo')}
                  helpText={__(
                    'Requests AI systems not to use your images for training purposes',
                    'prorank-seo'
                  )}
                  checked={settings.enable_noimageai_meta || false}
                  onChange={(checked) => handleSettingChange('enable_noimageai_meta', checked)}
                />
              </div>

              <Alert variant="warning" className="pr-mt-5">
                <strong>{__('Important Distinction:', 'prorank-seo')}</strong>
                <br /><br />
                <strong>{__('Meta Tags (this section):', 'prorank-seo')}</strong> {__('Polite requests that ethical AI companies may respect. Content can still be crawled and viewed.', 'prorank-seo')}
                <br /><br />
                <strong>{__('Robots.txt Blocking (Robots.txt tab):', 'prorank-seo')}</strong> {__('Completely blocks AI bots from accessing your site. This is more effective but prevents all access.', 'prorank-seo')}
                <br /><br />
                {__('For maximum protection, enable both features.', 'prorank-seo')}
              </Alert>
            
          </Card>

          <Card className="pr-mt-5">
            
              <div>
                <Heading level={3}>
                  {__('Rule-based Noindex for Thin Content', 'prorank-seo')}
                </Heading>
                <Text>
                  {__(
                    'Automatically noindex pages with thin content to improve overall site quality.',
                    'prorank-seo'
                  )}
                </Text>

                <div className="pr-mt-5">
                  <Toggle
                    label={__('Enable Rule-based Noindex', 'prorank-seo')}
                    helpText={__(
                      'Automatically add noindex to pages that meet your criteria',
                      'prorank-seo'
                    )}
                    checked={settings.enable_rule_noindex || false}
                    onChange={(checked) => handleSettingChange('enable_rule_noindex', checked)}
                    disabled={false}
                  />
                </div>

                {settings.enable_rule_noindex && (
                  <>
                    <div className="pr-mt-5">
                      <Input
                        label={__('Minimum word count for indexing', 'prorank-seo')}
                        helpText={__(
                          'Pages with fewer words than this will be noindexed',
                          'prorank-seo'
                        )}
                        type="number"
                        value={settings.rule_min_words || 300}
                        onChange={(e) =>
                          handleSettingChange('rule_min_words', parseInt(e.target.value) || 300)
                        }
                        min="0"
                      />
                    </div>

                    <div className="pr-mt-5">
                      <Input
                        label={__('Maximum post age for indexing (days)', 'prorank-seo')}
                        helpText={__(
                          'Posts older than this will be noindexed (0 for no limit)',
                          'prorank-seo'
                        )}
                        type="number"
                        value={settings.rule_max_age_days || 0}
                        onChange={(e) =>
                          handleSettingChange('rule_max_age_days', parseInt(e.target.value) || 0)
                        }
                        min="0"
                      />
                    </div>
                  </>
                )}

                {!isCore && (
                  <Alert variant="warning" className="pr-mt-5">
                    {__(
                      'Automatically manage thin content with rule-based noindex settings.',
                      'prorank-seo'
                    )}
                  </Alert>
                )}
              </div>
            
          </Card>
        </div>
      ),
    },
    {
      name: 'help',
      title: __('Help', 'prorank-seo'),
      content: (
        <div className="prorank-settings-help pr-transition-all pr-duration-300">
          <Card>
            
              <h3>{__('About Robots & Indexing', 'prorank-seo')}</h3>
              <Text>
                {__(
                  'Control how search engines crawl and index your site with these powerful tools:',
                  'prorank-seo'
                )}
              </Text>

              <ul className="pr-ml-5 pr-list-disc pr-mt-2 pr-space-y-1">
                <li>{__('Virtual robots.txt editor for easy management', 'prorank-seo')}</li>
                <li>
                  {__('IndexNow for instant URL submission to Bing and Yandex', 'prorank-seo')}
                </li>
                <li>{__('Google Indexing API for priority indexing', 'prorank-seo')}</li>
                <li>{__('Content safeguard to protect from AI scraping', 'prorank-seo')}</li>
              </ul>

              <h3 className="pr-mt-4">{__('Setup Guides', 'prorank-seo')}</h3>

              <Text className="pr-mt-2">
                <strong>{__('IndexNow:', 'prorank-seo')}</strong>
                <br />
                {__('1. Generate an API key from the IndexNow website', 'prorank-seo')}
                <br />
                {__('2. Enter the key in the settings above', 'prorank-seo')}
                <br />
                {__('3. ProRank will create the verification file automatically', 'prorank-seo')}
              </Text>

              <Text className="pr-mt-2">
                <strong>{__('Google Indexing API:', 'prorank-seo')}</strong>
                <br />
                {__('1. Create a project in Google Cloud Console', 'prorank-seo')}
                <br />
                {__('2. Enable the Indexing API', 'prorank-seo')}
                <br />
                {__('3. Create a service account and download the JSON key', 'prorank-seo')}
                <br />
                {__('4. Add the service account email to Search Console', 'prorank-seo')}
                <br />
                {__('5. Paste the JSON key above', 'prorank-seo')}
              </Text>

              <div className="prorank-help-links pr-mt-4">
                <Button
                  variant="link"
                  href="https://prorank.io/docs/robots-indexing"
                >
                  {__('View Full Documentation →', 'prorank-seo')}
                </Button>
              </div>
            
          </Card>
        </div>
      ),
    },
  ];

  if (loading) {
    return (
      <div className="prorank-settings-loading pr-text-center pr-p-5 pr-transition-all pr-duration-300">
        <Spinner />
        <p className="pr-mt-2">{__('Loading settings…', 'prorank-seo')}</p>
      </div>
    );
  }

  return (
    <div className="prorank-robots-indexing-settings">
      <Tabs className="prorank-settings-tabs" activeClass="is-active" tabs={tabs}>
        {(tab) => <div className="pr-transition-all pr-duration-300">{tab.content}</div>}
      </Tabs>

      <div className="prorank-settings-footer pr-transition-all pr-duration-300">
        <Button
          variant="primary"
          onClick={handleSave}
          loading={saving}
          disabled={saving || !hasChanges || !isCore}
        >
          {saving ? __('Saving…', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>

        {hasChanges && (
          <Text className="prorank-unsaved-notice pr-ml-3">
            {__('You have unsaved changes', 'prorank-seo')}
          </Text>
        )}
      </div>
    </div>
  );
};

export default RobotsIndexingSettings;
