import { useCallback, useEffect, useState } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { TabPanel } from '@wordpress/components';
import { Button, Input, Select, Card, Checkbox } from '../../../components/ui';
import { useNotification } from '../../../contexts/NotificationContext';
import RedirectEditPanel from '../../../components/redirects/RedirectEditPanel';
import NotFoundMonitor from '../../../components/redirects/NotFoundMonitor';

const RedirectSettings = () => {
  const [redirects, setRedirects] = useState([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [filterType, setFilterType] = useState('all');
  const [currentPage, setCurrentPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const [selectedRedirects, setSelectedRedirects] = useState([]);
  const [editingRedirect, setEditingRedirect] = useState(null);
  const [isAddingNew, setIsAddingNew] = useState(false);
  const { showNotification } = useNotification();

  const perPage = 20;

  const loadRedirects = useCallback(async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: `/prorank-seo/v1/redirects?page=${currentPage}&per_page=${perPage}&search=${encodeURIComponent(
          searchTerm
        )}&type=${filterType}`,
        parse: false,
      });

      const data = await response.json();
      const pages = parseInt(response.headers.get('X-WP-TotalPages') || '1', 10);

      setRedirects(Array.isArray(data) ? data : []);
      setTotalPages(pages);
    } catch (error) {
      setRedirects([]);
      setTotalPages(1);
      showNotification(__('Failed to load redirects', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  }, [currentPage, filterType, searchTerm, showNotification]);

  useEffect(() => {
    loadRedirects();
  }, [loadRedirects]);

  const handleDelete = async (id) => {
    if (!confirm(__('Are you sure you want to delete this redirect?', 'prorank-seo'))) {
      return;
    }

    try {
      await apiFetch({
        path: `/prorank-seo/v1/redirects/${id}`,
        method: 'DELETE',
      });

      showNotification(__('Redirect deleted successfully', 'prorank-seo'), 'success');
      loadRedirects();
    } catch (error) {
      showNotification(__('Failed to delete redirect', 'prorank-seo'), 'error');
    }
  };

  const handleBulkAction = async (action) => {
    if (!action || selectedRedirects.length === 0) return;

    try {
      await apiFetch({
        path: '/prorank-seo/v1/redirects/bulk',
        method: 'POST',
        data: {
          action,
          ids: selectedRedirects,
        },
      });

      showNotification(__('Bulk action completed', 'prorank-seo'), 'success');
      setSelectedRedirects([]);
      loadRedirects();
    } catch (error) {
      showNotification(__('Failed to perform bulk action', 'prorank-seo'), 'error');
    }
  };

  const tabs = [
    { name: 'redirects', title: __('Redirects', 'prorank-seo') },
    { name: '404-monitor', title: __('404 Monitor', 'prorank-seo') },
  ];

  return (
    <div className="pr-space-y-6">
      <TabPanel
        className="prorank-tabs"
        activeClass="is-active"
        tabs={tabs}
      >
        {(tab) =>
          tab.name === 'redirects' ? (
            <Card>
              <div className="pr-space-y-4">
                <div className="pr-flex pr-flex-col md:pr-flex-row md:pr-items-center md:pr-justify-between pr-gap-3">
                  <div>
                    <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
                      {__('Redirect Manager', 'prorank-seo')}
                    </h3>
                    <p className="pr-text-sm pr-text-gray-600">
                      {__('Create and manage 301/302 redirects to preserve SEO value.', 'prorank-seo')}
                    </p>
                  </div>
                  <Button variant="primary" onClick={() => setIsAddingNew(true)}>
                    {__('Add New Redirect', 'prorank-seo')}
                  </Button>
                </div>

                <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-3 pr-gap-3">
                  <Input
                    label={__('Search redirects', 'prorank-seo')}
                    value={searchTerm}
                    onChange={(event) => setSearchTerm(event.target.value)}
                    placeholder={__('Search by URL…', 'prorank-seo')}
                  />

                  <Select
                    label={__('Filter by type', 'prorank-seo')}
                    value={filterType}
                    onChange={(event) => setFilterType(event.target.value)}
                    options={[
                      { value: 'all', label: __('All Types', 'prorank-seo') },
                      { value: '301', label: __('301 Permanent', 'prorank-seo') },
                      { value: '302', label: __('302 Temporary', 'prorank-seo') },
                      { value: '307', label: __('307 Temporary', 'prorank-seo') },
                      { value: '308', label: __('308 Permanent', 'prorank-seo') },
                    ]}
                  />

                  <Select
                    label={__('Bulk action', 'prorank-seo')}
                    value=""
                    onChange={(event) => handleBulkAction(event.target.value)}
                    options={[
                      { value: '', label: __('Select action', 'prorank-seo') },
                      { value: 'activate', label: __('Activate', 'prorank-seo') },
                      { value: 'deactivate', label: __('Deactivate', 'prorank-seo') },
                      { value: 'delete', label: __('Delete', 'prorank-seo') },
                    ]}
                  />
                </div>

                {loading ? (
                  <p className="pr-text-sm pr-text-gray-500">{__('Loading redirects…', 'prorank-seo')}</p>
                ) : redirects.length === 0 ? (
                  <p className="pr-text-sm pr-text-gray-500">{__('No redirects found.', 'prorank-seo')}</p>
                ) : (
                  <div className="pr-overflow-x-auto">
                    <table className="pr-min-w-full pr-text-sm">
                      <thead>
                        <tr className="pr-text-left pr-text-gray-500">
                          <th className="pr-px-2 pr-py-2">
                            <Checkbox
                              checked={selectedRedirects.length === redirects.length}
                              onChange={(value) => {
                                const isChecked = typeof value === 'boolean' ? value : value?.target?.checked;
                                setSelectedRedirects(
                                  isChecked ? redirects.map((r) => r.id) : []
                                );
                              }}
                            />
                          </th>
                          <th className="pr-px-2 pr-py-2">{__('Source', 'prorank-seo')}</th>
                          <th className="pr-px-2 pr-py-2">{__('Target', 'prorank-seo')}</th>
                          <th className="pr-px-2 pr-py-2">{__('Type', 'prorank-seo')}</th>
                          <th className="pr-px-2 pr-py-2">{__('Status', 'prorank-seo')}</th>
                          <th className="pr-px-2 pr-py-2">{__('Actions', 'prorank-seo')}</th>
                        </tr>
                      </thead>
                      <tbody>
                        {redirects.map((redirect) => (
                          <tr key={redirect.id} className="pr-border-t">
                            <td className="pr-px-2 pr-py-2">
                              <Checkbox
                                checked={selectedRedirects.includes(redirect.id)}
                                onChange={(value) => {
                                  const isChecked = typeof value === 'boolean' ? value : value?.target?.checked;
                                  setSelectedRedirects((prev) =>
                                    isChecked
                                      ? [...prev, redirect.id]
                                      : prev.filter((id) => id !== redirect.id)
                                  );
                                }}
                              />
                            </td>
                            <td className="pr-px-2 pr-py-2 pr-text-gray-800">{redirect.source_url}</td>
                            <td className="pr-px-2 pr-py-2 pr-text-gray-800">{redirect.target_url}</td>
                            <td className="pr-px-2 pr-py-2 pr-text-gray-600">{redirect.type}</td>
                            <td className="pr-px-2 pr-py-2 pr-text-gray-600">{redirect.status}</td>
                            <td className="pr-px-2 pr-py-2 pr-space-x-2">
                              <Button
                                variant="secondary"
                                size="sm"
                                onClick={() => setEditingRedirect(redirect)}
                              >
                                {__('Edit', 'prorank-seo')}
                              </Button>
                              <Button variant="danger" size="sm" onClick={() => handleDelete(redirect.id)}>
                                {__('Delete', 'prorank-seo')}
                              </Button>
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                )}

                {totalPages > 1 && (
                  <div className="pr-flex pr-items-center pr-justify-between">
                    <Button
                      variant="secondary"
                      onClick={() => setCurrentPage((prev) => Math.max(prev - 1, 1))}
                      disabled={currentPage === 1}
                    >
                      {__('Previous', 'prorank-seo')}
                    </Button>
                    <span className="pr-text-sm pr-text-gray-500">
                      {sprintf(__('Page %1$d of %2$d', 'prorank-seo'), currentPage, totalPages)}
                    </span>
                    <Button
                      variant="secondary"
                      onClick={() => setCurrentPage((prev) => Math.min(prev + 1, totalPages))}
                      disabled={currentPage === totalPages}
                    >
                      {__('Next', 'prorank-seo')}
                    </Button>
                  </div>
                )}
              </div>
            </Card>
          ) : (
            <NotFoundMonitor onCreateRedirect={() => loadRedirects()} />
          )
        }
      </TabPanel>

      {(editingRedirect || isAddingNew) && (
        <RedirectEditPanel
          redirect={editingRedirect}
          onClose={() => {
            setEditingRedirect(null);
            setIsAddingNew(false);
            loadRedirects();
          }}
        />
      )}
    </div>
  );
};

export default RedirectSettings;
