import { useEffect, useState } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Button, Card, Input, RangeInput, Select, Textarea, Toggle } from '../../../components/ui';
import LoadingState from '../../../components/LoadingState';
import { useNotification } from '../../../contexts/NotificationContext';
import setupApiFetch from '../../../utils/setupApiFetch';
import * as React from 'react';

interface ImageSettings {
  webp_enabled: boolean;
  webp_quality: number;
  avif_enabled: boolean;
  avif_quality: number;
  jxl_enabled: boolean;
  jxl_quality: number;
  png_quality: number;
  compression_type: 'lossless' | 'glossy' | 'lossy';
  optimize_on_upload: boolean;
  backup_originals: boolean;
  lazyload_images: boolean;
  lazyload_iframes: boolean;
  lazyload_videos: boolean;
  lazyload_threshold: number;
  delivery_method: 'picture' | 'accept_header' | 'htaccess' | 'cdn' | 'none';
  exclude_paths: string;
}

interface ImageStats {
  total_images?: number;
  optimized_images?: number;
  saved_bytes?: number;
  optimization_percentage?: number;
  backup_size?: number;
  backup_size_formatted?: string;
  total?: number;
  optimized?: number;
  bytes_saved?: number;
}

interface BulkProgress {
  current: number;
  total: number;
  current_image: string;
  bytes_saved: number;
  status: 'idle' | 'running' | 'completed' | 'stopped' | 'error' | 'stale';
  error_message?: string;
  log?: Array<{
    image: string;
    status: string;
    bytes_saved?: number;
    ts?: number;
  }>;
  percent?: number;
  active_index?: number | null;
  summary?: {
    processed: number;
    saved_bytes: number;
    failed?: number;
    skipped?: number;
  };
}

interface ServerSupport {
  gd?: boolean;
  imagick?: boolean;
  webp?: boolean;
  avif?: boolean;
  jxl?: boolean;
  jpegli?: boolean;
  htaccess?: boolean;
  details?: {
    gd_version?: string;
    imagick_version?: string;
  };
  server?: {
    software?: string;
    php_version?: string;
    memory_limit?: string;
    max_execution_time?: string;
    upload_max_filesize?: string;
  };
}

const DEFAULT_SETTINGS: ImageSettings = {
  webp_enabled: true,
  webp_quality: 85,
  avif_enabled: false,
  avif_quality: 80,
  jxl_enabled: false,
  jxl_quality: 80,
  png_quality: 80,
  compression_type: 'glossy',
  optimize_on_upload: true,
  backup_originals: true,
  lazyload_images: true,
  lazyload_iframes: false,
  lazyload_videos: false,
  lazyload_threshold: 200,
  delivery_method: 'picture',
  exclude_paths: '',
};

const ImageOptimizationSettings: React.FC = () => {
  const [settings, setSettings] = useState<ImageSettings>(DEFAULT_SETTINGS);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [serverSupport, setServerSupport] = useState<ServerSupport | null>(null);
  const [supportLoading, setSupportLoading] = useState(false);
  const [stats, setStats] = useState<ImageStats>({});
  const [bulkProcessing, setBulkProcessing] = useState(false);
  const [progress, setProgress] = useState<BulkProgress>({
    current: 0,
    total: 0,
    current_image: '',
    bytes_saved: 0,
    status: 'idle',
    log: [],
  });
  const [startTime, setStartTime] = useState<number | null>(null);
  const [restoringBackups, setRestoringBackups] = useState(false);
  const [deletingBackups, setDeletingBackups] = useState(false);
  const { showNotification } = useNotification();

  const restData =
    (window as any).prorankSeo ||
    (window as any).prorankSeoAdmin ||
    (window as any).proRankSEO ||
    (window as any).proRankSeoAdmin ||
    {};
  const restNonce =
    restData.restNonce ||
    restData.nonce ||
    (window as any).wpApiSettings?.nonce ||
    '';
  const restRoot =
    restData.restUrl ||
    (window as any).wpApiSettings?.root ||
    '/wp-json/';

  const buildRestUrl = (path: string) => {
    if (path.startsWith('http')) {
      return path;
    }
    const base = restRoot.endsWith('/') ? restRoot : `${restRoot}/`;
    const normalizedPath = path.startsWith('/') ? path.slice(1) : path;
    return `${base}${normalizedPath}`;
  };

  const apiRequest = <T,>(options: {
    path: string;
    method?: string;
    data?: any;
    headers?: Record<string, string>;
  }) => {
    const { path, ...rest } = options;
    if (path.startsWith('http')) {
      return apiFetch<T>({
        ...rest,
        url: buildRestUrl(path),
        credentials: 'same-origin',
        headers: {
          ...(rest.headers || {}),
          ...(restNonce ? { 'X-WP-Nonce': restNonce } : {}),
        },
      });
    }
    return apiFetch<T>({
      ...rest,
      path,
      credentials: 'same-origin',
      headers: {
        ...(rest.headers || {}),
        ...(restNonce ? { 'X-WP-Nonce': restNonce } : {}),
      },
    });
  };

  const normalizeDeliveryMethod = (value?: string): ImageSettings['delivery_method'] => {
    if (value === 'accept') {
      return 'accept_header';
    }
    if (value === 'rewrite') {
      return 'htaccess';
    }
    if (value === 'picture' || value === 'accept_header' || value === 'htaccess' || value === 'cdn' || value === 'none') {
      return value;
    }
    return 'picture';
  };

  // Format bytes to human readable
  const formatBytes = (bytes: number): string => {
    if (bytes >= 1048576) {
      return sprintf(__('%s MB', 'prorank-seo'), (bytes / 1048576).toFixed(2));
    }
    return sprintf(__('%s KB', 'prorank-seo'), (bytes / 1024).toFixed(2));
  };

  // Calculate estimated time remaining
  const getTimeRemaining = (): string => {
    if (!startTime || progress.current === 0 || progress.total === 0) return '';
    const elapsed = (Date.now() - startTime) / 1000;
    const rate = progress.current / elapsed;
    const remaining = (progress.total - progress.current) / rate;
    if (remaining < 60) return sprintf(__('%d seconds', 'prorank-seo'), Math.round(remaining));
    if (remaining < 3600) return sprintf(__('%d minutes', 'prorank-seo'), Math.round(remaining / 60));
    return sprintf(__('%d hours', 'prorank-seo'), Math.round(remaining / 3600));
  };

  const getActiveIndex = (): number => {
    if (typeof progress.active_index === 'number') {
      return progress.active_index;
    }
    if (progress.current_image) {
      return (progress.current || 0) + 1;
    }
    return progress.current || 0;
  };

  const getOverallPercent = (): number => {
    if (typeof progress.percent === 'number') {
      return progress.percent;
    }
    if (progress.total > 0) {
      return Math.round((progress.current / progress.total) * 100);
    }
    return 0;
  };

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiRequest<{
        settings?: Partial<ImageSettings>;
      }>({
        path: '/prorank-seo/v1/performance/image-optimization/settings',
      });

      if (response?.settings) {
        const nextSettings = {
          ...response.settings,
          delivery_method: normalizeDeliveryMethod(response.settings.delivery_method),
        };
        setSettings((prev) => ({ ...prev, ...nextSettings }));
      }
    } catch (error) {
      showNotification(__('Failed to load image settings', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  const loadStats = async () => {
    try {
      const response = await apiRequest<{ stats?: ImageStats; data?: ImageStats | { stats?: ImageStats } }>({
        path: '/prorank-seo/v1/image-optimization/stats',
      });
      const statsPayload =
        response?.stats ||
        (response?.data && 'stats' in response.data ? response.data.stats : response?.data) ||
        null;
      if (statsPayload && typeof statsPayload === 'object') {
        setStats(statsPayload as ImageStats);
      }
    } catch (error) {
      console.warn('Failed to load image stats:', error);
    }
  };

  const loadSupport = async () => {
    setSupportLoading(true);
    try {
      const response = await apiRequest<{ support?: ServerSupport }>({
        path: '/prorank-seo/v1/image-optimization/check-support',
      });
      setServerSupport(response?.support || null);
    } catch (error) {
      setServerSupport(null);
      showNotification(__('Failed to load server capabilities', 'prorank-seo'), 'warning');
    } finally {
      setSupportLoading(false);
    }
  };

  const saveSettings = async (
    nextSettings?: Partial<ImageSettings>,
    options: { silent?: boolean } = {}
  ) => {
    setSaving(true);
    const isEventPayload =
      nextSettings && typeof nextSettings === 'object' && 'preventDefault' in nextSettings;
    const safeSettings = isEventPayload ? undefined : nextSettings;
    const mergedSettings = safeSettings ? { ...settings, ...safeSettings } : settings;
    const normalizedSettings = {
      ...mergedSettings,
      delivery_method: normalizeDeliveryMethod(mergedSettings.delivery_method),
    };
    try {
      await apiRequest({
        path: '/prorank-seo/v1/performance/image-optimization/settings',
        method: 'POST',
        data: {
          settings: normalizedSettings,
        },
      });
      setSettings(normalizedSettings);
      if (!options.silent) {
        showNotification(__('Image settings saved', 'prorank-seo'), 'success');
      }
    } catch (error: any) {
      const message =
        error?.message ||
        error?.data?.message ||
        __('Failed to save image settings', 'prorank-seo');
      showNotification(message, 'error');
    } finally {
      setSaving(false);
    }
  };

  const startBulkOptimization = async () => {
    setProgress({
      current: 0,
      total: 0,
      current_image: '',
      bytes_saved: 0,
      status: 'running',
      log: [],
    });
    setStartTime(Date.now());
    setBulkProcessing(true);

    try {
      const response = await apiRequest<{ message?: string; total?: number; scheduled?: number; complete?: boolean }>({
        path: '/prorank-seo/v1/image-optimization/bulk-convert',
        method: 'POST',
      });

      if (response?.total) {
        setProgress((prev) => ({ ...prev, total: response.total! }));
      }

      showNotification(
        response?.message || __('Bulk optimization started (local processing)', 'prorank-seo'),
        'info'
      );
    } catch (error: any) {
      setBulkProcessing(false);
      setStartTime(null);
      setProgress((prev) => ({ ...prev, status: 'error' }));
      showNotification(error?.message || __('Failed to start bulk optimization', 'prorank-seo'), 'error');
    }
  };

  const stopBulkOptimization = async () => {
    try {
      await apiRequest({
        path: '/prorank-seo/v1/image-optimization/stop',
        method: 'POST',
      });
      setBulkProcessing(false);
      setStartTime(null);
      setProgress((prev) => ({ ...prev, status: 'stopped' }));
      showNotification(__('Optimization stopped', 'prorank-seo'), 'info');
      await loadStats();
    } catch (error: any) {
      showNotification(error?.message || __('Failed to stop optimization', 'prorank-seo'), 'error');
    }
  };

  const restoreBackups = async () => {
    if (!confirm(__('Are you sure you want to restore all images from backups? This will replace optimized images with originals.', 'prorank-seo'))) {
      return;
    }
    setRestoringBackups(true);
    try {
      const response = await apiRequest<{ restored?: number; failed?: number; message?: string }>({
        path: '/prorank-seo/v1/image-optimization/restore-backups',
        method: 'POST',
      });
      showNotification(
        response?.message || sprintf(__('Restored %d images from backups', 'prorank-seo'), response?.restored ?? 0),
        'success'
      );
      await loadStats();
    } catch (error: any) {
      showNotification(error?.message || __('Failed to restore backups', 'prorank-seo'), 'error');
    } finally {
      setRestoringBackups(false);
    }
  };

  const deleteBackups = async () => {
    if (!confirm(__('Are you sure you want to delete all backup files? This cannot be undone.', 'prorank-seo'))) {
      return;
    }
    setDeletingBackups(true);
    try {
      await apiRequest({
        path: '/prorank-seo/v1/image-optimization/delete-backups',
        method: 'DELETE',
      });
      showNotification(__('All backup files deleted', 'prorank-seo'), 'success');
      await loadStats();
    } catch (error: any) {
      showNotification(error?.message || __('Failed to delete backups', 'prorank-seo'), 'error');
    } finally {
      setDeletingBackups(false);
    }
  };

  useEffect(() => {
    setupApiFetch();
    loadSettings();
    loadStats();
    loadSupport();
  }, []);

  useEffect(() => {
    if (loading) return;
    if (serverSupport?.avif !== false) return;
    if (!settings.avif_enabled) return;

    setSettings((prev) => ({ ...prev, avif_enabled: false }));
    saveSettings({ avif_enabled: false }, { silent: true });
  }, [loading, serverSupport?.avif, settings.avif_enabled]);

  // Poll for progress when bulk processing
  useEffect(() => {
    if (!bulkProcessing) return;

    const pollProgress = async () => {
      try {
        const response = await apiRequest<{ progress: BulkProgress }>({
          path: '/prorank-seo/v1/image-optimization/progress',
        });
        if (response?.progress) {
          setProgress(response.progress);

          if (response.progress.status === 'completed') {
            setBulkProcessing(false);
            setStartTime(null);
            showNotification(
              sprintf(
                __('Optimization complete! %d images processed, %s saved.', 'prorank-seo'),
                response.progress.current,
                formatBytes(response.progress.bytes_saved)
              ),
              'success'
            );
            await loadStats();
          } else if (response.progress.status === 'error') {
            setBulkProcessing(false);
            setStartTime(null);
            showNotification(
              response.progress.error_message || __('An error occurred during optimization', 'prorank-seo'),
              'error'
            );
          }
        }
      } catch (error) {
        console.error('Failed to poll progress:', error);
      }
    };

    const interval = setInterval(pollProgress, 1500);
    pollProgress();

    return () => clearInterval(interval);
  }, [bulkProcessing]);

  // Refresh stats when bulk finishes
  useEffect(() => {
    if (progress.status === 'completed') {
      loadStats();
    }
  }, [progress.status]);

  // Feature support badge - three states
  const FeatureBadge: React.FC<{ feature: 'jxl' | 'jpegli' | 'avif' | 'webp' }> = ({ feature }) => {
    const hasLocal = serverSupport?.[feature] === true;

    if (hasLocal) {
      return (
        <span className="pr-inline-flex pr-items-center pr-px-2 pr-py-0.5 pr-rounded pr-text-xs pr-font-medium pr-bg-green-100 pr-text-green-800">
          {__('Local', 'prorank-seo')}
        </span>
      );
    }
    return (
      <span className="pr-inline-flex pr-items-center pr-px-2 pr-py-0.5 pr-rounded pr-text-xs pr-font-medium pr-bg-yellow-100 pr-text-yellow-800">
        {__('Not available', 'prorank-seo')}
      </span>
    );
  };

  const parseSizeToMb = (value?: string): number => {
    if (!value) return 0;
    const normalized = String(value).trim().toLowerCase();
    const match = normalized.match(/^(\d+(?:\.\d+)?)([a-z]*)$/);
    if (!match) return 0;
    const num = parseFloat(match[1]);
    const unit = match[2];
    if (unit === 'g' || unit === 'gb') return num * 1024;
    if (unit === 'k' || unit === 'kb') return num / 1024;
    if (unit === 'm' || unit === 'mb' || unit === '') return num;
    return 0;
  };

  const recommendedBatchSize = () => {
    const memoryMb = parseSizeToMb(serverSupport?.server?.memory_limit);
    const maxExec = parseInt(serverSupport?.server?.max_execution_time || '0', 10);
    if (!memoryMb) return __('Automatic', 'prorank-seo');
    let batch = 50;
    if (memoryMb >= 512) batch = 300;
    else if (memoryMb >= 256) batch = 200;
    else if (memoryMb >= 128) batch = 100;
    if (maxExec > 0 && maxExec < 30) batch = Math.min(batch, 50);
    else if (maxExec > 0 && maxExec < 60) batch = Math.min(batch, 100);
    return String(batch);
  };

  const supportLabel = (value?: boolean) => (value ? __('Yes', 'prorank-seo') : __('No', 'prorank-seo'));

  if (loading) {
    return <LoadingState message={__('Loading image settings...', 'prorank-seo')} />;
  }

  // Get normalized stats
  const rawTotalImages = stats.total_images ?? stats.total ?? 0;
  const rawOptimizedImages = stats.optimized_images ?? stats.optimized ?? 0;
  const rawSavedBytes = stats.saved_bytes ?? stats.bytes_saved ?? 0;
  const totalImages = rawTotalImages > 0 ? rawTotalImages : progress.total || 0;
  const optimizedImages = rawOptimizedImages > 0 ? rawOptimizedImages : progress.current || 0;
  const savedBytes = rawSavedBytes > 0 ? rawSavedBytes : progress.bytes_saved || 0;
  const optimizationPercentage =
    stats.optimization_percentage ??
    (totalImages > 0 ? Math.round((optimizedImages / totalImages) * 100) : 0);
  const backupSize = stats.backup_size ?? 0;

  return (
    <div className="prorank-performance-container pr-space-y-6 pr-max-w-4xl">
      {/* Statistics Dashboard */}
      <Card variant="performance">
        <div className="pr-space-y-4">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Image Optimization Statistics', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Overview of your media library optimization status.', 'prorank-seo')}
            </p>
          </div>

          <div className="pr-grid pr-grid-cols-2 sm:pr-grid-cols-4 pr-gap-4">
            <div className="pr-text-center pr-p-4 pr-bg-gray-50 pr-rounded-lg">
              <div className="pr-text-2xl pr-font-bold pr-text-gray-900">{totalImages}</div>
              <div className="pr-text-sm pr-text-gray-600">{__('Total Images', 'prorank-seo')}</div>
            </div>
            <div className="pr-text-center pr-p-4 pr-bg-green-50 pr-rounded-lg">
              <div className="pr-text-2xl pr-font-bold pr-text-green-600">{optimizedImages}</div>
              <div className="pr-text-sm pr-text-gray-600">{__('Optimized', 'prorank-seo')}</div>
            </div>
            <div className="pr-text-center pr-p-4 pr-bg-blue-50 pr-rounded-lg">
              <div className="pr-text-2xl pr-font-bold pr-text-blue-600">{formatBytes(savedBytes)}</div>
              <div className="pr-text-sm pr-text-gray-600">{__('Space Saved', 'prorank-seo')}</div>
            </div>
            <div className="pr-text-center pr-p-4 pr-bg-orange-50 pr-rounded-lg">
              <div className="pr-text-2xl pr-font-bold pr-text-orange-600">{optimizationPercentage}%</div>
              <div className="pr-text-sm pr-text-gray-600">{__('Progress', 'prorank-seo')}</div>
            </div>
          </div>

          {/* Bulk Optimization */}
          {!bulkProcessing ? (
            <div className="pr-pt-4 pr-border-t pr-border-gray-200">
              <div className="pr-flex pr-flex-wrap pr-gap-3">
                <Button
                  variant="primary"
                  onClick={startBulkOptimization}
                  disabled={totalImages > 0 && totalImages === optimizedImages}
                >
                  {__('Optimize All Images', 'prorank-seo')}
                </Button>
                <Button variant="secondary" onClick={() => loadStats()}>
                  {__('Refresh Stats', 'prorank-seo')}
                </Button>
              </div>
              {totalImages === optimizedImages && totalImages > 0 && (
                <p className="pr-mt-2 pr-text-sm pr-text-green-600">
                  {__('All images are already optimized!', 'prorank-seo')}
                </p>
              )}
              <p className="pr-mt-2 pr-text-xs pr-text-gray-500">
                {__('Bulk optimization uses local processing.', 'prorank-seo')}
              </p>
            </div>
          ) : (
            <div className="pr-pt-4 pr-border-t pr-border-gray-200 pr-space-y-4">
              {/* Progress Bar */}
              <div>
                <div className="pr-flex pr-justify-between pr-mb-2">
                  <span className="pr-font-medium">
                    {progress.total > 0
                      ? sprintf(__('Processing %d of %d images', 'prorank-seo'), getActiveIndex(), progress.total)
                      : __('Initializing...', 'prorank-seo')}
                  </span>
                  <span className="pr-text-gray-600">{getOverallPercent()}%</span>
                </div>
                <div className="pr-w-full pr-bg-gray-200 pr-rounded-full pr-h-3">
                  <div
                    className="pr-bg-blue-600 pr-h-3 pr-rounded-full pr-transition-all pr-duration-300"
                    style={{ width: `${getOverallPercent()}%` }}
                  />
                </div>
              </div>

              {/* Current Image */}
              {progress.current_image && (
                <div className="pr-p-3 pr-bg-gray-50 pr-rounded-lg">
                  <div className="pr-text-xs pr-text-gray-500 pr-mb-1">{__('Currently processing:', 'prorank-seo')}</div>
                  <div className="pr-text-sm pr-font-mono pr-break-all">{progress.current_image}</div>
                </div>
              )}

              {/* Stats Grid */}
              <div className="pr-grid pr-grid-cols-3 pr-gap-4">
                <div className="pr-text-center pr-p-3 pr-bg-green-50 pr-rounded-lg">
                  <div className="pr-text-xl pr-font-bold pr-text-green-600">{formatBytes(progress.bytes_saved)}</div>
                  <div className="pr-text-xs pr-text-gray-600">{__('Saved', 'prorank-seo')}</div>
                </div>
                <div className="pr-text-center pr-p-3 pr-bg-blue-50 pr-rounded-lg">
                  <div className="pr-text-xl pr-font-bold pr-text-blue-600">{progress.current}</div>
                  <div className="pr-text-xs pr-text-gray-600">{__('Done', 'prorank-seo')}</div>
                </div>
                <div className="pr-text-center pr-p-3 pr-bg-yellow-50 pr-rounded-lg">
                  <div className="pr-text-xl pr-font-bold pr-text-yellow-600">{getTimeRemaining() || '...'}</div>
                  <div className="pr-text-xs pr-text-gray-600">{__('Remaining', 'prorank-seo')}</div>
                </div>
              </div>

              {/* Stop Button */}
              <Button variant="secondary" onClick={stopBulkOptimization} className="pr-bg-red-50 pr-text-red-600 pr-border-red-200">
                {__('Stop Optimization', 'prorank-seo')}
              </Button>

              <p className="pr-text-xs pr-text-gray-500">
                {__('You can close this window - optimization will continue in the background.', 'prorank-seo')}
              </p>
            </div>
          )}
        </div>
      </Card>

      {/* Backup Management */}
      {backupSize > 0 && (
        <Card variant="performance">
          <div className="pr-space-y-4">
            <div>
              <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
                {__('Backup Management', 'prorank-seo')}
              </h3>
              <p className="pr-text-sm pr-text-gray-600">
                {sprintf(__('Backup storage used: %s', 'prorank-seo'), stats.backup_size_formatted || formatBytes(backupSize))}
              </p>
            </div>
            <div className="pr-flex pr-flex-wrap pr-gap-3">
              <Button
                variant="secondary"
                onClick={restoreBackups}
                disabled={restoringBackups}
                loading={restoringBackups}
              >
                {__('Restore All Backups', 'prorank-seo')}
              </Button>
              <Button
                variant="secondary"
                onClick={deleteBackups}
                disabled={deletingBackups}
                loading={deletingBackups}
                className="pr-text-red-600 pr-border-red-200 hover:pr-bg-red-50"
              >
                {__('Delete All Backups', 'prorank-seo')}
              </Button>
            </div>
          </div>
        </Card>
      )}

      {/* Server Capabilities */}
      <Card variant="performance">
        <div className="pr-space-y-4">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Server Capabilities', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Detected from your hosting environment. Determines what can run locally.', 'prorank-seo')}
            </p>
          </div>

          {supportLoading && (
            <p className="pr-text-sm pr-text-gray-500">{__('Loading server details...', 'prorank-seo')}</p>
          )}

          {!supportLoading && serverSupport && (
            <div className="pr-grid pr-gap-4 pr-text-sm pr-text-gray-700">
              <div className="pr-grid pr-grid-cols-2 pr-gap-3">
                <div className="pr-flex pr-items-center pr-gap-2">
                  <strong>{__('GD Library', 'prorank-seo')}:</strong> {supportLabel(serverSupport.gd)}
                </div>
                <div className="pr-flex pr-items-center pr-gap-2">
                  <strong>{__('ImageMagick', 'prorank-seo')}:</strong> {supportLabel(serverSupport.imagick)}
                </div>
                <div className="pr-flex pr-items-center pr-gap-2">
                  <strong>{__('WebP', 'prorank-seo')}:</strong> {supportLabel(serverSupport.webp)} <FeatureBadge feature="webp" />
                </div>
                <div className="pr-flex pr-items-center pr-gap-2">
                  <strong>{__('AVIF', 'prorank-seo')}:</strong> {supportLabel(serverSupport.avif)} <FeatureBadge feature="avif" />
                </div>
                <div className="pr-flex pr-items-center pr-gap-2">
                  <strong>{__('JPEG XL', 'prorank-seo')}:</strong> {supportLabel(serverSupport.jxl)} <FeatureBadge feature="jxl" />
                </div>
                <div className="pr-flex pr-items-center pr-gap-2">
                  <strong>{__('Jpegli', 'prorank-seo')}:</strong> {supportLabel(serverSupport.jpegli)} <FeatureBadge feature="jpegli" />
                </div>
              </div>

              <div className="pr-border-t pr-border-gray-200 pr-pt-3 pr-grid pr-grid-cols-2 pr-gap-3">
                <div>
                  <strong>{__('PHP', 'prorank-seo')}:</strong> {serverSupport.server?.php_version || 'N/A'}
                </div>
                <div>
                  <strong>{__('Memory limit', 'prorank-seo')}:</strong> {serverSupport.server?.memory_limit || 'N/A'}
                </div>
                <div>
                  <strong>{__('Max execution', 'prorank-seo')}:</strong> {serverSupport.server?.max_execution_time || 'N/A'}s
                </div>
                <div>
                  <strong>{__('Upload limit', 'prorank-seo')}:</strong> {serverSupport.server?.upload_max_filesize || 'N/A'}
                </div>
                <div>
                  <strong>{__('htaccess rewrite', 'prorank-seo')}:</strong> {supportLabel(serverSupport.htaccess)}
                </div>
                <div>
                  <strong>{__('Recommended batch', 'prorank-seo')}:</strong> {recommendedBatchSize()}
                </div>
              </div>
            </div>
          )}
        </div>
      </Card>

      {/* Format & Compression Settings */}
      <Card variant="performance">
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Format & Compression', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Generate modern formats and balance quality with size.', 'prorank-seo')}
            </p>
          </div>

          <div className="pr-flex pr-flex-col pr-gap-1">
            <span className="pr-text-sm pr-font-medium pr-text-gray-700">
              {__('Optimization method', 'prorank-seo')}
            </span>
            <span className="pr-text-sm pr-text-gray-600">
              {__('Local server (default)', 'prorank-seo')}
            </span>
          </div>

          <div className="pr-flex pr-items-center pr-gap-3">
            <Toggle
              label={__('Enable WebP', 'prorank-seo')}
              checked={Boolean(settings.webp_enabled)}
              onChange={(value) => setSettings((prev) => ({ ...prev, webp_enabled: value }))}
            />
            <FeatureBadge feature="webp" />
          </div>
          <RangeInput
            label={__('WebP quality', 'prorank-seo')}
            value={Number(settings.webp_quality)}
            onChange={(value) => setSettings((prev) => ({ ...prev, webp_quality: value }))}
            min={50}
            max={100}
          />

          <div className="pr-flex pr-items-center pr-gap-3">
            <Toggle
              label={__('Enable AVIF', 'prorank-seo')}
              checked={Boolean(settings.avif_enabled)}
              onChange={(value) => setSettings((prev) => ({ ...prev, avif_enabled: value }))}
              disabled={serverSupport?.avif === false}
            />
            <FeatureBadge feature="avif" />
          </div>
          <RangeInput
            label={__('AVIF quality', 'prorank-seo')}
            value={Number(settings.avif_quality)}
            onChange={(value) => setSettings((prev) => ({ ...prev, avif_quality: value }))}
            min={40}
            max={100}
            disabled={serverSupport?.avif === false}
          />

          <div className="pr-flex pr-items-center pr-gap-3">
            <Toggle
              label={__('Enable JPEG XL', 'prorank-seo')}
              checked={Boolean(settings.jxl_enabled)}
              onChange={(value) => setSettings((prev) => ({ ...prev, jxl_enabled: value }))}
            />
            <FeatureBadge feature="jxl" />
          </div>
          <RangeInput
            label={__('JPEG XL quality', 'prorank-seo')}
            value={Number(settings.jxl_quality)}
            onChange={(value) => setSettings((prev) => ({ ...prev, jxl_quality: value }))}
            min={40}
            max={100}
          />

          <RangeInput
            label={__('PNG quality', 'prorank-seo')}
            value={Number(settings.png_quality)}
            onChange={(value) => setSettings((prev) => ({ ...prev, png_quality: value }))}
            min={50}
            max={100}
          />

          <Select
            label={__('Compression type', 'prorank-seo')}
            value={settings.compression_type}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, compression_type: event.target.value as ImageSettings['compression_type'] }))
            }
            options={[
              { label: __('Lossless (best quality)', 'prorank-seo'), value: 'lossless' },
              { label: __('Glossy (balanced)', 'prorank-seo'), value: 'glossy' },
              { label: __('Lossy (smallest files)', 'prorank-seo'), value: 'lossy' },
            ]}
          />
        </div>
      </Card>

      {/* Automation Settings */}
      <Card variant="performance">
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Automation', 'prorank-seo')}
            </h3>
          </div>

          <Toggle
            label={__('Optimize on upload', 'prorank-seo')}
            checked={Boolean(settings.optimize_on_upload)}
            onChange={(value) => setSettings((prev) => ({ ...prev, optimize_on_upload: value }))}
            description={__('Automatically optimize new uploads.', 'prorank-seo')}
          />

          <Toggle
            label={__('Keep backups of originals', 'prorank-seo')}
            checked={Boolean(settings.backup_originals)}
            onChange={(value) => setSettings((prev) => ({ ...prev, backup_originals: value }))}
            description={__('Store a backup for easy restore.', 'prorank-seo')}
          />
        </div>
      </Card>

      {/* Lazy Loading Settings */}
      <Card variant="performance">
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Lazy Loading', 'prorank-seo')}
            </h3>
          </div>

          <Toggle
            label={__('Lazy-load images', 'prorank-seo')}
            checked={Boolean(settings.lazyload_images)}
            onChange={(value) => setSettings((prev) => ({ ...prev, lazyload_images: value }))}
          />
          <Toggle
            label={__('Lazy-load iframes', 'prorank-seo')}
            checked={Boolean(settings.lazyload_iframes)}
            onChange={(value) => setSettings((prev) => ({ ...prev, lazyload_iframes: value }))}
          />
          <Toggle
            label={__('Lazy-load videos', 'prorank-seo')}
            checked={Boolean(settings.lazyload_videos)}
            onChange={(value) => setSettings((prev) => ({ ...prev, lazyload_videos: value }))}
          />

          <RangeInput
            label={__('Loading threshold (px)', 'prorank-seo')}
            value={Number(settings.lazyload_threshold)}
            onChange={(value) => setSettings((prev) => ({ ...prev, lazyload_threshold: value }))}
            min={0}
            max={800}
            step={10}
          />
        </div>
      </Card>

      {/* Delivery Settings */}
      <Card variant="performance">
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Delivery', 'prorank-seo')}
            </h3>
          </div>

          <Select
            label={__('WebP/AVIF delivery method', 'prorank-seo')}
            value={settings.delivery_method}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, delivery_method: event.target.value as ImageSettings['delivery_method'] }))
            }
            options={[
              { label: __('Picture tag (recommended)', 'prorank-seo'), value: 'picture' },
              { label: __('Accept header detection', 'prorank-seo'), value: 'accept_header' },
              { label: __('Rewrite rules', 'prorank-seo'), value: 'htaccess' },
              { label: __('CDN-based delivery', 'prorank-seo'), value: 'cdn' },
              { label: __('None (manual)', 'prorank-seo'), value: 'none' },
            ]}
          />

          <Textarea
            label={__('Exclude paths', 'prorank-seo')}
            value={settings.exclude_paths}
            onChange={(event) => setSettings((prev) => ({ ...prev, exclude_paths: event.target.value }))}
            placeholder={__('One path per line, e.g. /wp-admin/', 'prorank-seo')}
            rows={4}
          />
        </div>
      </Card>

      {/* Save Button */}
      <div className="pr-flex pr-gap-3">
        <Button variant="primary" onClick={() => saveSettings()} loading={saving} disabled={saving}>
          {saving ? __('Saving...', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>
        <Button variant="secondary" onClick={loadSettings} disabled={saving}>
          {__('Reload', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

export default ImageOptimizationSettings;
