import { useEffect, useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Button, Card, Textarea, Toggle } from '../../../components/ui';
import LoadingState from '../../../components/LoadingState';
import { useNotification } from '../../../contexts/NotificationContext';
import * as React from 'react';

interface CacheSettings {
  browser_cache_enabled: boolean;
  browser_cache_images: string;
  browser_cache_css: string;
  browser_cache_js: string;
  browser_cache_fonts: string;
  browser_cache_html: string;
  dns_prefetch_enabled: boolean;
  dns_prefetch_hosts: string;
  font_preload_enabled: boolean;
  font_preload_urls: string;
}

interface RumSettingsResponse {
  enabled?: boolean;
}

const DEFAULT_SETTINGS: CacheSettings = {
  browser_cache_enabled: false,
  browser_cache_images: '31536000',
  browser_cache_css: '31536000',
  browser_cache_js: '31536000',
  browser_cache_fonts: '31536000',
  browser_cache_html: '3600',
  dns_prefetch_enabled: false,
  dns_prefetch_hosts: '',
  font_preload_enabled: false,
  font_preload_urls: '',
};

// Cache duration options with human-readable labels
const CACHE_DURATION_OPTIONS = [
  { value: '3600', label: __('1 hour', 'prorank-seo') },
  { value: '86400', label: __('1 day', 'prorank-seo') },
  { value: '604800', label: __('1 week', 'prorank-seo') },
  { value: '2592000', label: __('1 month', 'prorank-seo') },
  { value: '15552000', label: __('6 months', 'prorank-seo') },
  { value: '31536000', label: __('1 year', 'prorank-seo') },
];

// Shorter options for HTML (which changes more frequently)
const HTML_CACHE_OPTIONS = [
  { value: '0', label: __('No cache', 'prorank-seo') },
  { value: '300', label: __('5 minutes', 'prorank-seo') },
  { value: '600', label: __('10 minutes', 'prorank-seo') },
  { value: '1800', label: __('30 minutes', 'prorank-seo') },
  { value: '3600', label: __('1 hour', 'prorank-seo') },
  { value: '86400', label: __('1 day', 'prorank-seo') },
];

const CachingDeliverySettings: React.FC = () => {
  const [settings, setSettings] = useState<CacheSettings>(DEFAULT_SETTINGS);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [rumEnabled, setRumEnabled] = useState(false);
  const [rumLoading, setRumLoading] = useState(false);
  const [rumSaving, setRumSaving] = useState(false);
  const { showNotification } = useNotification();
  const litespeedInfo = (window as any)?.prorankSeoAdmin?.server?.litespeed ?? {};
  const litespeedMessage = litespeedInfo.cache_plugin
    ? __('LiteSpeed Cache detected. ProRank will purge LSCache automatically on content updates.', 'prorank-seo')
    : litespeedInfo.server
      ? __('LiteSpeed server detected. ProRank is compatible and can purge LSCache when active.', 'prorank-seo')
      : __('LiteSpeed compatible. If you use LiteSpeed Cache, ProRank can purge it on content updates.', 'prorank-seo');
  const litespeedTone = litespeedInfo.cache_plugin || litespeedInfo.server
    ? 'prorank-notice prorank-notice-success'
    : 'prorank-notice prorank-notice-info';

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiFetch<{ settings?: Partial<CacheSettings> }>({
        path: '/prorank-seo/v1/performance/cache-settings',
      });

      if (response?.settings) {
        setSettings((prev) => ({ ...prev, ...response.settings }));
      }
    } catch (error) {
      showNotification(__('Failed to load cache settings', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  const loadRumSettings = async () => {
    setRumLoading(true);
    try {
      const response = await apiFetch<RumSettingsResponse>({
        path: '/prorank-seo/v1/performance/rum-settings',
      });
      setRumEnabled(Boolean(response?.enabled));
    } catch (error) {
      showNotification(__('Failed to load RUM settings', 'prorank-seo'), 'error');
    } finally {
      setRumLoading(false);
    }
  };

  const saveRumSettings = async (value: boolean) => {
    setRumSaving(true);
    try {
      await apiFetch({
        path: '/prorank-seo/v1/performance/rum-settings',
        method: 'POST',
        data: {
          enabled: value,
        },
      });
      setRumEnabled(value);
      showNotification(
        value
          ? __('RUM collection enabled', 'prorank-seo')
          : __('RUM collection disabled', 'prorank-seo'),
        'success'
      );
    } catch (error) {
      showNotification(__('Failed to update RUM setting', 'prorank-seo'), 'error');
    } finally {
      setRumSaving(false);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      await apiFetch({
        path: '/prorank-seo/v1/performance/cache-settings',
        method: 'POST',
        data: {
          settings,
        },
      });
      showNotification(__('Cache settings saved', 'prorank-seo'), 'success');
    } catch (error) {
      showNotification(__('Failed to save cache settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  useEffect(() => {
    loadSettings();
    loadRumSettings();
  }, []);

  if (loading) {
    return <LoadingState message={__('Loading cache settings…', 'prorank-seo')} />;
  }

  return (
    <div className="pr-space-y-6 pr-max-w-4xl">
      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Real User Monitoring (RUM)', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Collect Core Web Vitals (LCP, CLS, INP) from real visitors to measure field performance.', 'prorank-seo')}
            </p>
          </div>

          <Toggle
            label={__('Enable RUM collection', 'prorank-seo')}
            checked={Boolean(rumEnabled)}
            onChange={(value) => saveRumSettings(value)}
            description={__('Collects anonymous performance metrics and stores aggregate samples locally.', 'prorank-seo')}
            disabled={rumLoading || rumSaving}
          />
        </div>
      </Card>

      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Browser Cache', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Control cache headers for static assets so returning visitors load faster.', 'prorank-seo')}
            </p>
            <div className={`${litespeedTone} pr-mt-3`}>
              {litespeedMessage}
            </div>
          </div>

          <Toggle
            label={__('Enable browser caching', 'prorank-seo')}
            checked={Boolean(settings.browser_cache_enabled)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, browser_cache_enabled: value }))
            }
            description={__('Adds cache-control headers for images, CSS, JS, and fonts.', 'prorank-seo')}
          />

          <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-2 lg:pr-grid-cols-3 pr-gap-4">
            <div className="pr-space-y-1">
              <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700">
                {__('Images', 'prorank-seo')}
              </label>
              <select
                className="pr-w-full pr-rounded-md pr-border pr-border-gray-300 pr-bg-white pr-px-3 pr-py-2 pr-text-sm focus:pr-border-blue-500 focus:pr-outline-none focus:pr-ring-1 focus:pr-ring-blue-500"
                value={settings.browser_cache_images}
                onChange={(e) =>
                  setSettings((prev) => ({ ...prev, browser_cache_images: e.target.value }))
                }
              >
                {CACHE_DURATION_OPTIONS.map((opt) => (
                  <option key={opt.value} value={opt.value}>{opt.label}</option>
                ))}
              </select>
            </div>
            <div className="pr-space-y-1">
              <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700">
                {__('CSS', 'prorank-seo')}
              </label>
              <select
                className="pr-w-full pr-rounded-md pr-border pr-border-gray-300 pr-bg-white pr-px-3 pr-py-2 pr-text-sm focus:pr-border-blue-500 focus:pr-outline-none focus:pr-ring-1 focus:pr-ring-blue-500"
                value={settings.browser_cache_css}
                onChange={(e) =>
                  setSettings((prev) => ({ ...prev, browser_cache_css: e.target.value }))
                }
              >
                {CACHE_DURATION_OPTIONS.map((opt) => (
                  <option key={opt.value} value={opt.value}>{opt.label}</option>
                ))}
              </select>
            </div>
            <div className="pr-space-y-1">
              <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700">
                {__('JavaScript', 'prorank-seo')}
              </label>
              <select
                className="pr-w-full pr-rounded-md pr-border pr-border-gray-300 pr-bg-white pr-px-3 pr-py-2 pr-text-sm focus:pr-border-blue-500 focus:pr-outline-none focus:pr-ring-1 focus:pr-ring-blue-500"
                value={settings.browser_cache_js}
                onChange={(e) =>
                  setSettings((prev) => ({ ...prev, browser_cache_js: e.target.value }))
                }
              >
                {CACHE_DURATION_OPTIONS.map((opt) => (
                  <option key={opt.value} value={opt.value}>{opt.label}</option>
                ))}
              </select>
            </div>
            <div className="pr-space-y-1">
              <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700">
                {__('Fonts', 'prorank-seo')}
              </label>
              <select
                className="pr-w-full pr-rounded-md pr-border pr-border-gray-300 pr-bg-white pr-px-3 pr-py-2 pr-text-sm focus:pr-border-blue-500 focus:pr-outline-none focus:pr-ring-1 focus:pr-ring-blue-500"
                value={settings.browser_cache_fonts}
                onChange={(e) =>
                  setSettings((prev) => ({ ...prev, browser_cache_fonts: e.target.value }))
                }
              >
                {CACHE_DURATION_OPTIONS.map((opt) => (
                  <option key={opt.value} value={opt.value}>{opt.label}</option>
                ))}
              </select>
            </div>
            <div className="pr-space-y-1">
              <label className="pr-block pr-text-sm pr-font-medium pr-text-gray-700">
                {__('HTML Pages', 'prorank-seo')}
              </label>
              <select
                className="pr-w-full pr-rounded-md pr-border pr-border-gray-300 pr-bg-white pr-px-3 pr-py-2 pr-text-sm focus:pr-border-blue-500 focus:pr-outline-none focus:pr-ring-1 focus:pr-ring-blue-500"
                value={settings.browser_cache_html}
                onChange={(e) =>
                  setSettings((prev) => ({ ...prev, browser_cache_html: e.target.value }))
                }
              >
                {HTML_CACHE_OPTIONS.map((opt) => (
                  <option key={opt.value} value={opt.value}>{opt.label}</option>
                ))}
              </select>
            </div>
          </div>
        </div>
      </Card>

      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Resource Hints', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Preconnect or DNS-prefetch important third-party domains for faster connections.', 'prorank-seo')}
            </p>
          </div>

          <Toggle
            label={__('Enable DNS prefetch', 'prorank-seo')}
            checked={Boolean(settings.dns_prefetch_enabled)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, dns_prefetch_enabled: value }))
            }
            description={__('Adds dns-prefetch hints for the domains below.', 'prorank-seo')}
          />

          <Textarea
            label={__('DNS prefetch hosts', 'prorank-seo')}
            value={settings.dns_prefetch_hosts}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, dns_prefetch_hosts: event.target.value }))
            }
            placeholder={__('https://fonts.googleapis.com\nhttps://cdn.example.com', 'prorank-seo')}
            rows={4}
            helperText={__('One URL per line.', 'prorank-seo')}
          />

          <Toggle
            label={__('Enable font preloading', 'prorank-seo')}
            checked={Boolean(settings.font_preload_enabled)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, font_preload_enabled: value }))
            }
            description={__('Preload specific font files for faster first paint.', 'prorank-seo')}
          />

          <Textarea
            label={__('Font preload URLs', 'prorank-seo')}
            value={settings.font_preload_urls}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, font_preload_urls: event.target.value }))
            }
            placeholder={__('https://example.com/fonts/your-font.woff2', 'prorank-seo')}
            rows={3}
            helperText={__('One URL per line.', 'prorank-seo')}
          />
        </div>
      </Card>

      <div className="pr-flex pr-gap-3">
        <Button variant="primary" onClick={saveSettings} loading={saving} disabled={saving}>
          {saving ? __('Saving…', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>
        <Button variant="secondary" onClick={loadSettings} disabled={saving}>
          {__('Reload', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

export default CachingDeliverySettings;
