import { useEffect, useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Button, Card, Toggle } from '../../../components/ui';
import LoadingState from '../../../components/LoadingState';
import { useNotification } from '../../../contexts/NotificationContext';
import * as React from 'react';

interface AssetSettings {
  js_minify_enabled: boolean;
  js_minify_inline: boolean;
  host_google_fonts_locally: boolean;
  font_display_swap: boolean;
}

const DEFAULT_SETTINGS: AssetSettings = {
  js_minify_enabled: false,
  js_minify_inline: false,
  host_google_fonts_locally: false,
  font_display_swap: false,
};

const AssetOptimizationSettings: React.FC = () => {
  const [settings, setSettings] = useState<AssetSettings>(DEFAULT_SETTINGS);
  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const { showNotification } = useNotification();

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiFetch<{ settings?: Partial<AssetSettings> }>({
        path: '/prorank-seo/v1/performance/asset-settings',
      });

      if (response?.settings) {
        setSettings((prev) => ({ ...prev, ...response.settings }));
      }
    } catch (error) {
      showNotification(__('Failed to load asset settings', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      await apiFetch({
        path: '/prorank-seo/v1/performance/asset-settings',
        method: 'POST',
        data: {
          settings,
        },
      });

      showNotification(__('Asset settings saved', 'prorank-seo'), 'success');
    } catch (error) {
      showNotification(__('Failed to save asset settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  useEffect(() => {
    loadSettings();
  }, []);

  if (loading) {
    return <LoadingState message={__('Loading asset settings…', 'prorank-seo')} />;
  }

  return (
    <div className="pr-space-y-6 pr-max-w-4xl">
      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('JavaScript Optimization', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Reduce JavaScript size and processing time with safe minification.', 'prorank-seo')}
            </p>
          </div>

          <Toggle
            label={__('Minify JavaScript files', 'prorank-seo')}
            checked={Boolean(settings.js_minify_enabled)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, js_minify_enabled: value }))
            }
            description={__('Minify enqueued scripts for faster delivery.', 'prorank-seo')}
          />

          <Toggle
            label={__('Minify inline scripts', 'prorank-seo')}
            checked={Boolean(settings.js_minify_inline)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, js_minify_inline: value }))
            }
            description={__('Minify inline script blocks on your pages.', 'prorank-seo')}
          />
        </div>
      </Card>

      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Font Optimization', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Improve font loading behavior for faster text rendering.', 'prorank-seo')}
            </p>
          </div>

          <Toggle
            label={__('Host Google Fonts locally', 'prorank-seo')}
            checked={Boolean(settings.host_google_fonts_locally)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, host_google_fonts_locally: value }))
            }
            description={__('Serve Google Fonts from your own server to reduce external requests.', 'prorank-seo')}
          />

          <Toggle
            label={__('Use font-display: swap', 'prorank-seo')}
            checked={Boolean(settings.font_display_swap)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, font_display_swap: value }))
            }
            description={__('Ensure text remains visible while fonts load.', 'prorank-seo')}
          />
        </div>
      </Card>

      <div className="pr-flex pr-gap-3">
        <Button variant="primary" onClick={saveSettings} loading={saving} disabled={saving}>
          {saving ? __('Saving…', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>
        <Button variant="secondary" onClick={loadSettings} disabled={saving}>
          {__('Reload', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

export default AssetOptimizationSettings;
