/**
 * Site Information Tab Component
 * 
 * Manages basic site information including name, tagline, separator, and images
 * 
 * @package ProRank\SEO
 */

import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import {
  TextControl,
  TextareaControl,
  SelectControl,
  __experimentalVStack as VStack,
  __experimentalText as Text,
} from '@wordpress/components';
import apiFetch from '@wordpress/api-fetch';
import ProrankButton from '../../../../components/ProrankButton';

const SiteInfoTab = ({ settings, updateSettings, saveSettings }) => {
  const [isSaving, setIsSaving] = useState(false);
  const [hasChanges, setHasChanges] = useState(false);

  // Get WordPress site name and tagline if not set
  useEffect(() => {
    // Get WordPress site name and tagline if not set
    if (!settings.site_name && window.prorankSeoAdmin && window.prorankSeoAdmin.site_name) {
      updateSettings({ site_name: window.prorankSeoAdmin.site_name });
    }
    if (!settings.site_tagline && window.prorankSeoAdmin && window.prorankSeoAdmin.site_tagline) {
      updateSettings({ site_tagline: window.prorankSeoAdmin.site_tagline });
    }
  }, []);

  // Title separators with their display and actual values
  const separatorOptions = [
    { label: '- (Dash)', value: '-' },
    { label: '– (En Dash)', value: '–' },
    { label: '— (Em Dash)', value: '—' },
    { label: ': (Colon)', value: ':' },
    { label: '· (Middle Dot)', value: '·' },
    { label: '• (Bullet)', value: '•' },
    { label: '* (Asterisk)', value: '*' },
    { label: '⋆ (Star)', value: '⋆' },
    { label: '| (Pipe)', value: '|' },
    { label: '~ (Tilde)', value: '~' },
    { label: '« (Left Guillemet)', value: '«' },
    { label: '» (Right Guillemet)', value: '»' },
    { label: '> (Greater Than)', value: '>' },
    { label: '< (Less Than)', value: '<' },
  ];

  const handleChange = (key, value) => {
    updateSettings({ [key]: value });
    setHasChanges(true);
    
    // Update WordPress site title and tagline if changed
    if (key === 'site_name') {
      updateWordPressSetting('blogname', value);
    } else if (key === 'site_tagline') {
      updateWordPressSetting('blogdescription', value);
    }
  };

  const updateWordPressSetting = async (option, value) => {
    try {
      await apiFetch({
        path: '/wp/v2/settings',
        method: 'POST',
        data: { [option]: value },
      });
    } catch (error) {
      // Failed to update WordPress setting
    }
  };

  const handleSave = async () => {
    setIsSaving(true);
    const success = await saveSettings(settings);
    if (success) {
      setHasChanges(false);
    }
    setIsSaving(false);
  };

  const openMediaLibrary = (key) => {
    if (!window.wp || !window.wp.media) {
      // Media library not available
      return;
    }

    const mediaUploader = window.wp.media({
      title: __('Select Image', 'prorank-seo'),
      button: {
        text: __('Use this image', 'prorank-seo')
      },
      multiple: false,
      library: {
        type: 'image'
      }
    });

    mediaUploader.on('select', () => {
      const attachment = mediaUploader.state().get('selection').first().toJSON();
      updateSettings({ 
        [key]: attachment.url,
        [`${key}_id`]: attachment.id 
      });
      setHasChanges(true);
    });

    mediaUploader.open();
  };

  const handleRemoveImage = (key) => {
    updateSettings({ 
      [key]: '',
      [`${key}_id`]: '' 
    });
    setHasChanges(true);
  };

  return (
    <VStack spacing={4}>
      <div className="prorank-card">
        <div className="prorank-card-header">
          <h3>{__('Website Information', 'prorank-seo')}</h3>
        </div>
        <div className="prorank-card-body">
          <VStack spacing={4}>
            <TextControl
              label={__('Website Name', 'prorank-seo')}
              value={settings.site_name || ''}
              onChange={(value) => handleChange('site_name', value)}
              help={__('This is your site\'s name. It appears in search results and browser tabs.', 'prorank-seo')}
              __nextHasNoMarginBottom={true}
              __next40pxDefaultSize={true}
            />

            <TextControl
              label={__('Alternate Website Name', 'prorank-seo')}
              value={settings.site_alternate_name || ''}
              onChange={(value) => handleChange('site_alternate_name', value)}
              help={__('Use for acronyms or a shorter version of your website\'s name.', 'prorank-seo')}
              __nextHasNoMarginBottom={true}
              __next40pxDefaultSize={true}
            />

            <TextareaControl
              label={__('Tagline', 'prorank-seo')}
              value={settings.site_tagline || ''}
              onChange={(value) => handleChange('site_tagline', value)}
              help={__('This field updates the tagline in your WordPress settings. Use it as a brief description of your site.', 'prorank-seo')}
              rows={2}
              __nextHasNoMarginBottom={true}
            />

            <SelectControl
              label={__('Title Separator', 'prorank-seo')}
              value={settings.title_separator || '-'}
              options={separatorOptions}
              onChange={(value) => handleChange('title_separator', value)}
              help={__('This separator is used between your page title and site name in search results.', 'prorank-seo')}
              __nextHasNoMarginBottom={true}
              __next40pxDefaultSize={true}
            />

            <div className="prorank-preview-box">
              <Text size="12" variant="muted" style={{ display: 'block', marginBottom: '8px' }}>
                {__('Preview:', 'prorank-seo')}
              </Text>
              <div style={{ 
                fontSize: '16px', 
                fontWeight: '500',
                color: '#1a0dab',
                fontFamily: 'arial, sans-serif'
              }}>
                {__('Page Title', 'prorank-seo')} {settings.title_separator || '-'} {settings.site_name || __('Your Site Name', 'prorank-seo')}
              </div>
              <div style={{
                fontSize: '14px',
                color: '#006621',
                marginTop: '4px'
              }}>
                {window.location.hostname || 'yoursite.com'}
              </div>
            </div>
          </VStack>
        </div>
      </div>

      <div className="prorank-card">
        <div className="prorank-card-header">
          <h3>{__('Site Images', 'prorank-seo')}</h3>
        </div>
        <div className="prorank-card-body">
          <div className="prorank-site-images-grid">
            {/* Site Logo */}
            <div className="prorank-image-upload-box">
              <div className="prorank-image-upload-header">
                <div className="prorank-image-upload-icon">
                  <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M12 2L2 7V17C2 17.55 2.45 18 3 18H9V12H15V18H21C21.55 18 22 17.55 22 17V7L12 2Z" fill="#3b82f6"/>
                  </svg>
                </div>
                <div className="prorank-image-upload-info">
                  <h4>{__('Site Logo', 'prorank-seo')}</h4>
                  <p>{__('Appears in search results and social shares', 'prorank-seo')}</p>
                  <span className="prorank-image-upload-specs">{__('Recommended: Square image, min 512x512px', 'prorank-seo')}</span>
                </div>
              </div>
              
              <div className="prorank-image-upload-area">
                {settings.site_logo ? (
                  <div className="prorank-image-preview-modern">
                    <div className="prorank-image-preview-wrapper">
                      <img src={settings.site_logo} alt={__('Site Logo', 'prorank-seo')} />
                      <div className="prorank-image-overlay">
                        <ProrankButton 
                          variant="primary"
                          size="small"
                          onClick={() => openMediaLibrary('site_logo')}
                        >
                          <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"/>
                          </svg>
                          {__('Change', 'prorank-seo')}
                        </ProrankButton>
                        <ProrankButton 
                          variant="outline"
                          size="small"
                          onClick={() => handleRemoveImage('site_logo')}
                        >
                          <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/>
                          </svg>
                          {__('Remove', 'prorank-seo')}
                        </ProrankButton>
                      </div>
                    </div>
                    <div className="prorank-image-filename">
                      {settings.site_logo.split('/').pop()}
                    </div>
                  </div>
                ) : (
                  <button 
                    className="prorank-image-upload-trigger"
                    onClick={() => openMediaLibrary('site_logo')}
                  >
                    <svg width="48" height="48" viewBox="0 0 24 24" fill="none">
                      <path d="M9 16.2L4.8 12l-1.4 1.4L9 19 21 7l-1.4-1.4L9 16.2z" fill="none"/>
                      <rect x="3" y="3" width="18" height="18" rx="2" stroke="#3b82f6" strokeWidth="2" strokeDasharray="3 3"/>
                      <path d="M12 8v8M16 12H8" stroke="#3b82f6" strokeWidth="2" strokeLinecap="round"/>
                    </svg>
                    <span className="prorank-upload-text">{__('Click to upload logo', 'prorank-seo')}</span>
                    <span className="prorank-upload-hint">{__('or drag and drop', 'prorank-seo')}</span>
                  </button>
                )}
              </div>
            </div>

            {/* Default Site Image */}
            <div className="prorank-image-upload-box">
              <div className="prorank-image-upload-header">
                <div className="prorank-image-upload-icon">
                  <svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                    <path d="M21 19V5C21 3.9 20.1 3 19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19ZM8.5 13.5L11 16.51L14.5 12L19 18H5L8.5 13.5Z" fill="#2563eb"/>
                  </svg>
                </div>
                <div className="prorank-image-upload-info">
                  <h4>{__('Default Site Image', 'prorank-seo')}</h4>
                  <p>{__('Fallback for posts without featured images', 'prorank-seo')}</p>
                  <span className="prorank-image-upload-specs">{__('Recommended: 1200x675px (16:9 ratio)', 'prorank-seo')}</span>
                </div>
              </div>
              
              <div className="prorank-image-upload-area">
                {settings.site_image ? (
                  <div className="prorank-image-preview-modern">
                    <div className="prorank-image-preview-wrapper og-preview">
                      <img src={settings.site_image} alt={__('Site Image', 'prorank-seo')} />
                      <div className="prorank-image-overlay">
                        <ProrankButton 
                          variant="primary"
                          size="small"
                          onClick={() => openMediaLibrary('site_image')}
                        >
                          <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z"/>
                          </svg>
                          {__('Change', 'prorank-seo')}
                        </ProrankButton>
                        <ProrankButton 
                          variant="outline"
                          size="small"
                          onClick={() => handleRemoveImage('site_image')}
                        >
                          <svg width="16" height="16" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M6 19c0 1.1.9 2 2 2h8c1.1 0 2-.9 2-2V7H6v12zM19 4h-3.5l-1-1h-5l-1 1H5v2h14V4z"/>
                          </svg>
                          {__('Remove', 'prorank-seo')}
                        </ProrankButton>
                      </div>
                    </div>
                    <div className="prorank-image-filename">
                      {settings.site_image.split('/').pop()}
                    </div>
                  </div>
                ) : (
                  <button 
                    className="prorank-image-upload-trigger"
                    onClick={() => openMediaLibrary('site_image')}
                  >
                    <svg width="48" height="48" viewBox="0 0 24 24" fill="none">
                      <rect x="3" y="3" width="18" height="18" rx="2" stroke="#2563eb" strokeWidth="2" strokeDasharray="3 3"/>
                      <path d="M12 8v8M16 12H8" stroke="#2563eb" strokeWidth="2" strokeLinecap="round"/>
                    </svg>
                    <span className="prorank-upload-text">{__('Click to upload image', 'prorank-seo')}</span>
                    <span className="prorank-upload-hint">{__('or drag and drop', 'prorank-seo')}</span>
                  </button>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>

      {hasChanges && (
        <div className="prorank-card">
          <div className="prorank-card-body">
            <div className="prorank-btn-group" style={{ justifyContent: 'flex-end' }}>
              <ProrankButton
                variant="secondary"
                onClick={() => {
                  window.location.reload();
                }}
                disabled={isSaving}
              >
                {__('Discard Changes', 'prorank-seo')}
              </ProrankButton>
              <ProrankButton
                variant="success"
                onClick={handleSave}
                disabled={isSaving}
              >
                {isSaving ? __('Saving...', 'prorank-seo') : __('Save Changes', 'prorank-seo')}
              </ProrankButton>
            </div>
          </div>
        </div>
      )}
    </VStack>
  );
};

export default SiteInfoTab;