import { useEffect, useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Button, Card, Input, Select, Toggle } from '../../../components/ui';
import LoadingState from '../../../components/LoadingState';
import { useNotification } from '../../../contexts/NotificationContext';

const PodcastSeoSettings = () => {
  const [settings, setSettings] = useState({
    podcast_title: '',
    podcast_subtitle: '',
    podcast_author: '',
    podcast_email: '',
    podcast_category: '',
    podcast_subcategory: '',
    podcast_explicit: 'no',
    podcast_language: '',
    podcast_copyright: '',
    podcast_image: '',
    podcast_owner_name: '',
    podcast_owner_email: '',
    enable_podcast_feed: false,
    podcast_feed_limit: '',
    enable_itunes_tags: true,
    podcast_redirect_feed: '',
  });

  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const { showNotification } = useNotification();

  const podcastCategories = [
    { value: '', label: __('-- Select Category --', 'prorank-seo') },
    { value: 'Arts', label: __('Arts', 'prorank-seo') },
    { value: 'Business', label: __('Business', 'prorank-seo') },
    { value: 'Comedy', label: __('Comedy', 'prorank-seo') },
    { value: 'Education', label: __('Education', 'prorank-seo') },
    { value: 'Health & Fitness', label: __('Health & Fitness', 'prorank-seo') },
    { value: 'News', label: __('News', 'prorank-seo') },
    { value: 'Science', label: __('Science', 'prorank-seo') },
    { value: 'Society & Culture', label: __('Society & Culture', 'prorank-seo') },
    { value: 'Sports', label: __('Sports', 'prorank-seo') },
    { value: 'Technology', label: __('Technology', 'prorank-seo') },
    { value: 'True Crime', label: __('True Crime', 'prorank-seo') },
    { value: 'TV & Film', label: __('TV & Film', 'prorank-seo') },
  ];

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/podcast/settings',
      });

      if (response?.success && response?.data) {
        setSettings((prev) => ({ ...prev, ...response.data }));
      }
    } catch (error) {
      // Allow empty settings on first run.
    } finally {
      setLoading(false);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/podcast/settings',
        method: 'POST',
        data: { settings },
      });

      if (response?.success) {
        showNotification(__('Podcast settings saved', 'prorank-seo'), 'success');
      } else {
        showNotification(__('Failed to save podcast settings', 'prorank-seo'), 'error');
      }
    } catch (error) {
      showNotification(__('Failed to save podcast settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  if (loading) {
    return <LoadingState message={__('Loading podcast settings…', 'prorank-seo')} />;
  }

  return (
    <div className="pr-space-y-6 pr-max-w-4xl">
      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Podcast Details', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Basic information used for podcast feeds and schema.', 'prorank-seo')}
            </p>
          </div>

          <Input
            label={__('Podcast title', 'prorank-seo')}
            value={settings.podcast_title}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_title: event.target.value }))
            }
          />

          <Input
            label={__('Podcast subtitle', 'prorank-seo')}
            value={settings.podcast_subtitle}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_subtitle: event.target.value }))
            }
          />

          <Input
            label={__('Podcast author', 'prorank-seo')}
            value={settings.podcast_author}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_author: event.target.value }))
            }
          />

          <Input
            label={__('Contact email', 'prorank-seo')}
            value={settings.podcast_email}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_email: event.target.value }))
            }
          />

          <Select
            label={__('Category', 'prorank-seo')}
            value={settings.podcast_category}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_category: event.target.value }))
            }
            options={podcastCategories}
          />

          <Input
            label={__('Language', 'prorank-seo')}
            value={settings.podcast_language}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_language: event.target.value }))
            }
            placeholder="en"
          />

          <Input
            label={__('Podcast image URL', 'prorank-seo')}
            value={settings.podcast_image}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_image: event.target.value }))
            }
          />
        </div>
      </Card>

      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Feed Settings', 'prorank-seo')}
            </h3>
          </div>

          <Toggle
            label={__('Enable podcast feed', 'prorank-seo')}
            checked={Boolean(settings.enable_podcast_feed)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, enable_podcast_feed: value }))
            }
          />

          <Input
            type="number"
            label={__('Feed item limit', 'prorank-seo')}
            value={settings.podcast_feed_limit}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_feed_limit: event.target.value }))
            }
            placeholder={__('Leave empty for default', 'prorank-seo')}
          />

          <Toggle
            label={__('Enable iTunes tags', 'prorank-seo')}
            checked={Boolean(settings.enable_itunes_tags)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, enable_itunes_tags: value }))
            }
          />

          <Input
            label={__('Redirect feed URL', 'prorank-seo')}
            value={settings.podcast_redirect_feed}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, podcast_redirect_feed: event.target.value }))
            }
            placeholder={__('https://example.com/feed/podcast', 'prorank-seo')}
          />
        </div>
      </Card>

      <div className="pr-flex pr-gap-3">
        <Button variant="primary" onClick={saveSettings} loading={saving} disabled={saving}>
          {saving ? __('Saving…', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>
        <Button variant="secondary" onClick={loadSettings} disabled={saving}>
          {__('Reload', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

export default PodcastSeoSettings;
