import { useEffect, useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Button, Card, Checkbox, Input, Textarea, Toggle } from '../../../components/ui';
import LoadingState from '../../../components/LoadingState';
import { useNotification } from '../../../contexts/NotificationContext';

const InternalLinkingSettings = () => {
  const [settings, setSettings] = useState({
    enabled: true,
    auto_linking: false,
    max_links_per_post: 5,
    min_word_count: 100,
    link_post_types: ['post', 'page'],
    exclude_categories: [],
    ignore_words: '',
    open_internal_new_tab: false,
    open_external_new_tab: true,
    add_nofollow: false,
    add_title_attribute: true,
  });

  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [categories, setCategories] = useState([]);
  const { showNotification } = useNotification();

  useEffect(() => {
    loadSettings();
    loadCategories();
  }, []);

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/internal-linking',
      });

      if (response?.data) {
        setSettings((prev) => ({ ...prev, ...response.data }));
      }
    } catch (error) {
      showNotification(__('Failed to load internal linking settings', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  const loadCategories = async () => {
    try {
      const response = await apiFetch({
        path: '/wp/v2/categories?per_page=100',
      });

      setCategories(
        Array.isArray(response)
          ? response.map((cat) => ({
              label: cat.name,
              value: String(cat.id),
            }))
          : []
      );
    } catch (error) {
      setCategories([]);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/internal-linking',
        method: 'POST',
        data: settings,
      });

      if (response?.success) {
        showNotification(__('Settings saved successfully', 'prorank-seo'), 'success');
      } else {
        showNotification(__('Failed to save settings', 'prorank-seo'), 'error');
      }
    } catch (error) {
      showNotification(__('Failed to save settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  const togglePostType = (postType) => {
    setSettings((prev) => {
      const next = new Set(prev.link_post_types || []);
      if (next.has(postType)) {
        next.delete(postType);
      } else {
        next.add(postType);
      }
      return { ...prev, link_post_types: Array.from(next) };
    });
  };

  const toggleCategory = (categoryId) => {
    setSettings((prev) => {
      const next = new Set(prev.exclude_categories || []);
      if (next.has(categoryId)) {
        next.delete(categoryId);
      } else {
        next.add(categoryId);
      }
      return { ...prev, exclude_categories: Array.from(next) };
    });
  };

  if (loading) {
    return <LoadingState message={__('Loading internal linking…', 'prorank-seo')} />;
  }

  return (
    <div className="pr-space-y-6 pr-max-w-4xl">
      <Card>
        <div className="pr-space-y-5">
          <div>
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Internal Linking Basics', 'prorank-seo')}
            </h3>
            <p className="pr-text-sm pr-text-gray-600">
              {__('Configure how ProRank suggests and inserts internal links.', 'prorank-seo')}
            </p>
          </div>

          <Toggle
            label={__('Enable internal linking', 'prorank-seo')}
            checked={Boolean(settings.enabled)}
            onChange={(value) => setSettings((prev) => ({ ...prev, enabled: value }))}
          />

          <Toggle
            label={__('Auto-link content', 'prorank-seo')}
            checked={Boolean(settings.auto_linking)}
            onChange={(value) => setSettings((prev) => ({ ...prev, auto_linking: value }))}
            description={__('Automatically insert suggested links while editing.', 'prorank-seo')}
          />

          <Input
            type="number"
            label={__('Max links per post', 'prorank-seo')}
            value={String(settings.max_links_per_post ?? 0)}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, max_links_per_post: Number(event.target.value) }))
            }
          />

          <Input
            type="number"
            label={__('Minimum word count', 'prorank-seo')}
            value={String(settings.min_word_count ?? 0)}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, min_word_count: Number(event.target.value) }))
            }
          />
        </div>
      </Card>

      <Card>
        <div className="pr-space-y-4">
          <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
            {__('Content Scope', 'prorank-seo')}
          </h3>

          <div className="pr-space-y-2">
            <p className="pr-text-sm pr-text-gray-600">
              {__('Select post types to include in internal link suggestions.', 'prorank-seo')}
            </p>
            {['post', 'page'].map((postType) => (
              <Checkbox
                key={postType}
                label={postType === 'post' ? __('Posts', 'prorank-seo') : __('Pages', 'prorank-seo')}
                checked={Boolean(settings.link_post_types?.includes(postType))}
                onChange={() => togglePostType(postType)}
              />
            ))}
          </div>

          {categories.length > 0 && (
            <div className="pr-space-y-2">
              <p className="pr-text-sm pr-text-gray-600">
                {__('Exclude categories from suggestions.', 'prorank-seo')}
              </p>
              <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-2 pr-gap-2">
                {categories.map((category) => (
                  <Checkbox
                    key={category.value}
                    label={category.label}
                    checked={Boolean(settings.exclude_categories?.includes(category.value))}
                    onChange={() => toggleCategory(category.value)}
                  />
                ))}
              </div>
            </div>
          )}
        </div>
      </Card>

      <Card>
        <div className="pr-space-y-5">
          <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
            {__('Link Attributes', 'prorank-seo')}
          </h3>

          <Toggle
            label={__('Open internal links in new tab', 'prorank-seo')}
            checked={Boolean(settings.open_internal_new_tab)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, open_internal_new_tab: value }))
            }
          />

          <Toggle
            label={__('Open external links in new tab', 'prorank-seo')}
            checked={Boolean(settings.open_external_new_tab)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, open_external_new_tab: value }))
            }
          />

          <Toggle
            label={__('Add nofollow to external links', 'prorank-seo')}
            checked={Boolean(settings.add_nofollow)}
            onChange={(value) => setSettings((prev) => ({ ...prev, add_nofollow: value }))}
          />

          <Toggle
            label={__('Add title attributes', 'prorank-seo')}
            checked={Boolean(settings.add_title_attribute)}
            onChange={(value) =>
              setSettings((prev) => ({ ...prev, add_title_attribute: value }))
            }
          />

          <Textarea
            label={__('Ignore words', 'prorank-seo')}
            value={settings.ignore_words || ''}
            onChange={(event) =>
              setSettings((prev) => ({ ...prev, ignore_words: event.target.value }))
            }
            placeholder={__('Comma-separated keywords to ignore', 'prorank-seo')}
            rows={4}
          />
        </div>
      </Card>

      <div className="pr-flex pr-gap-3">
        <Button variant="primary" onClick={saveSettings} loading={saving} disabled={saving}>
          {saving ? __('Saving…', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>
        <Button variant="secondary" onClick={loadSettings} disabled={saving}>
          {__('Reload', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

export default InternalLinkingSettings;
