/**
 * Breadcrumbs Settings Component
 *
 * @package
 * @since   1.0.0
 */

import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { useNotification } from '../../../contexts/NotificationContext';
import { Card, Button, Input, Select, Toggle, Alert, Textarea } from '../../../components/ui';
import LoadingState from '../../../components/LoadingState';

const BreadcrumbsSettings = () => {
  const [settings, setSettings] = useState({
    enabled: true,
    separator: '›',
    home_text: 'Home',
    show_current: true,
    show_home: true,
    home_icon: false,
    auto_insert: false,
    position: 'before',
    enable_schema: true,
    schema_type: 'jsonld', // 'microdata', 'jsonld', 'both' - 2025 recommendation is JSON-LD
    style: 'default', // 'default', 'arrow', 'slash', 'pipe'
    display_style: 'inline', // 'inline', 'background', 'compact', 'large'
    show_on_home: false,
    show_on_posts: true,
    show_on_pages: true,
    show_on_archives: true,
    show_on_404: false,
    custom_css: '',
    enable_cache: false,
  });

  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const { showNotification } = useNotification();

  useEffect(() => {
    loadSettings();
  }, []); // eslint-disable-line react-hooks/exhaustive-deps

  const loadSettings = async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/breadcrumbs',
      });

      if (response.success && response.data) {
        setSettings((prev) => ({ ...prev, ...response.data }));
      }
    } catch (error) {
      showNotification(__('Failed to load breadcrumbs settings', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/breadcrumbs',
        method: 'POST',
        data: settings,
      });

      if (response.success) {
        showNotification(__('Breadcrumbs settings saved successfully', 'prorank-seo'), 'success');
      }
    } catch (error) {
      showNotification(__('Failed to save breadcrumbs settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  const updateSetting = (key, value) => {
    setSettings((prev) => ({ ...prev, [key]: value }));
  };

  if (loading) {
    return <LoadingState message={__('Loading breadcrumbs settings…', 'prorank-seo')} />;
  }

  return (
    <div className="pr-space-y-6">
      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('General Settings', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-6">
          <Toggle
            label={__('Enable Breadcrumbs', 'prorank-seo')}
            description={__('Enable breadcrumb navigation on your site', 'prorank-seo')}
            checked={settings.enabled}
            onChange={(value) => updateSetting('enabled', value)}
          />

          <Input
            label={__('Separator', 'prorank-seo')}
            helperText={__('Character used to separate breadcrumb items', 'prorank-seo')}
            value={settings.separator}
            onChange={(e) => updateSetting('separator', e.target.value)}
          />

          <Input
            label={__('Home Text', 'prorank-seo')}
            helperText={__('Text to display for the home link', 'prorank-seo')}
            value={settings.home_text}
            onChange={(e) => updateSetting('home_text', e.target.value)}
          />

          <Toggle
            label={__('Show Current Page', 'prorank-seo')}
            description={__('Display the current page in breadcrumbs', 'prorank-seo')}
            checked={settings.show_current}
            onChange={(value) => updateSetting('show_current', value)}
          />

          <Toggle
            label={__('Show Home Link', 'prorank-seo')}
            description={__('Display a link to the homepage', 'prorank-seo')}
            checked={settings.show_home}
            onChange={(value) => updateSetting('show_home', value)}
          />

          {settings.show_home && (
            <Toggle
              label={__('Show Home Icon', 'prorank-seo')}
              description={__('Display a home icon (🏠) before the home text', 'prorank-seo')}
              checked={settings.home_icon}
              onChange={(value) => updateSetting('home_icon', value)}
            />
          )}
        </div>
      </Card>

      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Display Settings', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-6">
          <Toggle
            label={__('Auto Insert', 'prorank-seo')}
            description={__(
              'Automatically insert breadcrumbs into content (or use [prorank_breadcrumbs] shortcode)',
              'prorank-seo'
            )}
            checked={settings.auto_insert}
            onChange={(value) => updateSetting('auto_insert', value)}
          />

          {settings.auto_insert && (
            <Select
              label={__('Position', 'prorank-seo')}
              helperText={__('Where to insert breadcrumbs in content', 'prorank-seo')}
              value={settings.position}
              options={[
                { label: __('Before Content', 'prorank-seo'), value: 'before' },
                { label: __('After Content', 'prorank-seo'), value: 'after' },
              ]}
              onChange={(e) => updateSetting('position', e.target.value)}
            />
          )}

          <hr className="pr-border-t pr-border-gray-200 pr-my-6" />

          <h3 className="pr-text-base pr-font-semibold pr-text-gray-900">
            {__('Show Breadcrumbs On', 'prorank-seo')}
          </h3>

          <Toggle
            label={__('Homepage', 'prorank-seo')}
            checked={settings.show_on_home}
            onChange={(value) => updateSetting('show_on_home', value)}
          />

          <Toggle
            label={__('Posts', 'prorank-seo')}
            checked={settings.show_on_posts}
            onChange={(value) => updateSetting('show_on_posts', value)}
          />

          <Toggle
            label={__('Pages', 'prorank-seo')}
            checked={settings.show_on_pages}
            onChange={(value) => updateSetting('show_on_pages', value)}
          />

          <Toggle
            label={__('Archives', 'prorank-seo')}
            checked={settings.show_on_archives}
            onChange={(value) => updateSetting('show_on_archives', value)}
          />

          <Toggle
            label={__('404 Pages', 'prorank-seo')}
            checked={settings.show_on_404}
            onChange={(value) => updateSetting('show_on_404', value)}
          />
        </div>
      </Card>

      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Appearance Settings', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-6">
          <Select
            label={__('Separator Style', 'prorank-seo')}
            helperText={__('Choose the visual style for breadcrumb separators', 'prorank-seo')}
            value={settings.style}
            options={[
              { label: __('Default (›)', 'prorank-seo'), value: 'default' },
              { label: __('Arrow (→)', 'prorank-seo'), value: 'arrow' },
              { label: __('Slash (/)', 'prorank-seo'), value: 'slash' },
              { label: __('Pipe (|)', 'prorank-seo'), value: 'pipe' },
            ]}
            onChange={(e) => updateSetting('style', e.target.value)}
          />

          <Select
            label={__('Display Style', 'prorank-seo')}
            helperText={__('Choose how breadcrumbs are displayed', 'prorank-seo')}
            value={settings.display_style}
            options={[
              { label: __('Inline', 'prorank-seo'), value: 'inline' },
              { label: __('With Background', 'prorank-seo'), value: 'background' },
              { label: __('Compact', 'prorank-seo'), value: 'compact' },
              { label: __('Large', 'prorank-seo'), value: 'large' },
            ]}
            onChange={(e) => updateSetting('display_style', e.target.value)}
          />
        </div>
      </Card>

      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Schema Settings', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-6">
          <Toggle
            label={__('Enable Schema Markup', 'prorank-seo')}
            description={__('Add structured data markup to breadcrumbs for better SEO', 'prorank-seo')}
            checked={settings.enable_schema}
            onChange={(value) => updateSetting('enable_schema', value)}
          />

          {settings.enable_schema && (
            <Select
              label={__('Schema Type', 'prorank-seo')}
              helperText={__('Choose which schema format to use. Google recommends JSON-LD as of 2025.', 'prorank-seo')}
              value={settings.schema_type}
              options={[
                { label: __('JSON-LD Only (Recommended)', 'prorank-seo'), value: 'jsonld' },
                { label: __('Both Formats', 'prorank-seo'), value: 'both' },
                { label: __('Microdata Only (Legacy)', 'prorank-seo'), value: 'microdata' },
              ]}
              onChange={(e) => updateSetting('schema_type', e.target.value)}
            />
          )}
        </div>
      </Card>

      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Custom Styling', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-6">
          <Textarea
            label={__('Custom CSS', 'prorank-seo')}
            helperText={__('Add custom CSS to style your breadcrumbs. Use .prorank-breadcrumbs as the main selector.', 'prorank-seo')}
            value={settings.custom_css}
            onChange={(e) => updateSetting('custom_css', e.target.value)}
            rows={6}
            placeholder={`.prorank-breadcrumbs {
  /* Your custom styles */
  font-size: 14px;
}

.prorank-breadcrumb-item a {
  color: #0073aa;
}`}
          />
        </div>
      </Card>

      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Performance', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-6">
          <Toggle
            label={__('Enable Caching', 'prorank-seo')}
            description={__('Cache breadcrumb trails to improve page load performance. Cache is automatically cleared when content is updated.', 'prorank-seo')}
            checked={settings.enable_cache}
            onChange={(value) => updateSetting('enable_cache', value)}
          />
          <Alert variant="info" title={__('Cache Duration', 'prorank-seo')}>
            {__('Breadcrumb cache is stored for 1 hour and is automatically invalidated when posts, pages, or categories are modified.', 'prorank-seo')}
          </Alert>
        </div>
      </Card>

      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Live Preview', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-4">
          <p className="pr-text-sm pr-text-gray-600">
            {__('This is how your breadcrumbs will appear on the frontend:', 'prorank-seo')}
          </p>
          <div className={`
            pr-p-4 pr-rounded-md pr-border pr-border-gray-200
            ${settings.display_style === 'background' ? 'pr-bg-gray-50' : ''}
            ${settings.display_style === 'compact' ? 'pr-text-sm' : ''}
            ${settings.display_style === 'large' ? 'pr-text-lg' : ''}
          `}>
            <ol className="pr-flex pr-flex-wrap pr-items-center pr-gap-2">
              {settings.show_home && (
                <li className="pr-flex pr-items-center pr-gap-2">
                  <a href="#" className="pr-text-primary-500 hover:pr-text-primary-600 pr-transition-colors">
                    {settings.home_icon && <span aria-hidden="true">🏠 </span>}
                    {settings.home_text}
                  </a>
                  <span className="pr-text-gray-400" aria-hidden="true">
                    {settings.style === 'arrow' ? '→' : settings.style === 'slash' ? '/' : settings.style === 'pipe' ? '|' : settings.separator}
                  </span>
                </li>
              )}
              <li className="pr-flex pr-items-center pr-gap-2">
                <a href="#" className="pr-text-primary-500 hover:pr-text-primary-600 pr-transition-colors">
                  {__('Category', 'prorank-seo')}
                </a>
                <span className="pr-text-gray-400" aria-hidden="true">
                  {settings.style === 'arrow' ? '→' : settings.style === 'slash' ? '/' : settings.style === 'pipe' ? '|' : settings.separator}
                </span>
              </li>
              {settings.show_current && (
                <li className="pr-text-gray-600 pr-font-medium">
                  {__('Current Page', 'prorank-seo')}
                </li>
              )}
            </ol>
          </div>
        </div>
      </Card>

      <Card
        header={
          <div className="pr-p-6 pr-border-b pr-border-gray-200 pr-bg-gray-50">
            <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
              {__('Shortcode', 'prorank-seo')}
            </h3>
          </div>
        }
        padding="none"
      >
        <div className="pr-p-6 pr-space-y-6">
          <p className="pr-text-sm pr-text-gray-600">
            {__(
              'Use the following shortcode to manually add breadcrumbs to your content:',
              'prorank-seo'
            )}
          </p>
          <code className="pr-block pr-bg-gray-100 pr-px-3 pr-py-2 pr-rounded pr-font-mono pr-text-sm pr-text-gray-900">
            [prorank_breadcrumbs]
          </code>

          <hr className="pr-border-t pr-border-gray-200 pr-my-6" />

          <div className="pr-space-y-3">
            <p className="pr-text-sm pr-font-medium pr-text-gray-700">
              {__('Shortcode parameters:', 'prorank-seo')}
            </p>
            <ul className="pr-space-y-2 pr-text-sm pr-text-gray-600">
              <li className="pr-flex pr-gap-2">
                <code className="pr-bg-gray-100 pr-px-2 pr-py-0.5 pr-rounded pr-font-mono pr-text-xs">
                  separator=&quot;›&quot;
                </code>
                <span>- {__('Custom separator', 'prorank-seo')}</span>
              </li>
              <li className="pr-flex pr-gap-2">
                <code className="pr-bg-gray-100 pr-px-2 pr-py-0.5 pr-rounded pr-font-mono pr-text-xs">
                  home_text=&quot;Home&quot;
                </code>
                <span>- {__('Custom home text', 'prorank-seo')}</span>
              </li>
              <li className="pr-flex pr-gap-2">
                <code className="pr-bg-gray-100 pr-px-2 pr-py-0.5 pr-rounded pr-font-mono pr-text-xs">
                  show_current=&quot;true&quot;
                </code>
                <span>- {__('Show/hide current page', 'prorank-seo')}</span>
              </li>
              <li className="pr-flex pr-gap-2">
                <code className="pr-bg-gray-100 pr-px-2 pr-py-0.5 pr-rounded pr-font-mono pr-text-xs">
                  style=&quot;arrow&quot;
                </code>
                <span>- {__('Separator style (arrow, slash, pipe)', 'prorank-seo')}</span>
              </li>
              <li className="pr-flex pr-gap-2">
                <code className="pr-bg-gray-100 pr-px-2 pr-py-0.5 pr-rounded pr-font-mono pr-text-xs">
                  class=&quot;custom-class&quot;
                </code>
                <span>- {__('Add custom CSS classes', 'prorank-seo')}</span>
              </li>
            </ul>
          </div>

          <hr className="pr-border-t pr-border-gray-200 pr-my-6" />

          <Alert variant="info" title={__('2025 Google Update', 'prorank-seo')}>
            {__('As of January 2025, Google no longer displays breadcrumbs in mobile search results (only the domain is shown). However, breadcrumbs still appear on desktop SERPs and remain crucial for site navigation and internal linking. JSON-LD is now the preferred structured data format.', 'prorank-seo')}
          </Alert>
        </div>
      </Card>

      <div className="pr-flex pr-justify-end pr-gap-3">
        <Button variant="primary" onClick={saveSettings} disabled={saving} loading={saving}>
          {__('Save Settings', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

export default BreadcrumbsSettings;
