import { useState, useCallback } from '@wordpress/element';
import { useEffect } from 'react';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import PageHeader from '../../components/PageHeader';
import ModuleNav from '../../components/ModuleNav';
import headerBadges from '../../utils/headerBadges';
import { Card } from '../../components/ui';
import {
  MapIcon,
  ArrowPathIcon,
  ShieldCheckIcon,
} from '@heroicons/react/24/outline';
import { useNotification } from '../../contexts/NotificationContext';
import LoadingState from '../../components/LoadingState';
import SitemapSettings from '../../settings/TechnicalSEO/Sitemaps/SitemapSettings';
import RedirectSettings from '../../settings/TechnicalSEO/Redirects/RedirectSettings';
import RobotsIndexingSettings from '../../settings/TechnicalSEO/RobotsIndexing/RobotsIndexingSettings';

/**
 * Technical SEO Page - Free Version
 * Only includes free modules: Sitemaps, Robots & Indexing, and Redirects
 */
const TechnicalSEO = () => {
  const [activeTab, setActiveTab] = useState('sitemaps');
  const [modules, setModules] = useState([]);
  const [loading, setLoading] = useState(true);
  const [updating, setUpdating] = useState({});
  const { showNotification } = useNotification();

  const loadModules = useCallback(async () => {
    setLoading(true);
    // Free modules only
    const freeModules = [
      {
        slug: 'sitemaps',
        name: __('Sitemaps', 'prorank-seo'),
        description: __(
          'Generate XML, HTML, Video, and News sitemaps to help search engines discover your content',
          'prorank-seo'
        ),
        icon: MapIcon,
        enabled: true,
        available: true,
      },
      {
        slug: 'robots-indexing',
        name: __('Robots & Indexing', 'prorank-seo'),
        description: __(
          'Control robots.txt rules and indexing signals for search engines',
          'prorank-seo'
        ),
        icon: ShieldCheckIcon,
        enabled: true,
        available: true,
      },
      {
        slug: 'redirects',
        name: __('Redirect Manager', 'prorank-seo'),
        description: __(
          'Manage 301, 302, and other redirects to preserve SEO value',
          'prorank-seo'
        ),
        icon: ArrowPathIcon,
        enabled: false,
        available: true,
      },
    ];

    setModules(freeModules);
    setLoading(false);
  }, []);

  // Set page title
  useEffect(() => {
    document.title =
      __('Technical SEO', 'prorank-seo') +
      ' - ' +
      __('ProRank SEO', 'prorank-seo') +
      ' - WordPress';
  }, []);

  // Load modules
  useEffect(() => {
    loadModules();
  }, [loadModules]);

  const toggleModule = async (module) => {
    const slug = module.slug;
    const newState = !module.enabled;
    setUpdating((prev) => ({ ...prev, [slug]: true }));
    try {
      await apiFetch({
        path: `/prorank-seo/v1/modules/${slug}/toggle`,
        method: 'POST',
        data: { enabled: newState },
      });

      setModules((prev) => prev.map((m) => (m.slug === slug ? { ...m, enabled: newState } : m)));
      showNotification(
        newState
          ? __('Module enabled successfully', 'prorank-seo')
          : __('Module disabled successfully', 'prorank-seo'),
        'success'
      );
    } catch (error) {
      showNotification(error.message || __('Failed to toggle module', 'prorank-seo'), 'error');
    } finally {
      setUpdating((prev) => ({ ...prev, [slug]: false }));
    }
  };

  const renderTabContent = () => {
    switch (activeTab) {
      case 'sitemaps':
        return <SitemapSettings />;
      case 'robots-indexing':
        return <RobotsIndexingSettings />;
      case 'redirects':
        return <RedirectSettings />;
      default:
        return (
          <Card>
            <div className="pr-text-center pr-py-20">
              <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-2">
                {__('Settings', 'prorank-seo')}
              </h3>
              <p className="pr-text-gray-600">
                {__('Settings for this module are coming soon.', 'prorank-seo')}
              </p>
            </div>
          </Card>
        );
    }
  };

  if (loading) {
    return <LoadingState message={__('Loading modules...', 'prorank-seo')} />;
  }

  return (
    <div className="pr-min-h-screen pr-bg-gray-50">
      {/* Header */}
      <PageHeader
        title={__('Technical SEO', 'prorank-seo')}
        subtitle={__(
          "Optimize your site's technical foundation for better search engine crawling and indexing",
          'prorank-seo'
        )}
        badges={headerBadges['technical-seo']}
      />

      {/* Main Content */}
      <div className="pr-p-8 pr-space-y-6">
        {/* Module Navigation */}
        <ModuleNav
          modules={modules.map((module) => ({
            id: module.slug,
            label: module.name,
            hasToggle: true,
            toggleValue: module.enabled,
            toggleDisabled: updating[module.slug],
          }))}
          activeModule={activeTab}
          onModuleChange={setActiveTab}
          onToggleChange={(moduleId, checked) => {
            const module = modules.find((item) => item.slug === moduleId);
            if (!module || module.enabled === checked) return;
            toggleModule(module);
          }}
        />

        {/* Tab Content */}
        <div>{renderTabContent()}</div>
      </div>
    </div>
  );
};

export default TechnicalSEO;
