import { useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import { Card, Button, Input } from '../../components/ui';
import { MagnifyingGlassIcon } from '@heroicons/react/24/outline';

const SitemapsDocumentation = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [activeTab, setActiveTab] = useState('overview');

  const handleSearch = () => {
    if (searchQuery.trim()) {
      const searchElement = document.querySelector(`[data-search*="${searchQuery.toLowerCase()}"]`);
      if (searchElement) {
        searchElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
        searchElement.style.backgroundColor = '#fff6e5';
        setTimeout(() => {
          searchElement.style.backgroundColor = '';
        }, 2000);
      }
    }
  };

  const documentationTabs = [
    { name: 'overview', title: __('Overview', 'prorank-seo') },
    { name: 'setup', title: __('Setup Guide', 'prorank-seo') },
    { name: 'features', title: __('Features', 'prorank-seo') },
    { name: 'advanced', title: __('Advanced', 'prorank-seo') },
    { name: 'troubleshooting', title: __('Troubleshooting', 'prorank-seo') },
  ];

  const renderTabContent = () => {
    switch (activeTab) {
      case 'overview':
        return <OverviewTab />;
      case 'setup':
        return <SetupTab />;
      case 'features':
        return <FeaturesTab />;
      case 'advanced':
        return <AdvancedTab />;
      case 'troubleshooting':
        return <TroubleshootingTab />;
      default:
        return null;
    }
  };

  return (
    <div className="pr-max-w-7xl pr-mx-auto pr-space-y-6">
      {/* Header */}
      <div className="pr-mb-6">
        <h2 className="pr-text-2xl pr-font-semibold pr-text-gray-900 pr-mb-2">
          {__('Sitemaps Documentation', 'prorank-seo')}
        </h2>
        <p className="pr-text-sm pr-text-gray-600">
          {__(
            'Complete guide to ProRank SEO sitemaps - Setup, configuration, and optimization for maximum search visibility',
            'prorank-seo'
          )}
        </p>
      </div>

      {/* Search */}
      <Card>
        <div className="pr-flex pr-items-center pr-gap-2">
          <Input
            type="text"
            placeholder={__('Search for topics...', 'prorank-seo')}
            value={searchQuery}
            onChange={(e) => setSearchQuery(e.target.value)}
            onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
            className="pr-flex-1"
          />
          <Button
            variant="primary"
            icon={MagnifyingGlassIcon}
            onClick={handleSearch}
            disabled={!searchQuery.trim()}
          >
            {__('Search', 'prorank-seo')}
          </Button>
        </div>
      </Card>

      {/* Tabs */}
      <Card>
        <div className="pr-border-b pr-border-gray-200">
          <nav className="pr-flex pr-gap-8" aria-label="Documentation tabs">
            {documentationTabs.map((tab) => (
              <button
                key={tab.name}
                onClick={() => setActiveTab(tab.name)}
                className={`pr-py-4 pr-px-1 pr-border-b-2 pr-font-medium pr-text-sm pr-transition-colors ${
                  activeTab === tab.name
                    ? 'pr-border-primary-500 pr-text-primary-600'
                    : 'pr-border-transparent pr-text-gray-500 hover:pr-text-gray-700 hover:pr-border-gray-300'
                }`}
              >
                {tab.title}
              </button>
            ))}
          </nav>
        </div>
      </Card>

      {/* Tab Content */}
      <div className="pr-mt-6">{renderTabContent()}</div>
    </div>
  );
};

// Overview Tab Component
const OverviewTab = () => (
  <div className="pr-space-y-6">
    <Card data-search="overview introduction sitemaps">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">
        {__('ProRank SEO Sitemaps - Complete Solution', 'prorank-seo')}
      </h3>
      <p className="pr-text-sm pr-text-gray-600 pr-mb-6">
        {__(
          "ProRank SEO offers the most comprehensive sitemap solution for WordPress, fully compliant with Google's 2025 requirements. Our sitemaps help search engines discover and index your content efficiently.",
          'prorank-seo'
        )}
      </p>

      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-2 lg:pr-grid-cols-3 pr-gap-4">
        {[
          { title: __('✅ XML Sitemaps', 'prorank-seo'), desc: __('Automatic generation for all content', 'prorank-seo') },
          { title: __('✅ Video Sitemaps', 'prorank-seo'), desc: __('YouTube Shorts support (2025)', 'prorank-seo') },
          { title: __('✅ News Sitemaps', 'prorank-seo'), desc: __('Google News compliance', 'prorank-seo') },
          { title: __('✅ HTML Sitemaps', 'prorank-seo'), desc: __('User-friendly navigation', 'prorank-seo') },
          { title: __('✅ AI Content Labels', 'prorank-seo'), desc: __('2025 transparency requirements', 'prorank-seo') },
          { title: __('✅ API Caching', 'prorank-seo'), desc: __('Optimize YouTube quota', 'prorank-seo') },
        ].map((feature, index) => (
          <div key={index} className="pr-p-4 pr-bg-gray-50 pr-rounded-lg pr-border pr-border-gray-200">
            <strong className="pr-text-sm pr-font-semibold pr-text-gray-900 pr-block pr-mb-1">{feature.title}</strong>
            <p className="pr-text-xs pr-text-gray-600">{feature.desc}</p>
          </div>
        ))}
      </div>
    </Card>

    <Card data-search="comparison features rankmath yoast">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">{__('Feature Comparison', 'prorank-seo')}</h3>

      <div className="pr-overflow-x-auto">
        <table className="pr-w-full pr-text-sm">
          <thead>
            <tr className="pr-border-b pr-border-gray-200">
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('Feature', 'prorank-seo')}</th>
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('ProRank SEO', 'prorank-seo')}</th>
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('RankMath', 'prorank-seo')}</th>
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('Yoast', 'prorank-seo')}</th>
            </tr>
          </thead>
          <tbody className="pr-divide-y pr-divide-gray-200">
            {[
              [__('XML Sitemaps', 'prorank-seo'), '✅', '✅', '✅'],
              [__('Video Sitemaps', 'prorank-seo'), '✅', '✅', '⚠️ ' + __('Limited', 'prorank-seo')],
              [__('News Sitemaps', 'prorank-seo'), '✅', '✅', '⚠️ ' + __('Limited', 'prorank-seo')],
              [__('HTML Sitemaps', 'prorank-seo'), '✅', '✅', '❌'],
              [__('YouTube Shorts Support', 'prorank-seo'), '✅ ' + __('2025', 'prorank-seo'), '❌', '❌'],
              [__('AI Content Labeling', 'prorank-seo'), '✅ ' + __('2025', 'prorank-seo'), '❌', '❌'],
              [__('API Quota Caching', 'prorank-seo'), '✅', '❌', '❌'],
              [__('Custom Fields Scanning', 'prorank-seo'), '✅', '⚠️ ' + __('Limited', 'prorank-seo'), '❌'],
            ].map((row, index) => (
              <tr key={index}>
                <td className="pr-py-3 pr-text-gray-700">{row[0]}</td>
                <td className="pr-py-3 pr-text-gray-700">{row[1]}</td>
                <td className="pr-py-3 pr-text-gray-700">{row[2]}</td>
                <td className="pr-py-3 pr-text-gray-700">{row[3]}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </Card>

    <Card data-search="urls structure sitemap index">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">
        {__('Sitemap URLs Structure', 'prorank-seo')}
      </h3>

      <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4">
        <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
          <code>
            {`yoursite.com/
├── sitemap_index.xml          # Main index file
├── post-sitemap.xml           # Blog posts
├── page-sitemap.xml           # Static pages
├── category-sitemap.xml       # Categories
├── post_tag-sitemap.xml       # Tags
├── author-sitemap.xml         # Author archives
├── video-sitemap.xml          # Video content (if enabled)
└── news-sitemap.xml           # News articles (if enabled)`}
          </code>
        </pre>
      </div>
    </Card>
  </div>
);

// Setup Tab Component
const SetupTab = () => (
  <div className="pr-space-y-6">
    <Card data-search="quick setup enable sitemaps">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('Quick Setup Steps', 'prorank-seo')}</h3>

      <div className="pr-space-y-6">
        {[
          {
            number: '1️⃣',
            title: __('Enable Sitemaps', 'prorank-seo'),
            desc: __('Navigate to ProRank SEO → Technical SEO → Sitemaps → XML Sitemap → Enable', 'prorank-seo'),
          },
          {
            number: '2️⃣',
            title: __('Configure Content', 'prorank-seo'),
            desc: __('Go to Content Types Tab → Select Post Types & Taxonomies you want to include', 'prorank-seo'),
          },
          {
            number: '3️⃣',
            title: __('Submit to Google', 'prorank-seo'),
            desc: __('Open Google Search Console → Sitemaps → Add: sitemap_index.xml', 'prorank-seo'),
          },
        ].map((step) => (
          <div key={step.number} className="pr-flex pr-gap-4">
            <div className="pr-flex-shrink-0 pr-text-2xl">{step.number}</div>
            <div>
              <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-1">{step.title}</h4>
              <p className="pr-text-sm pr-text-gray-600">{step.desc}</p>
            </div>
          </div>
        ))}
      </div>
    </Card>

    <Card data-search="content types posts pages categories">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">
        {__('Content Types Configuration', 'prorank-seo')}
      </h3>

      <div className="pr-space-y-4">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Post Types Selection', 'prorank-seo')}</h4>
          <ul className="pr-list-disc pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>{__('Posts - Your blog posts', 'prorank-seo')}</li>
            <li>{__('Pages - Static pages', 'prorank-seo')}</li>
            <li>{__('Custom Post Types - Any registered custom types', 'prorank-seo')}</li>
          </ul>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Taxonomies', 'prorank-seo')}</h4>
          <ul className="pr-list-disc pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>{__('Categories - Post categories', 'prorank-seo')}</li>
            <li>{__('Tags - Post tags', 'prorank-seo')}</li>
            <li>{__('Custom Taxonomies - Any registered custom taxonomies', 'prorank-seo')}</li>
          </ul>
        </div>

        <div className="pr-bg-gray-50 pr-p-4 pr-rounded-lg pr-border pr-border-gray-200 pr-mt-4">
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Exclusion Settings', 'prorank-seo')}</h4>
          <p className="pr-text-sm pr-text-gray-600 pr-mb-1">
            <strong>{__('Exclude Posts:', 'prorank-seo')}</strong> {__('Enter comma-separated post IDs (e.g., 123, 456, 789)', 'prorank-seo')}
          </p>
          <p className="pr-text-sm pr-text-gray-600">
            <strong>{__('Exclude Terms:', 'prorank-seo')}</strong> {__('Enter comma-separated term IDs (e.g., 12, 34, 56)', 'prorank-seo')}
          </p>
        </div>
      </div>
    </Card>

    <Card data-search="submit search console bing webmaster">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">
        {__('Submit to Search Engines', 'prorank-seo')}
      </h3>

      <div className="pr-space-y-6">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-3">{__('Google Search Console', 'prorank-seo')}</h4>
          <ol className="pr-list-decimal pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>
              {__('Go to', 'prorank-seo')}{' '}
              <a
                href="https://search.google.com/search-console"
                target="_blank"
                rel="noopener noreferrer"
                className="pr-text-primary-500 pr-underline hover:pr-text-primary-700"
              >
                {__('Google Search Console', 'prorank-seo')}
              </a>
            </li>
            <li>{__('Select your property', 'prorank-seo')}</li>
            <li>{__('Click Sitemaps in left menu', 'prorank-seo')}</li>
            <li>{__('Enter: sitemap_index.xml', 'prorank-seo')}</li>
            <li>{__('Click Submit', 'prorank-seo')}</li>
          </ol>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-3">{__('Bing Webmaster Tools', 'prorank-seo')}</h4>
          <ol className="pr-list-decimal pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>
              {__('Go to', 'prorank-seo')}{' '}
              <a
                href="https://www.bing.com/webmasters"
                target="_blank"
                rel="noopener noreferrer"
                className="pr-text-primary-500 pr-underline hover:pr-text-primary-700"
              >
                {__('Bing Webmaster Tools', 'prorank-seo')}
              </a>
            </li>
            <li>{__('Select Sitemaps from menu', 'prorank-seo')}</li>
            <li>{__('Submit same URLs as Google', 'prorank-seo')}</li>
          </ol>
        </div>

        <div className="pr-bg-info-50 pr-border pr-border-info-200 pr-rounded-lg pr-p-4">
          <p className="pr-text-sm pr-text-info-800">
            {__(
              'ProRank automatically pings search engines when content is updated if "Ping search engines" is enabled.',
              'prorank-seo'
            )}
          </p>
        </div>
      </div>
    </Card>
  </div>
);

// Features Tab Component
const FeaturesTab = () => (
  <div className="pr-space-y-6">
    <Card data-search="video sitemap youtube shorts api">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">
        {__('Video Sitemaps (2025 Enhanced)', 'prorank-seo')}
      </h3>

      <div className="pr-space-y-4">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('New 2025 Features', 'prorank-seo')}</h4>
          <ul className="pr-space-y-2 pr-text-sm pr-text-gray-600">
            <li>
              <strong>{__('YouTube Shorts Detection:', 'prorank-seo')}</strong>{' '}
              {__('Automatically identifies Shorts (≤60 seconds)', 'prorank-seo')}
            </li>
            <li>
              <strong>{__('View Count Types:', 'prorank-seo')}</strong>{' '}
              {__('Supports new "starts and replays" metric', 'prorank-seo')}
            </li>
            <li>
              <strong>{__('AI Content Labels:', 'prorank-seo')}</strong> {__('Mark synthetic/AI-generated videos', 'prorank-seo')}
            </li>
            <li>
              <strong>{__('API Caching:', 'prorank-seo')}</strong>{' '}
              {__('15-minute cache to optimize 10,000 daily quota', 'prorank-seo')}
            </li>
          </ul>
        </div>

        <div className="pr-bg-gray-50 pr-p-4 pr-rounded-lg pr-border pr-border-gray-200">
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-3">{__('Configuration', 'prorank-seo')}</h4>
          <div className="pr-space-y-2 pr-text-sm pr-text-gray-600">
            <p>
              <strong>{__('Enable Video Sitemap:', 'prorank-seo')}</strong> ON
            </p>
            <p>
              <strong>{__('YouTube API Key:', 'prorank-seo')}</strong> {__('Add your API key for enhanced data', 'prorank-seo')}
            </p>
            <p>
              <strong>{__('Custom Fields:', 'prorank-seo')}</strong> video_url, video_embed, custom_field
            </p>
            <p>
              <strong>{__('Detect Shorts:', 'prorank-seo')}</strong> ON {__('(2025)', 'prorank-seo')}
            </p>
            <p>
              <strong>{__('Enable Caching:', 'prorank-seo')}</strong> ON {__('(2025)', 'prorank-seo')}
            </p>
            <p>
              <strong>{__('Cache Duration:', 'prorank-seo')}</strong> 15 {__('minutes', 'prorank-seo')}
            </p>
          </div>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Getting YouTube API Key', 'prorank-seo')}</h4>
          <ol className="pr-list-decimal pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>
              {__('Visit', 'prorank-seo')}{' '}
              <a
                href="https://console.cloud.google.com"
                target="_blank"
                rel="noopener noreferrer"
                className="pr-text-primary-500 pr-underline hover:pr-text-primary-700"
              >
                {__('Google Cloud Console', 'prorank-seo')}
              </a>
            </li>
            <li>{__('Create new project or select existing', 'prorank-seo')}</li>
            <li>{__('Enable YouTube Data API v3', 'prorank-seo')}</li>
            <li>{__('Create credentials (API Key)', 'prorank-seo')}</li>
            <li>{__('Enter key in ProRank settings', 'prorank-seo')}</li>
          </ol>
        </div>
      </div>
    </Card>

    <Card data-search="news sitemap google news publication">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">
        {__('News Sitemaps (Google News)', 'prorank-seo')}
      </h3>

      <div className="pr-space-y-4">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Key Requirements', 'prorank-seo')}</h4>
          <ul className="pr-space-y-1 pr-text-sm pr-text-gray-600">
            <li>✅ {__('Only includes last 2 days of content', 'prorank-seo')}</li>
            <li>✅ {__('Maximum 1,000 URLs', 'prorank-seo')}</li>
            <li>✅ {__('Auto-removes site name from titles', 'prorank-seo')}</li>
            <li>✅ {__('No keywords/stock tickers (deprecated)', 'prorank-seo')}</li>
          </ul>
        </div>

        <div className="pr-bg-gray-50 pr-p-4 pr-rounded-lg pr-border pr-border-gray-200">
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-3">{__('Configuration', 'prorank-seo')}</h4>
          <div className="pr-space-y-2 pr-text-sm pr-text-gray-600">
            <p>
              <strong>{__('Enable News Sitemap:', 'prorank-seo')}</strong> ON
            </p>
            <p>
              <strong>{__('Publication Name:', 'prorank-seo')}</strong> {__('Your Site Name', 'prorank-seo')}
            </p>
            <p>
              <strong>{__('Language:', 'prorank-seo')}</strong> en
            </p>
            <p>
              <strong>{__('Post Types:', 'prorank-seo')}</strong> [Posts]
            </p>
            <p>
              <strong>{__('Filter Categories:', 'prorank-seo')}</strong> {__('[Optional]', 'prorank-seo')}
            </p>
            <p>
              <strong>{__('Exclude Terms:', 'prorank-seo')}</strong> {__('[Excluded IDs]', 'prorank-seo')}
            </p>
          </div>
        </div>

        <div className="pr-bg-warning-50 pr-border pr-border-warning-200 pr-rounded-lg pr-p-4">
          <p className="pr-text-sm pr-text-warning-800">
            {__(
              'Remember: News sitemaps only show content from the last 2 days. This is normal and required by Google.',
              'prorank-seo'
            )}
          </p>
        </div>
      </div>
    </Card>

    <Card data-search="html sitemap shortcode user friendly">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">{__('HTML Sitemaps', 'prorank-seo')}</h3>

      <div className="pr-space-y-4">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Shortcode Usage', 'prorank-seo')}</h4>
          <div className="pr-bg-gray-900 pr-rounded-lg pr-p-3">
            <code className="pr-text-white pr-text-sm">[prorank_html_sitemap]</code>
          </div>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Display Options', 'prorank-seo')}</h4>
          <ul className="pr-list-disc pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>{__('Show Post Types', 'prorank-seo')}</li>
            <li>{__('Show Taxonomies', 'prorank-seo')}</li>
            <li>{__('Posts per Category: 10', 'prorank-seo')}</li>
            <li>{__('Sort Order: Date/Title/Modified', 'prorank-seo')}</li>
            <li>{__('Custom CSS Support', 'prorank-seo')}</li>
          </ul>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Custom CSS Example', 'prorank-seo')}</h4>
          <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4">
            <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
              <code>
                {`.prorank-html-sitemap {
    columns: 2;
    column-gap: 30px;
}
.prorank-html-sitemap h2 {
    color: #2271b1;
    break-after: avoid;
}`}
              </code>
            </pre>
          </div>
        </div>
      </div>
    </Card>
  </div>
);

// Advanced Tab Component
const AdvancedTab = () => (
  <div className="pr-space-y-6">
    <Card data-search="developer hooks filters actions">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('Developer Hooks', 'prorank-seo')}</h3>

      <div className="pr-space-y-6">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Add Custom Sitemap Provider', 'prorank-seo')}</h4>
          <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4">
            <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
              <code>
                {`add_filter('prorank_seo_sitemap_providers', function($providers) {
    $providers[] = new CustomSitemapProvider();
    return $providers;
});`}
              </code>
            </pre>
          </div>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Modify URL Data', 'prorank-seo')}</h4>
          <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4">
            <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
              <code>
                {`add_filter('prorank_seo_sitemap_url_data', function($url_data, $post, $type) {
    // Add custom data
    $url_data['custom_field'] = 'value';
    return $url_data;
}, 10, 3);`}
              </code>
            </pre>
          </div>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Exclude Content Programmatically', 'prorank-seo')}</h4>
          <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4">
            <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
              <code>
                {`add_filter('prorank_seo_exclude_from_sitemap', function($exclude, $post_id) {
    if (get_post_meta($post_id, 'exclude_from_sitemap', true)) {
        return true;
    }
    return $exclude;
}, 10, 2);`}
              </code>
            </pre>
          </div>
        </div>
      </div>
    </Card>

    <Card data-search="performance optimization large sites">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">
        {__('Performance Optimization', 'prorank-seo')}
      </h3>

      <div className="pr-space-y-6">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('For Large Sites (10,000+ posts)', 'prorank-seo')}</h4>
          <ul className="pr-list-disc pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>{__('Reduce posts per sitemap to 500-1000', 'prorank-seo')}</li>
            <li>{__('Enable selective post types', 'prorank-seo')}</li>
            <li>{__('Use exclusion rules effectively', 'prorank-seo')}</li>
            <li>{__('Consider separate news/video sitemaps', 'prorank-seo')}</li>
          </ul>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-3">{__('Caching Strategy', 'prorank-seo')}</h4>
          <div className="pr-overflow-x-auto">
            <table className="pr-w-full pr-text-sm">
              <thead>
                <tr className="pr-border-b pr-border-gray-200">
                  <th className="pr-text-left pr-pb-2 pr-font-semibold pr-text-gray-900">
                    {__('Sitemap Type', 'prorank-seo')}
                  </th>
                  <th className="pr-text-left pr-pb-2 pr-font-semibold pr-text-gray-900">
                    {__('Cache Duration', 'prorank-seo')}
                  </th>
                </tr>
              </thead>
              <tbody className="pr-divide-y pr-divide-gray-200">
                <tr>
                  <td className="pr-py-2 pr-text-gray-700">{__('Regular Sitemaps', 'prorank-seo')}</td>
                  <td className="pr-py-2 pr-text-gray-700">
                    12 {__('hours', 'prorank-seo')}
                  </td>
                </tr>
                <tr>
                  <td className="pr-py-2 pr-text-gray-700">{__('News Sitemaps', 'prorank-seo')}</td>
                  <td className="pr-py-2 pr-text-gray-700">
                    1 {__('hour', 'prorank-seo')}
                  </td>
                </tr>
                <tr>
                  <td className="pr-py-2 pr-text-gray-700">{__('Video Sitemaps', 'prorank-seo')}</td>
                  <td className="pr-py-2 pr-text-gray-700">
                    6 {__('hours', 'prorank-seo')}
                  </td>
                </tr>
              </tbody>
            </table>
          </div>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Server Requirements', 'prorank-seo')}</h4>
          <ul className="pr-list-disc pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
            <li>
              PHP 7.4+ {__('recommended', 'prorank-seo')}
            </li>
            <li>
              {__('Memory limit:', 'prorank-seo')} 256MB {__('minimum', 'prorank-seo')}
            </li>
            <li>
              {__('Max execution time:', 'prorank-seo')} 300 {__('seconds', 'prorank-seo')}
            </li>
            <li>
              MySQL 5.6+ {__('or', 'prorank-seo')} MariaDB 10.0+
            </li>
          </ul>
        </div>
      </div>
    </Card>

    <Card data-search="2025 compliance google requirements">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('2025 Google Compliance', 'prorank-seo')}</h3>

      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-3 pr-gap-6">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Video Sitemaps', 'prorank-seo')}</h4>
          <ul className="pr-space-y-1 pr-text-sm pr-text-gray-600">
            <li>✅ {__('YouTube Shorts detection', 'prorank-seo')}</li>
            <li>✅ {__('New view counting metrics', 'prorank-seo')}</li>
            <li>✅ {__('AI/Synthetic content labels', 'prorank-seo')}</li>
            <li>✅ {__('API quota optimization', 'prorank-seo')}</li>
          </ul>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('News Sitemaps', 'prorank-seo')}</h4>
          <ul className="pr-space-y-1 pr-text-sm pr-text-gray-600">
            <li>✅ {__('2-day window enforcement', 'prorank-seo')}</li>
            <li>✅ {__('Site name auto-removal', 'prorank-seo')}</li>
            <li>✅ {__('Deprecated fields removed', 'prorank-seo')}</li>
            <li>✅ {__('1,000 URL limit', 'prorank-seo')}</li>
          </ul>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('General Updates', 'prorank-seo')}</h4>
          <ul className="pr-space-y-1 pr-text-sm pr-text-gray-600">
            <li>✅ {__('Lastmod prioritized (changefreq ignored)', 'prorank-seo')}</li>
            <li>✅ {__('Mobile-first indexing ready', 'prorank-seo')}</li>
            <li>✅ {__('Core Web Vitals friendly', 'prorank-seo')}</li>
            <li>✅ {__('UTF-8 encoding', 'prorank-seo')}</li>
          </ul>
        </div>
      </div>
    </Card>
  </div>
);

// Troubleshooting Tab Component
const TroubleshootingTab = () => (
  <div className="pr-space-y-6">
    <Card data-search="common issues problems solutions">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">
        {__('Common Issues and Solutions', 'prorank-seo')}
      </h3>

      <div className="pr-overflow-x-auto">
        <table className="pr-w-full pr-text-sm">
          <thead>
            <tr className="pr-border-b pr-border-gray-200">
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('Issue', 'prorank-seo')}</th>
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('Solution', 'prorank-seo')}</th>
            </tr>
          </thead>
          <tbody className="pr-divide-y pr-divide-gray-200">
            {[
              [__('404 Error', 'prorank-seo'), __('Settings → Permalinks → Save', 'prorank-seo')],
              [__('Not Updating', 'prorank-seo'), __('Clear all caches (plugin, server, CDN)', 'prorank-seo')],
              [
                __('Missing Content', 'prorank-seo'),
                __('Check: Published, Not noindex, Not excluded', 'prorank-seo'),
              ],
              [__('Empty Video Sitemap', 'prorank-seo'), __('Verify videos exist, API key valid', 'prorank-seo')],
              [__('Old News Content', 'prorank-seo'), __('Normal - only shows 2 days', 'prorank-seo')],
            ].map((row, index) => (
              <tr key={index}>
                <td className="pr-py-3 pr-font-medium pr-text-gray-900">{row[0]}</td>
                <td className="pr-py-3 pr-text-gray-600">{row[1]}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </Card>

    <Card data-search="validation tools testing">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('Validation Tools', 'prorank-seo')}</h3>

      <div className="pr-space-y-4">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Test Your Sitemaps', 'prorank-seo')}</h4>

          <div className="pr-space-y-3">
            <div>
              <strong className="pr-text-gray-900 pr-text-sm pr-block pr-mb-1">
                {__('Google Search Console', 'prorank-seo')}
              </strong>
              <ul className="pr-list-disc pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
                <li>{__('Submit sitemap URL', 'prorank-seo')}</li>
                <li>{__('Check for errors/warnings', 'prorank-seo')}</li>
                <li>{__('Monitor indexing status', 'prorank-seo')}</li>
              </ul>
            </div>

            <div>
              <strong className="pr-text-gray-900 pr-text-sm pr-block pr-mb-1">
                {__('External Validators', 'prorank-seo')}
              </strong>
              <ul className="pr-list-disc pr-list-inside pr-text-sm pr-text-gray-600 pr-space-y-1">
                <li>
                  <a
                    href="https://www.xml-sitemaps.com/validate-xml-sitemap.html"
                    target="_blank"
                    rel="noopener noreferrer"
                    className="pr-text-primary-500 pr-underline hover:pr-text-primary-700"
                  >
                    {__('XML Sitemaps Validator', 'prorank-seo')}
                  </a>
                </li>
                <li>
                  <a
                    href="https://search.google.com/test/rich-results"
                    target="_blank"
                    rel="noopener noreferrer"
                    className="pr-text-primary-500 pr-underline hover:pr-text-primary-700"
                  >
                    {__('Google Rich Results Test', 'prorank-seo')}
                  </a>
                </li>
              </ul>
            </div>
          </div>
        </div>

        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Health Checks', 'prorank-seo')}</h4>
          <ul className="pr-space-y-1 pr-text-sm pr-text-gray-600">
            <li>✓ {__('Sitemap accessible (200 status)', 'prorank-seo')}</li>
            <li>✓ {__('Valid XML format', 'prorank-seo')}</li>
            <li>✓ {__('Under size limits', 'prorank-seo')}</li>
            <li>✓ {__('All URLs return 200', 'prorank-seo')}</li>
            <li>✓ {__('Lastmod dates accurate', 'prorank-seo')}</li>
          </ul>
        </div>
      </div>
    </Card>

    <Card data-search="debug mode logging">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">{__('Debug Mode', 'prorank-seo')}</h3>
      <p className="pr-text-sm pr-text-gray-600 pr-mb-4">
        {__('Add to wp-config.php for detailed logging:', 'prorank-seo')}
      </p>
      <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4">
        <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
          <code>
            {`// Add to wp-config.php
define('PRORANK_SEO_DEBUG', true);
define('PRORANK_SEO_SITEMAP_DEBUG', true);`}
          </code>
        </pre>
      </div>
    </Card>

    <Card data-search="best practices dos donts">
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('Best Practices', 'prorank-seo')}</h3>

      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-2 pr-gap-6">
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-3">✅ {__("DO's", 'prorank-seo')}</h4>
          <ul className="pr-space-y-1 pr-text-sm pr-text-gray-600">
            <li>{__('Submit all sitemaps to Search Console', 'prorank-seo')}</li>
            <li>{__('Keep under 5,000 URLs per sitemap', 'prorank-seo')}</li>
            <li>{__('Update content regularly', 'prorank-seo')}</li>
            <li>{__('Use exclusion rules wisely', 'prorank-seo')}</li>
            <li>{__('Monitor indexing weekly', 'prorank-seo')}</li>
            <li>{__('Label AI content appropriately (2025)', 'prorank-seo')}</li>
          </ul>
        </div>
        <div>
          <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-3">❌ {__("DON'Ts", 'prorank-seo')}</h4>
          <ul className="pr-space-y-1 pr-text-sm pr-text-gray-600">
            <li>{__("Don't include noindex pages", 'prorank-seo')}</li>
            <li>{__("Don't exceed 50MB/50,000 limits", 'prorank-seo')}</li>
            <li>{__("Don't edit XML files manually", 'prorank-seo')}</li>
            <li>{__("Don't include private content", 'prorank-seo')}</li>
            <li>{__("Don't disable auto-ping", 'prorank-seo')}</li>
            <li>{__("Don't ignore Search Console errors", 'prorank-seo')}</li>
          </ul>
        </div>
      </div>
    </Card>
  </div>
);

export default SitemapsDocumentation;
