import { useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import { Card, Button, Input, Alert, Toggle } from '../../components/ui';
import {
  MagnifyingGlassIcon,
  InformationCircleIcon,
  Cog6ToothIcon,
  QuestionMarkCircleIcon,
  ShieldCheckIcon,
} from '@heroicons/react/24/outline';

const RobotsIndexingDocumentation = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [checkedItems, setCheckedItems] = useState({});
  const [activeTab, setActiveTab] = useState('overview');

  const handleSearch = () => {
    if (searchQuery.trim()) {
      const searchElement = document.querySelector(`[data-search*="${searchQuery.toLowerCase()}"]`);
      if (searchElement) {
        searchElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
        searchElement.style.backgroundColor = '#fff6e5';
        setTimeout(() => {
          searchElement.style.backgroundColor = '';
        }, 2000);
      }
    }
  };

  const handleCheckboxChange = (itemId) => {
    setCheckedItems(prev => ({
      ...prev,
      [itemId]: !prev[itemId]
    }));
  };

  const documentationTabs = [
    {
      name: 'overview',
      title: __('Overview', 'prorank-seo'),
      className: 'tab-overview',
    },
    {
      name: 'quickstart',
      title: __('Quick Start', 'prorank-seo'),
      className: 'tab-quickstart',
    },
    {
      name: 'features',
      title: __('Features Guide', 'prorank-seo'),
      className: 'tab-features',
    },
    {
      name: 'aiprotection',
      title: __('AI Protection', 'prorank-seo'),
      className: 'tab-aiprotection',
    },
    {
      name: 'bestpractices',
      title: __('Best Practices', 'prorank-seo'),
      className: 'tab-bestpractices',
    },
    {
      name: 'troubleshooting',
      title: __('Troubleshooting', 'prorank-seo'),
      className: 'tab-troubleshooting',
    },
  ];

  const renderOverviewTab = () => (
    <div className="pr-space-y-6">
      <Card className="pr-mb-6" data-search="overview robots indexing what is">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            🤖 {__('What is Robots & Indexing?', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <p className="pr-text-sm pr-text-gray-600">{__('Think of Robots & Indexing as your website\'s security guard and tour guide for search engines. It controls:', 'prorank-seo')}</p>
          
          <div className="pr-mt-5">
            <div className="pr-mb-2.5">
              <strong>✅ {__('What Search Engines Can See', 'prorank-seo')}</strong>
              <p>{__('Tell Google which pages to show in search results', 'prorank-seo')}</p>
            </div>
            <div className="pr-mb-2.5">
              <strong>❌ {__('What to Keep Private', 'prorank-seo')}</strong>
              <p>{__('Hide admin pages, thank you pages, or private content', 'prorank-seo')}</p>
            </div>
            <div className="pr-mb-2.5">
              <strong>🚀 {__('Faster Indexing', 'prorank-seo')}</strong>
              <p>{__('Get your content into search results in hours, not days', 'prorank-seo')}</p>
            </div>
            <div className="pr-mb-2.5">
              <strong>🛡️ {__('AI Protection', 'prorank-seo')}</strong>
              <p>{__('Block AI bots from using your content for training', 'prorank-seo')}</p>
            </div>
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="features included">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Included Features', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <ul>
            <li>{__('Control what search engines see', 'prorank-seo')}</li>
            <li>{__('AI protection with meta tags', 'prorank-seo')}</li>
            <li>{__('Hide specific pages from Google', 'prorank-seo')}</li>
            <li>{__('Global noindex settings', 'prorank-seo')}</li>
            <li>{__('Block 50+ AI bots completely', 'prorank-seo')}</li>
            <li>{__('Instant indexing with IndexNow', 'prorank-seo')}</li>
            <li>{__('Custom robots.txt editor', 'prorank-seo')}</li>
          </ul>
        </div>
      </Card>
    </div>
  );

  const renderQuickStartTab = () => (
    <div className="pr-space-y-6">
      <Card className="pr-mb-6" data-search="quick start setup checklist">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Quick Setup Checklist', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <h4>{__('For New Sites:', 'prorank-seo')}</h4>
          <div className="pr-mb-5 pr-space-y-3">
            <Toggle
              label={__('Enable robots.txt editor', 'prorank-seo')}
              checked={checkedItems['new1'] || false}
              onChange={() => handleCheckboxChange('new1')}
            />
            <Toggle
              label={__('Block AI bots (optional)', 'prorank-seo')}
              checked={checkedItems['new2'] || false}
              onChange={() => handleCheckboxChange('new2')}
            />
            <Toggle
              label={__('Enable IndexNow for fast indexing', 'prorank-seo')}
              checked={checkedItems['new3'] || false}
              onChange={() => handleCheckboxChange('new3')}
            />
            <Toggle
              label={__('Noindex search/404/attachment pages', 'prorank-seo')}
              checked={checkedItems['new4'] || false}
              onChange={() => handleCheckboxChange('new4')}
            />
            <Toggle
              label={__('Add AI protection meta tags', 'prorank-seo')}
              checked={checkedItems['new5'] || false}
              onChange={() => handleCheckboxChange('new5')}
            />
            <Toggle
              label={__('Save settings and test robots.txt', 'prorank-seo')}
              checked={checkedItems['new6'] || false}
              onChange={() => handleCheckboxChange('new6')}
            />
          </div>

          <h4>{__('For Existing Sites:', 'prorank-seo')}</h4>
          <div className="pr-space-y-3">
            <Toggle
              label={__('Review current robots.txt', 'prorank-seo')}
              checked={checkedItems['exist1'] || false}
              onChange={() => handleCheckboxChange('exist1')}
            />
            <Toggle
              label={__('Check current noindex settings', 'prorank-seo')}
              checked={checkedItems['exist2'] || false}
              onChange={() => handleCheckboxChange('exist2')}
            />
            <Toggle
              label={__('Enable IndexNow for faster updates', 'prorank-seo')}
              checked={checkedItems['exist3'] || false}
              onChange={() => handleCheckboxChange('exist3')}
            />
            <Toggle
              label={__('Add AI protection if needed', 'prorank-seo')}
              checked={checkedItems['exist4'] || false}
              onChange={() => handleCheckboxChange('exist4')}
            />
            <Toggle
              label={__('Monitor for 1 week', 'prorank-seo')}
              checked={checkedItems['exist5'] || false}
              onChange={() => handleCheckboxChange('exist5')}
            />
            <Toggle
              label={__('Adjust based on results', 'prorank-seo')}
              checked={checkedItems['exist6'] || false}
              onChange={() => handleCheckboxChange('exist6')}
            />
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="step by step access settings">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Step-by-Step Setup', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <ol className="pr-leading-relaxed">
            <li>{__('Go to ProRank SEO in your WordPress admin', 'prorank-seo')}</li>
            <li>{__('Click Technical SEO', 'prorank-seo')}</li>
            <li>{__('Click Robots & Indexing tab', 'prorank-seo')}</li>
            <li>{__('Make sure the module toggle is ON (green)', 'prorank-seo')}</li>
            <li>{__('Configure each tab based on your needs', 'prorank-seo')}</li>
            <li>{__('Save your settings', 'prorank-seo')}</li>
            <li>{__('Test at yoursite.com/robots.txt', 'prorank-seo')}</li>
          </ol>
        </div>
      </Card>
    </div>
  );

  const renderFeaturesTab = () => (
    <div className="pr-space-y-6">
      <Card className="pr-mb-6" data-search="robots txt editor virtual">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Robots.txt Editor', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <Alert variant="info">
            <strong>{__('What is robots.txt?', 'prorank-seo')}</strong><br />
            {__('It\'s like a "Do Not Enter" sign for parts of your website. Search engines check this file first before visiting your site.', 'prorank-seo')}
          </Alert>

          <div className="pr-mt-5">
            <h4>{__('Key Settings:', 'prorank-seo')}</h4>
            
            <div className="pr-mb-4 pr-p-2.5 pr-bg-gray-100 pr-rounded">
              <strong>{__('Enable Virtual Robots.txt Editor', 'prorank-seo')}</strong>
              <p>{__('Recommendation: ✅ Always enable unless you have a physical file', 'prorank-seo')}</p>
            </div>

            <div className="pr-mb-4 pr-p-2.5 pr-bg-gray-100 pr-rounded">
              <strong>{__('Block AI/ML Training Bots', 'prorank-seo')}</strong>
              <p>{__('Blocks 50+ AI bots including ChatGPT, Claude, Gemini, MidJourney', 'prorank-seo')}</p>
              <p>{__('Recommendation: ✅ Enable for content protection', 'prorank-seo')}</p>
            </div>

            <div className="pr-p-2.5 pr-bg-gray-100 pr-rounded">
              <strong>{__('Custom Rules', 'prorank-seo')}</strong>
              <p>{__('Add specific blocking rules if needed', 'prorank-seo')}</p>
              <pre className="pr-bg-white pr-p-2 pr-rounded-sm">
{`User-agent: *
Disallow: /private/
Disallow: /tmp/`}
              </pre>
            </div>
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="indexnow api instant indexing">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('IndexNow - Instant Indexing', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <Alert variant="success">
            <strong>{__('What is IndexNow?', 'prorank-seo')}</strong><br />
            {__('It\'s like texting search engines instead of waiting for them to call. Your updates appear in hours instead of days!', 'prorank-seo')}
          </Alert>

          <div className="pr-mt-5">
            <h4>{__('Setup Steps:', 'prorank-seo')}</h4>
            <ol>
              <li>{__('Toggle "Enable IndexNow" to ON', 'prorank-seo')}</li>
              <li>{__('Get your API key from', 'prorank-seo')} <a href="https://www.indexnow.org/keys" target="_blank" rel="noopener noreferrer" className="pr-text-primary-600 hover:pr-text-primary-700 pr-underline">{__('IndexNow.org', 'prorank-seo')}</a></li>
              <li>{__('Paste the key in the field', 'prorank-seo')}</li>
              <li>{__('Click "Validate Key"', 'prorank-seo')}</li>
              <li>{__('Save settings', 'prorank-seo')}</li>
            </ol>

            <h4>{__('Benefits:', 'prorank-seo')}</h4>
            <ul>
              <li>⚡ {__('Pages indexed within minutes', 'prorank-seo')}</li>
              <li>🌍 {__('Works with Bing, Yandex, Seznam', 'prorank-seo')}</li>
              <li>🔄 {__('Automatic on every publish/update', 'prorank-seo')}</li>
              <li>💰 {__('Free to use', 'prorank-seo')}</li>
            </ul>
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="noindex global settings">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Global Noindex Settings', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <Alert variant="warning">
            <strong>{__('What is Noindex?', 'prorank-seo')}</strong><br />
            {__('It tells search engines "You can look, but don\'t show this in search results."', 'prorank-seo')}
          </Alert>

          <div className="pr-mt-5">
            <h4>{__('Recommended Settings:', 'prorank-seo')}</h4>
            
            <table className="widefat striped">
              <thead>
                <tr>
                  <th>{__('Setting', 'prorank-seo')}</th>
                  <th>{__('Recommendation', 'prorank-seo')}</th>
                  <th>{__('Why', 'prorank-seo')}</th>
                </tr>
              </thead>
              <tbody>
                <tr>
                  <td>{__('Media Attachments', 'prorank-seo')}</td>
                  <td>✅ {__('Noindex', 'prorank-seo')}</td>
                  <td>{__('Prevents thin attachment pages', 'prorank-seo')}</td>
                </tr>
                <tr>
                  <td>{__('Date Archives', 'prorank-seo')}</td>
                  <td>✅ {__('Noindex', 'prorank-seo')}</td>
                  <td>{__('Not useful for search', 'prorank-seo')}</td>
                </tr>
                <tr>
                  <td>{__('Search Results', 'prorank-seo')}</td>
                  <td>✅ {__('Noindex', 'prorank-seo')}</td>
                  <td>{__('Internal search shouldn\'t be in Google', 'prorank-seo')}</td>
                </tr>
                <tr>
                  <td>{__('404 Pages', 'prorank-seo')}</td>
                  <td>✅ {__('Noindex', 'prorank-seo')}</td>
                  <td>{__('Error pages shouldn\'t rank', 'prorank-seo')}</td>
                </tr>
                <tr>
                  <td>{__('Posts', 'prorank-seo')}</td>
                  <td>❌ {__('Keep indexed', 'prorank-seo')}</td>
                  <td>{__('Your content should be found!', 'prorank-seo')}</td>
                </tr>
                <tr>
                  <td>{__('Pages', 'prorank-seo')}</td>
                  <td>❌ {__('Keep indexed', 'prorank-seo')}</td>
                  <td>{__('Important for SEO', 'prorank-seo')}</td>
                </tr>
              </tbody>
            </table>
          </div>
        </div>
      </Card>
    </div>
  );

  const renderAIProtectionTab = () => (
    <div className="pr-space-y-6">
      <Card className="pr-mb-6" data-search="ai protection blocking bots">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Understanding AI Protection', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <Alert variant="info">
            <strong>{__('Two Layers of Protection:', 'prorank-seo')}</strong><br />
            {__('ProRank offers two complementary ways to protect your content from AI:', 'prorank-seo')}
          </Alert>

          <div className="pr-mt-5 pr-grid pr-grid-cols-2 pr-gap-5">
            <div className="pr-p-4 pr-bg-yellow-50 pr-rounded">
              <h4>🚫 {__('Robots.txt Blocking', 'prorank-seo')}</h4>
              <p><strong>{__('Like a fence around your house', 'prorank-seo')}</strong></p>
              <ul>
                <li>{__('Completely blocks access', 'prorank-seo')}</li>
                <li>{__('Very effective', 'prorank-seo')}</li>
                <li>{__('Blocks 50+ AI bots', 'prorank-seo')}</li>
                <li>{__('No content can be accessed', 'prorank-seo')}</li>
              </ul>
              <p className="pr-mt-2.5"><strong>{__('Best for:', 'prorank-seo')}</strong> {__('Maximum protection', 'prorank-seo')}</p>
            </div>

            <div className="pr-p-4 pr-bg-blue-50 pr-rounded">
              <h4>🏷️ {__('Meta Tag Protection', 'prorank-seo')}</h4>
              <p><strong>{__('Like a "Please don\'t copy" sign', 'prorank-seo')}</strong></p>
              <ul>
                <li>{__('Polite request to AI', 'prorank-seo')}</li>
                <li>{__('Respected by ethical companies', 'prorank-seo')}</li>
                <li>{__('Content still accessible', 'prorank-seo')}</li>
                <li>{__('Good for legitimate crawlers', 'prorank-seo')}</li>
              </ul>
              <p className="pr-mt-2.5"><strong>{__('Best for:', 'prorank-seo')}</strong> {__('Balanced approach', 'prorank-seo')}</p>
            </div>
          </div>

          <div className="pr-mt-5 pr-p-4 pr-bg-green-50 pr-rounded">
            <h4>✅ {__('Recommendation:', 'prorank-seo')}</h4>
            <p>{__('Enable BOTH for maximum protection. The robots.txt blocks known bad actors, while meta tags signal your intent to ethical crawlers.', 'prorank-seo')}</p>
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="ai bots list blocked">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('AI Bots We Block', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <div className="pr-grid pr-grid-cols-[repeat(auto-fit,minmax(200px,1fr))] pr-gap-4">
            <div>
              <h4>{__('Text AI', 'prorank-seo')}</h4>
              <ul className="pr-text-sm">
                <li>GPTBot (ChatGPT)</li>
                <li>Claude-Web</li>
                <li>Gemini-Bot</li>
                <li>Bard-Bot</li>
                <li>PerplexityBot</li>
                <li>YouBot</li>
              </ul>
            </div>
            <div>
              <h4>{__('Image AI', 'prorank-seo')}</h4>
              <ul className="pr-text-sm">
                <li>MidJourney-Bot</li>
                <li>DALL-E-Bot</li>
                <li>StableDiffusion-Bot</li>
                <li>img2dataset</li>
                <li>Bytespider</li>
              </ul>
            </div>
            <div>
              <h4>{__('Data Scrapers', 'prorank-seo')}</h4>
              <ul className="pr-text-sm">
                <li>CCBot</li>
                <li>CommonCrawl</li>
                <li>DataForSeoBot</li>
                <li>omgilibot</li>
                <li>Diffbot</li>
              </ul>
            </div>
            <div>
              <h4>{__('Research Bots', 'prorank-seo')}</h4>
              <ul className="pr-text-sm">
                <li>Anthropic-AI</li>
                <li>AI2Bot</li>
                <li>cohere-ai</li>
                <li>Meta-AI</li>
                <li>Amazonbot</li>
              </ul>
            </div>
          </div>

          <Alert variant="info" className="pr-mt-5">
            {__('Plus 30+ other AI crawlers. List updated regularly for new threats.', 'prorank-seo')}
          </Alert>
        </div>
      </Card>
    </div>
  );

  const renderBestPracticesTab = () => (
    <div className="pr-space-y-6">
      <Card className="pr-mb-6" data-search="best practices recommendations">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Best Practices by Site Type', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <div className="pr-mb-6">
            <h4>🛍️ {__('E-commerce Sites', 'prorank-seo')}</h4>
            <ul>
              <li>✅ {__('Block cart and checkout pages', 'prorank-seo')}</li>
              <li>✅ {__('Block filtered URLs (color, size, price)', 'prorank-seo')}</li>
              <li>✅ {__('Noindex thank you pages', 'prorank-seo')}</li>
              <li>✅ {__('Enable IndexNow for product updates', 'prorank-seo')}</li>
              <li>❌ {__('Don\'t noindex product categories', 'prorank-seo')}</li>
            </ul>
          </div>

          <div className="pr-mb-6">
            <h4>📝 {__('Blogs', 'prorank-seo')}</h4>
            <ul>
              <li>✅ {__('Noindex tag pages if you have many', 'prorank-seo')}</li>
              <li>✅ {__('Noindex author pages if single author', 'prorank-seo')}</li>
              <li>✅ {__('Set minimum word count to 300', 'prorank-seo')}</li>
              <li>✅ {__('Block AI bots to protect content', 'prorank-seo')}</li>
              <li>❌ {__('Don\'t noindex categories', 'prorank-seo')}</li>
            </ul>
          </div>

          <div className="pr-mb-6">
            <h4>📰 {__('News Sites', 'prorank-seo')}</h4>
            <ul>
              <li>✅ {__('Enable Google Indexing API', 'prorank-seo')}</li>
              <li>✅ {__('Enable IndexNow', 'prorank-seo')}</li>
              <li>✅ {__('Keep all content indexed', 'prorank-seo')}</li>
              <li>❌ {__('Don\'t block Google News bot', 'prorank-seo')}</li>
              <li>❌ {__('Don\'t use old post noindex', 'prorank-seo')}</li>
            </ul>
          </div>

          <div>
            <h4>🏢 {__('Corporate Sites', 'prorank-seo')}</h4>
            <ul>
              <li>✅ {__('Noindex internal pages', 'prorank-seo')}</li>
              <li>✅ {__('Block AI training bots', 'prorank-seo')}</li>
              <li>✅ {__('Protect proprietary content', 'prorank-seo')}</li>
              <li>✅ {__('Use robots.txt for sensitive areas', 'prorank-seo')}</li>
            </ul>
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="dos donts mistakes avoid">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Do\'s and Don\'ts', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <div className="pr-grid pr-grid-cols-2 pr-gap-5">
            <div className="pr-p-4 pr-bg-green-50 pr-rounded">
              <h4>✅ {__('DO This', 'prorank-seo')}</h4>
              <ul>
                <li>{__('Test changes on one page first', 'prorank-seo')}</li>
                <li>{__('Monitor Search Console weekly', 'prorank-seo')}</li>
                <li>{__('Keep sitemaps updated', 'prorank-seo')}</li>
                <li>{__('Use default settings when unsure', 'prorank-seo')}</li>
                <li>{__('Block unnecessary bot traffic', 'prorank-seo')}</li>
                <li>{__('Document your changes', 'prorank-seo')}</li>
              </ul>
            </div>

            <div className="pr-p-4 pr-bg-red-50 pr-rounded">
              <h4>❌ {__('DON\'T Do This', 'prorank-seo')}</h4>
              <ul>
                <li>{__('Noindex your homepage', 'prorank-seo')}</li>
                <li>{__('Noindex all posts/pages', 'prorank-seo')}</li>
                <li>{__('Block Googlebot entirely', 'prorank-seo')}</li>
                <li>{__('Add random rules', 'prorank-seo')}</li>
                <li>{__('Ignore error messages', 'prorank-seo')}</li>
                <li>{__('Make massive changes at once', 'prorank-seo')}</li>
              </ul>
            </div>
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="monitoring success metrics">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('How to Monitor Success', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <h4>{__('Weekly Checks:', 'prorank-seo')}</h4>
          <ol>
            <li><strong>{__('Google Search Console', 'prorank-seo')}</strong>
              <ul>
                <li>{__('Coverage report for errors', 'prorank-seo')}</li>
                <li>{__('Crawl stats for efficiency', 'prorank-seo')}</li>
                <li>{__('Indexing status for new content', 'prorank-seo')}</li>
              </ul>
            </li>
            <li><strong>{__('Your Analytics', 'prorank-seo')}</strong>
              <ul>
                <li>{__('Organic traffic changes', 'prorank-seo')}</li>
                <li>{__('Page visibility trends', 'prorank-seo')}</li>
                <li>{__('Bot traffic patterns', 'prorank-seo')}</li>
              </ul>
            </li>
          </ol>

          <h4>{__('Success Indicators:', 'prorank-seo')}</h4>
          <ul>
            <li>✅ {__('Important pages indexed quickly', 'prorank-seo')}</li>
            <li>✅ {__('No critical indexing errors', 'prorank-seo')}</li>
            <li>✅ {__('Reduced bot traffic from AI crawlers', 'prorank-seo')}</li>
            <li>✅ {__('Improved crawl efficiency', 'prorank-seo')}</li>
            <li>✅ {__('Better search rankings', 'prorank-seo')}</li>
          </ul>
        </div>
      </Card>
    </div>
  );

  const renderTroubleshootingTab = () => (
    <div className="pr-space-y-6">
      <Card className="pr-mb-6" data-search="common problems issues fixes">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Common Issues & Solutions', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <div className="pr-mb-5">
            <h4>❓ {__('Robots.txt not updating?', 'prorank-seo')}</h4>
            <ol>
              <li>{__('Check for physical robots.txt file (it overrides virtual)', 'prorank-seo')}</li>
              <li>{__('Clear all caches (WordPress, CDN, browser)', 'prorank-seo')}</li>
              <li>{__('Verify settings are saved', 'prorank-seo')}</li>
              <li>{__('Test at: yoursite.com/robots.txt', 'prorank-seo')}</li>
            </ol>
          </div>

          <div className="pr-mb-5">
            <h4>❓ {__('Pages not getting indexed?', 'prorank-seo')}</h4>
            <ol>
              <li>{__('Check if pages are set to noindex', 'prorank-seo')}</li>
              <li>{__('Verify pages aren\'t blocked in robots.txt', 'prorank-seo')}</li>
              <li>{__('Ensure content quality (300+ words)', 'prorank-seo')}</li>
              <li>{__('Submit sitemap to Search Console', 'prorank-seo')}</li>
              <li>{__('Enable IndexNow for faster indexing', 'prorank-seo')}</li>
            </ol>
          </div>

          <div className="pr-mb-5">
            <h4>❓ {__('AI bots still accessing site?', 'prorank-seo')}</h4>
            <ol>
              <li>{__('Verify robots.txt shows blocking rules', 'prorank-seo')}</li>
              <li>{__('Some bots may ignore robots.txt (unethical)', 'prorank-seo')}</li>
              <li>{__('Consider server-level blocking for persistent bots', 'prorank-seo')}</li>
              <li>{__('Monitor access logs for bot activity', 'prorank-seo')}</li>
            </ol>
          </div>

          <div>
            <h4>❓ {__('IndexNow not working?', 'prorank-seo')}</h4>
            <ol>
              <li>{__('Verify API key is correct', 'prorank-seo')}</li>
              <li>{__('Check key file exists at: yoursite.com/[key].txt', 'prorank-seo')}</li>
              <li>{__('Test with the Validate button', 'prorank-seo')}</li>
            </ol>
          </div>
        </div>
      </Card>

      <Card className="pr-mb-6" data-search="faq frequently asked questions">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Frequently Asked Questions', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <div className="pr-mb-5">
            <h4>{__('Q: Should I block AI bots?', 'prorank-seo')}</h4>
            <p><strong>{__('A:', 'prorank-seo')}</strong> {__('It depends on your content:', 'prorank-seo')}</p>
            <ul>
              <li><strong>{__('Block if:', 'prorank-seo')}</strong> {__('You create original content, art, or unique data', 'prorank-seo')}</li>
              <li><strong>{__('Don\'t block if:', 'prorank-seo')}</strong> {__('You want maximum visibility or don\'t mind AI training', 'prorank-seo')}</li>
            </ul>
          </div>

          <div className="pr-mb-5">
            <h4>{__('Q: What\'s the difference between robots.txt and meta tags?', 'prorank-seo')}</h4>
            <p><strong>{__('A:', 'prorank-seo')}</strong> {__('Think of it like your house:', 'prorank-seo')}</p>
            <ul>
              <li><strong>{__('Robots.txt:', 'prorank-seo')}</strong> {__('A fence that keeps visitors out completely', 'prorank-seo')}</li>
              <li><strong>{__('Meta tags:', 'prorank-seo')}</strong> {__('A sign asking visitors to be respectful', 'prorank-seo')}</li>
            </ul>
          </div>

          <div className="pr-mb-5">
            <h4>{__('Q: Should I noindex categories?', 'prorank-seo')}</h4>
            <p><strong>{__('A:', 'prorank-seo')}</strong> {__('Usually NO. Only noindex if you have:', 'prorank-seo')}</p>
            <ul>
              <li>{__('Duplicate content issues', 'prorank-seo')}</li>
              <li>{__('Very few posts per category', 'prorank-seo')}</li>
              <li>{__('Categories with no unique value', 'prorank-seo')}</li>
            </ul>
          </div>

        </div>
      </Card>

      <Card className="pr-mb-6" data-search="before contacting support help">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Before Contacting Support', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <Toggle
            label={__('Check robots.txt is accessible at yoursite.com/robots.txt', 'prorank-seo')}
            checked={checkedItems['support1'] || false}
            onChange={() => handleCheckboxChange('support1')}
          />
          <Toggle
            label={__('Clear all caches (WordPress, CDN, browser)', 'prorank-seo')}
            checked={checkedItems['support2'] || false}
            onChange={() => handleCheckboxChange('support2')}
          />
          <Toggle
            label={__('Verify settings are saved', 'prorank-seo')}
            checked={checkedItems['support3'] || false}
            onChange={() => handleCheckboxChange('support3')}
          />
          <Toggle
            label={__('Check Google Search Console for errors', 'prorank-seo')}
            checked={checkedItems['support4'] || false}
            onChange={() => handleCheckboxChange('support4')}
          />
          <Toggle
            label={__('Test with default settings', 'prorank-seo')}
            checked={checkedItems['support5'] || false}
            onChange={() => handleCheckboxChange('support5')}
          />

          <div className="pr-mt-5 pr-p-4 pr-bg-gray-100 pr-rounded">
            <h4>{__('Still Need Help?', 'prorank-seo')}</h4>
            <p>📧 {__('Email:', 'prorank-seo')} support@prorank.com</p>
            <p>💬 {__('Forum:', 'prorank-seo')} community.prorank.com</p>
            <p>📚 {__('Docs:', 'prorank-seo')} prorank.com/docs</p>
          </div>
        </div>
      </Card>
    </div>
  );

  return (
    <div className="pr-max-w-7xl pr-mx-auto pr-space-y-6">
      <div className="pr-mb-6">
        <h2 className="pr-text-2xl pr-font-semibold pr-text-gray-900 pr-mb-2">{__('Robots & Indexing Guide', 'prorank-seo')}</h2>
        <p className="pr-text-sm pr-text-gray-600">
          {__('Complete guide to controlling search engines and protecting your content from AI bots', 'prorank-seo')}
        </p>
      </div>

      <Card className="pr-mb-6">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Search Documentation', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <div className="pr-flex pr-items-center pr-gap-2">
            <Input
              type="text"
              placeholder={__('Search for help...', 'prorank-seo')}
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
              className="pr-flex-1"
            />
            <Button variant="primary" icon={MagnifyingGlassIcon} onClick={handleSearch} disabled={!searchQuery.trim()}>
              {__('Search', 'prorank-seo')}
            </Button>
          </div>
        </div>
      </Card>

      {/* Tabs */}
      <Card>
        <div className="pr-border-b pr-border-gray-200">
          <nav className="pr-flex pr-gap-8" aria-label="Documentation tabs">
            {documentationTabs.map((tab) => (
              <button
                key={tab.name}
                onClick={() => setActiveTab(tab.name)}
                className={`pr-py-4 pr-px-1 pr-border-b-2 pr-font-medium pr-text-sm pr-transition-colors ${
                  activeTab === tab.name
                    ? 'pr-border-primary-500 pr-text-primary-600'
                    : 'pr-border-transparent pr-text-gray-500 hover:pr-text-gray-700 hover:pr-border-gray-300'
                }`}
              >
                {tab.title}
              </button>
            ))}
          </nav>
        </div>

        {/* Tab Content */}
        <div className="pr-mt-6">
          {activeTab === 'overview' && renderOverviewTab()}
          {activeTab === 'quickstart' && renderQuickStartTab()}
          {activeTab === 'features' && renderFeaturesTab()}
          {activeTab === 'aiprotection' && renderAIProtectionTab()}
          {activeTab === 'bestpractices' && renderBestPracticesTab()}
          {activeTab === 'troubleshooting' && renderTroubleshootingTab()}
        </div>
      </Card>

      <Card className="pr-mb-6">
        <div className="pr-mb-4">
          <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900">
            {__('Quick Links', 'prorank-seo')}
          </h3>
        </div>
        <div className="pr-space-y-4">
          <div className="pr-flex pr-gap-4 pr-flex-wrap">
            <Button isLink href="/wp-admin/admin.php?page=prorank-seo-settings&tab=technical-seo&section=robots-indexing">
              {__('Robots & Indexing Settings', 'prorank-seo')}
            </Button>
            
            <Button isLink href="/robots.txt" target="_blank">
              {__('View Your Robots.txt', 'prorank-seo')}
            </Button>
            
            <Button isLink href="https://search.google.com/search-console" target="_blank">
              {__('Google Search Console', 'prorank-seo')}
            </Button>
            
            <Button isLink href="https://www.indexnow.org" target="_blank">
              {__('IndexNow Documentation', 'prorank-seo')}
            </Button>
            
            <Button isLink href="mailto:support@prorank.com">
              {__('Contact Support', 'prorank-seo')}
            </Button>
          </div>
        </div>
      </Card>
    </div>
  );
};

export default RobotsIndexingDocumentation;
