import { useState } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import { Card, Button, Input, Alert } from '../../components/ui';
import {
  MagnifyingGlassIcon,
  ArrowRightIcon,
  CheckCircleIcon,
  ExclamationTriangleIcon,
  Cog6ToothIcon,
  QuestionMarkCircleIcon,
} from '@heroicons/react/24/outline';

const RedirectsDocumentation = () => {
  const [searchQuery, setSearchQuery] = useState('');
  const [copiedCode, setCopiedCode] = useState(null);
  const [activeTab, setActiveTab] = useState('overview');

  const handleSearch = () => {
    if (searchQuery.trim()) {
      const searchElement = document.querySelector(`[data-search*="${searchQuery.toLowerCase()}"]`);
      if (searchElement) {
        searchElement.scrollIntoView({ behavior: 'smooth', block: 'center' });
        searchElement.classList.add('pr-bg-warning-50');
        setTimeout(() => {
          searchElement.classList.remove('pr-bg-warning-50');
        }, 2000);
      }
    }
  };

  const copyToClipboard = (code, id) => {
    navigator.clipboard.writeText(code);
    setCopiedCode(id);
    setTimeout(() => setCopiedCode(null), 2000);
  };

  const documentationTabs = [
    { name: 'overview', title: __('Overview', 'prorank-seo') },
    { name: 'setup', title: __('Setup Guide', 'prorank-seo') },
    { name: 'features', title: __('Features', 'prorank-seo') },
    { name: 'advanced', title: __('Advanced', 'prorank-seo') },
    { name: 'api', title: __('API', 'prorank-seo') },
    { name: 'troubleshooting', title: __('Troubleshooting', 'prorank-seo') },
  ];

  const renderTabContent = () => {
    switch (activeTab) {
      case 'overview':
        return <OverviewTab />;
      case 'setup':
        return <SetupTab />;
      case 'features':
        return <FeaturesTab />;
      case 'advanced':
        return <AdvancedTab copyToClipboard={copyToClipboard} copiedCode={copiedCode} />;
      case 'api':
        return <ApiTab copyToClipboard={copyToClipboard} copiedCode={copiedCode} />;
      case 'troubleshooting':
        return <TroubleshootingTab />;
      default:
        return null;
    }
  };

  return (
    <div className="pr-max-w-7xl pr-mx-auto pr-space-y-6">
      {/* Header */}
      <Card>
        <div className="pr-flex pr-items-center pr-justify-between pr-mb-6">
          <div>
            <h2 className="pr-text-2xl pr-font-semibold pr-text-gray-900">
              {__('Redirect Manager Documentation', 'prorank-seo')}
            </h2>
            <p className="pr-mt-2 pr-text-sm pr-text-gray-600">
              {__('Complete guide to managing URL redirections and 404 errors', 'prorank-seo')}
            </p>
          </div>
          <div className="pr-flex pr-items-center pr-gap-2">
            <Input
              type="text"
              placeholder={__('Search documentation...', 'prorank-seo')}
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              onKeyPress={(e) => e.key === 'Enter' && handleSearch()}
              fullWidth={false}
              className="pr-w-64"
            />
            <Button variant="secondary" icon={MagnifyingGlassIcon} onClick={handleSearch}>
              {__('Search', 'prorank-seo')}
            </Button>
          </div>
        </div>

        {/* Tabs */}
        <div className="pr-border-b pr-border-gray-200">
          <nav className="pr-flex pr-gap-8" aria-label="Documentation tabs">
            {documentationTabs.map((tab) => (
              <button
                key={tab.name}
                onClick={() => setActiveTab(tab.name)}
                className={`pr-py-4 pr-px-1 pr-border-b-2 pr-font-medium pr-text-sm pr-transition-colors ${
                  activeTab === tab.name
                    ? 'pr-border-primary-500 pr-text-primary-600'
                    : 'pr-border-transparent pr-text-gray-500 hover:pr-text-gray-700 hover:pr-border-gray-300'
                }`}
              >
                {tab.title}
              </button>
            ))}
          </nav>
        </div>
      </Card>

      {/* Tab Content */}
      <div className="pr-mt-6">{renderTabContent()}</div>
    </div>
  );
};

// Overview Tab Component
const OverviewTab = () => (
  <div className="pr-space-y-6" data-search="overview introduction redirect manager">
    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">
        {__('What is Redirect Manager?', 'prorank-seo')}
      </h3>
      <p className="pr-text-sm pr-text-gray-600 pr-mb-6">
        {__(
          'The ProRank SEO Redirect Manager is a comprehensive solution for managing URL redirections and monitoring 404 errors. It helps maintain SEO value when URLs change and provides insights into broken links.',
          'prorank-seo'
        )}
      </p>

      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-3 pr-gap-6">
        <div className="pr-p-4 pr-bg-primary-50 pr-rounded-lg pr-border pr-border-primary-100">
          <div className="pr-flex pr-items-center pr-gap-3 pr-mb-3">
            <ArrowRightIcon className="pr-w-6 pr-h-6 pr-text-primary-500" />
            <h4 className="pr-font-semibold pr-text-gray-900">{__('Smart Redirects', 'prorank-seo')}</h4>
          </div>
          <p className="pr-text-sm pr-text-gray-600">
            {__('Automatically create redirects when URLs change to prevent 404 errors.', 'prorank-seo')}
          </p>
        </div>

        <div className="pr-p-4 pr-bg-warning-50 pr-rounded-lg pr-border pr-border-warning-100">
          <div className="pr-flex pr-items-center pr-gap-3 pr-mb-3">
            <ExclamationTriangleIcon className="pr-w-6 pr-h-6 pr-text-warning-500" />
            <h4 className="pr-font-semibold pr-text-gray-900">{__('404 Monitoring', 'prorank-seo')}</h4>
          </div>
          <p className="pr-text-sm pr-text-gray-600">
            {__('Track all 404 errors and get suggestions for fixing them.', 'prorank-seo')}
          </p>
        </div>

        <div className="pr-p-4 pr-bg-success-50 pr-rounded-lg pr-border pr-border-success-100">
          <div className="pr-flex pr-items-center pr-gap-3 pr-mb-3">
            <Cog6ToothIcon className="pr-w-6 pr-h-6 pr-text-success-500" />
            <h4 className="pr-font-semibold pr-text-gray-900">{__('Performance Optimized', 'prorank-seo')}</h4>
          </div>
          <p className="pr-text-sm pr-text-gray-600">
            {__('Built-in caching ensures fast redirect processing without impacting site speed.', 'prorank-seo')}
          </p>
        </div>
      </div>
    </Card>

    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">{__('Key Benefits', 'prorank-seo')}</h3>
      <ul className="pr-space-y-3">
        {[
          __('Preserve SEO rankings when URLs change', 'prorank-seo'),
          __('Improve user experience by eliminating 404 errors', 'prorank-seo'),
          __('Save time with automatic redirect creation', 'prorank-seo'),
          __('Monitor and fix broken links proactively', 'prorank-seo'),
          __('Optimize site performance with intelligent caching', 'prorank-seo'),
        ].map((benefit, index) => (
          <li key={index} className="pr-flex pr-items-start pr-gap-3">
            <CheckCircleIcon className="pr-w-5 pr-h-5 pr-text-success-500 pr-flex-shrink-0 pr-mt-0.5" />
            <span className="pr-text-sm pr-text-gray-700">{benefit}</span>
          </li>
        ))}
      </ul>

      <Alert variant="info" className="pr-mt-6">
        {__(
          'ProRank SEO Redirect Manager helps you manage URL redirects and monitor 404 errors efficiently.',
          'prorank-seo'
        )}
      </Alert>
    </Card>
  </div>
);

// Setup Tab Component
const SetupTab = () => (
  <div className="pr-space-y-6" data-search="setup installation configuration getting started">
    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('Getting Started', 'prorank-seo')}</h3>

      <div className="pr-space-y-6">
        {[
          {
            number: '1',
            title: __('Access Redirect Manager', 'prorank-seo'),
            desc: __('Navigate to ProRank SEO → Technical SEO → Redirect Manager', 'prorank-seo'),
          },
          {
            number: '2',
            title: __('Create Your First Redirect', 'prorank-seo'),
            desc: __('Click "Add New Redirect" and enter source and target URLs', 'prorank-seo'),
          },
          {
            number: '3',
            title: __('Configure Settings', 'prorank-seo'),
            desc: __('Go to Settings tab to enable auto-redirects and caching', 'prorank-seo'),
          },
          {
            number: '4',
            title: __('Monitor 404 Errors', 'prorank-seo'),
            desc: __('Check the 404 Monitor tab to track and fix broken links', 'prorank-seo'),
          },
        ].map((step) => (
          <div key={step.number} className="pr-flex pr-gap-4">
            <div className="pr-flex-shrink-0 pr-w-8 pr-h-8 pr-bg-primary-500 pr-text-white pr-rounded-full pr-flex pr-items-center pr-justify-center pr-font-semibold">
              {step.number}
            </div>
            <div>
              <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-1">{step.title}</h4>
              <p className="pr-text-sm pr-text-gray-600">{step.desc}</p>
            </div>
          </div>
        ))}
      </div>
    </Card>

    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">
        {__('Initial Configuration', 'prorank-seo')}
      </h3>

      <div className="pr-bg-gray-50 pr-rounded-lg pr-p-6">
        <h4 className="pr-font-semibold pr-text-gray-900 pr-mb-4">{__('Recommended Settings', 'prorank-seo')}</h4>
        <table className="pr-w-full pr-text-sm">
          <thead>
            <tr className="pr-border-b pr-border-gray-200">
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('Setting', 'prorank-seo')}</th>
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">{__('Value', 'prorank-seo')}</th>
              <th className="pr-text-left pr-pb-3 pr-font-semibold pr-text-gray-900">
                {__('Description', 'prorank-seo')}
              </th>
            </tr>
          </thead>
          <tbody className="pr-divide-y pr-divide-gray-200">
            <tr>
              <td className="pr-py-3 pr-font-medium pr-text-gray-900">{__('Auto-Redirects', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-700">{__('Enabled', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-600">
                {__('Automatically create redirects on URL changes', 'prorank-seo')}
              </td>
            </tr>
            <tr>
              <td className="pr-py-3 pr-font-medium pr-text-gray-900">{__('Redirect Type', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-700">{__('301 Permanent', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-600">
                {__('Best for SEO when URLs permanently change', 'prorank-seo')}
              </td>
            </tr>
            <tr>
              <td className="pr-py-3 pr-font-medium pr-text-gray-900">{__('Caching', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-700">{__('Enabled (3600s)', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-600">{__('Improves redirect processing speed', 'prorank-seo')}</td>
            </tr>
            <tr>
              <td className="pr-py-3 pr-font-medium pr-text-gray-900">{__('404 Monitoring', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-700">{__('Enabled', 'prorank-seo')}</td>
              <td className="pr-py-3 pr-text-gray-600">
                {__('Track and fix broken links automatically', 'prorank-seo')}
              </td>
            </tr>
          </tbody>
        </table>
      </div>
    </Card>
  </div>
);

// Features Tab Component
const FeaturesTab = () => (
  <div className="pr-space-y-6" data-search="features capabilities functionality">
    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('Feature Highlights', 'prorank-seo')}</h3>

      <ul className="pr-space-y-3">
        {[
          __('Basic Redirects (301, 302, 307, 308)', 'prorank-seo'),
          __('Bulk Management', 'prorank-seo'),
          __('CSV Import/Export', 'prorank-seo'),
          __('Auto-Redirect on URL Change', 'prorank-seo'),
          __('Redirect Chain Detection', 'prorank-seo'),
          __('404 Error Monitoring', 'prorank-seo'),
        ].map((feature, index) => (
          <li key={index} className="pr-flex pr-items-start pr-gap-2">
            <CheckCircleIcon className="pr-w-5 pr-h-5 pr-text-success-500 pr-flex-shrink-0" />
            <span className="pr-text-sm pr-text-gray-700">{feature}</span>
          </li>
        ))}
      </ul>
    </Card>

    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">
        {__('Redirect Types Explained', 'prorank-seo')}
      </h3>

      <div className="pr-grid pr-grid-cols-1 md:pr-grid-cols-2 pr-gap-6">
        {[
          {
            code: '301',
            title: __('Moved Permanently', 'prorank-seo'),
            when: __('Content has permanently moved to a new URL', 'prorank-seo'),
            seo: __('Passes ~90-99% of link equity', 'prorank-seo'),
          },
          {
            code: '302',
            title: __('Found (Temporary)', 'prorank-seo'),
            when: __('Content is temporarily at a different URL', 'prorank-seo'),
            seo: __('Does not pass link equity', 'prorank-seo'),
          },
          {
            code: '307',
            title: __('Temporary Redirect', 'prorank-seo'),
            when: __('HTTP/1.1 temporary redirect needed', 'prorank-seo'),
            seo: __('Does not pass link equity', 'prorank-seo'),
          },
          {
            code: '308',
            title: __('Permanent Redirect', 'prorank-seo'),
            when: __('HTTP/1.1 permanent redirect needed', 'prorank-seo'),
            seo: __('Passes link equity like 301', 'prorank-seo'),
          },
        ].map((type) => (
          <div key={type.code} className="pr-p-4 pr-bg-gray-50 pr-rounded-lg pr-border pr-border-gray-200">
            <h4 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-2">
              {type.code} - {type.title}
            </h4>
            <p className="pr-text-sm pr-text-gray-600 pr-mb-2">
              <span className="pr-font-medium">{__('Use when:', 'prorank-seo')}</span> {type.when}
            </p>
            <p className="pr-text-sm pr-text-primary-600">
              <span className="pr-font-medium">{__('SEO Impact:', 'prorank-seo')}</span> {type.seo}
            </p>
          </div>
        ))}
      </div>
    </Card>
  </div>
);

// Advanced Tab Component
const AdvancedTab = ({ copyToClipboard, copiedCode }) => (
  <div className="pr-space-y-6" data-search="advanced regex patterns hooks filters">
    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('Advanced Features', 'prorank-seo')}</h3>

      <div className="pr-space-y-6">
        <div>
          <h4 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-3">
            {__('WordPress Hooks', 'prorank-seo')}
          </h4>
          <p className="pr-text-sm pr-text-gray-600 pr-mb-4">
            {__('Extend functionality with these hooks:', 'prorank-seo')}
          </p>

          <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4 pr-relative">
            <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
              <code>
                {`// Track redirect execution
add_action('prorank_redirect_created', function($redirect_id) {
    error_log('Redirect created: ' . $redirect_id);
});

// Modify 404 suggestions
add_filter('prorank_404_suggestion', function($suggestion, $url) {
    // Custom suggestion logic
    return $suggestion;
}, 10, 2);`}
              </code>
            </pre>
            <Button
              variant="secondary"
              size="sm"
              onClick={() =>
                copyToClipboard(
                  `add_action('prorank_redirect_created', function($redirect_id) {\n    error_log('Redirect created: ' . $redirect_id);\n});`,
                  'hook1'
                )
              }
              className="pr-absolute pr-top-2 pr-right-2"
            >
              {copiedCode === 'hook1' ? __('Copied!', 'prorank-seo') : __('Copy', 'prorank-seo')}
            </Button>
          </div>
        </div>

        <div>
          <h4 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-3">
            {__('Performance Optimization', 'prorank-seo')}
          </h4>
          <ul className="pr-space-y-2">
            {[
              __('Enable caching for high-traffic sites', 'prorank-seo'),
              __('Clean old 404 logs regularly', 'prorank-seo'),
              __('Use exact matches when possible', 'prorank-seo'),
              __('Set appropriate cache expiry times', 'prorank-seo'),
            ].map((tip, index) => (
              <li key={index} className="pr-flex pr-items-start pr-gap-2">
                <span className="pr-text-primary-500 pr-flex-shrink-0">•</span>
                <span className="pr-text-sm pr-text-gray-700">{tip}</span>
              </li>
            ))}
          </ul>
        </div>
      </div>
    </Card>
  </div>
);

// API Tab Component
const ApiTab = ({ copyToClipboard, copiedCode }) => (
  <div className="pr-space-y-6" data-search="api rest endpoints integration">
    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">{__('REST API Reference', 'prorank-seo')}</h3>

      <Alert variant="info" className="pr-mb-6">
        {__('All endpoints require WordPress authentication with manage_options capability.', 'prorank-seo')}
      </Alert>

      <div className="pr-space-y-6">
        <div>
          <h4 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-4">
            {__('Redirect Endpoints', 'prorank-seo')}
          </h4>

          <div className="pr-space-y-4">
            {[
              {
                method: 'GET',
                path: '/wp-json/prorank-seo/v1/redirects',
                desc: __('List all redirects with pagination', 'prorank-seo'),
                example: `// Parameters\npage: 1\nper_page: 20\nsearch: "old-page"\ntype: "301"\nstatus: "active"`,
              },
              {
                method: 'POST',
                path: '/wp-json/prorank-seo/v1/redirects',
                desc: __('Create a new redirect', 'prorank-seo'),
                example: `{\n  "source_url": "/old-page",\n  "target_url": "/new-page",\n  "type": 301,\n  "status": "active"\n}`,
                copyId: 'api1',
              },
              {
                method: 'PUT',
                path: '/wp-json/prorank-seo/v1/redirects/{id}',
                desc: __('Update existing redirect', 'prorank-seo'),
              },
              {
                method: 'DELETE',
                path: '/wp-json/prorank-seo/v1/redirects/{id}',
                desc: __('Delete redirect', 'prorank-seo'),
              },
            ].map((endpoint, index) => (
              <div key={index} className="pr-border pr-border-gray-200 pr-rounded-lg pr-p-4">
                <div className="pr-flex pr-items-center pr-gap-3 pr-mb-2">
                  <span
                    className={`pr-px-2 pr-py-1 pr-rounded pr-text-xs pr-font-semibold pr-uppercase ${
                      endpoint.method === 'GET'
                        ? 'pr-bg-info-100 pr-text-info-700'
                        : endpoint.method === 'POST'
                        ? 'pr-bg-success-100 pr-text-success-700'
                        : endpoint.method === 'PUT'
                        ? 'pr-bg-warning-100 pr-text-warning-700'
                        : 'pr-bg-error-100 pr-text-error-700'
                    }`}
                  >
                    {endpoint.method}
                  </span>
                  <code className="pr-text-sm pr-text-gray-700">{endpoint.path}</code>
                </div>
                <p className="pr-text-sm pr-text-gray-600 pr-mb-3">{endpoint.desc}</p>
                {endpoint.example && (
                  <div className="pr-bg-gray-900 pr-rounded pr-p-3 pr-relative">
                    <pre className="pr-text-white pr-text-xs pr-overflow-x-auto">
                      <code>{endpoint.example}</code>
                    </pre>
                    {endpoint.copyId && (
                      <Button
                        variant="secondary"
                        size="xs"
                        onClick={() => copyToClipboard(endpoint.example, endpoint.copyId)}
                        className="pr-absolute pr-top-1 pr-right-1"
                      >
                        {copiedCode === endpoint.copyId ? __('Copied!', 'prorank-seo') : __('Copy', 'prorank-seo')}
                      </Button>
                    )}
                  </div>
                )}
              </div>
            ))}
          </div>
        </div>

        <div>
          <h4 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-4">
            {__('404 Monitor Endpoints', 'prorank-seo')}
          </h4>

          <div className="pr-space-y-4">
            {[
              {
                method: 'GET',
                path: '/wp-json/prorank-seo/v1/404-monitor/stats',
                desc: __('Get 404 statistics and top errors', 'prorank-seo'),
              },
              {
                method: 'POST',
                path: '/wp-json/prorank-seo/v1/404-monitor/create-redirect',
                desc: __('Create redirect from 404 error', 'prorank-seo'),
              },
            ].map((endpoint, index) => (
              <div key={index} className="pr-border pr-border-gray-200 pr-rounded-lg pr-p-4">
                <div className="pr-flex pr-items-center pr-gap-3 pr-mb-2">
                  <span
                    className={`pr-px-2 pr-py-1 pr-rounded pr-text-xs pr-font-semibold pr-uppercase ${
                      endpoint.method === 'GET'
                        ? 'pr-bg-info-100 pr-text-info-700'
                        : 'pr-bg-success-100 pr-text-success-700'
                    }`}
                  >
                    {endpoint.method}
                  </span>
                  <code className="pr-text-sm pr-text-gray-700">{endpoint.path}</code>
                </div>
                <p className="pr-text-sm pr-text-gray-600">{endpoint.desc}</p>
              </div>
            ))}
          </div>
        </div>
      </div>
    </Card>
  </div>
);

// Troubleshooting Tab Component
const TroubleshootingTab = () => (
  <div className="pr-space-y-6" data-search="troubleshooting problems issues fixes">
    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-6">
        {__('Common Issues & Solutions', 'prorank-seo')}
      </h3>

      <div className="pr-space-y-6">
        {[
          {
            issue: __('Redirects Not Working', 'prorank-seo'),
            problem: __('Created redirects are not functioning', 'prorank-seo'),
            solutions: [
              __('Verify redirect status is "Active"', 'prorank-seo'),
              __('Clear browser and site cache', 'prorank-seo'),
              __('Check for conflicting redirects or plugins', 'prorank-seo'),
              __('Ensure correct URL format (relative vs absolute)', 'prorank-seo'),
              __('Run redirect chain detection', 'prorank-seo'),
            ],
          },
          {
            issue: __('404 Monitor Not Tracking', 'prorank-seo'),
            problem: __('404 errors are not being recorded', 'prorank-seo'),
            solutions: [
              __('Check 404 monitoring is enabled in Settings', 'prorank-seo'),
              __('Ensure database tables exist (reactivate plugin)', 'prorank-seo'),
              __('Test with a non-existent URL', 'prorank-seo'),
            ],
          },
          {
            issue: __('High Memory Usage', 'prorank-seo'),
            problem: __('Plugin causing high memory consumption', 'prorank-seo'),
            solutions: [
              __('Reduce cache expiry time', 'prorank-seo'),
              __('Clean old 404 logs (Settings > 404 Retention)', 'prorank-seo'),
              __('Remove unused redirects', 'prorank-seo'),
              __('Limit regex redirects', 'prorank-seo'),
              __('Enable object caching (Redis/Memcached)', 'prorank-seo'),
            ],
          },
          {
            issue: __('Import/Export Issues', 'prorank-seo'),
            problem: __('CSV import failing or export not working', 'prorank-seo'),
            solutions: [
              __('Verify CSV format matches template', 'prorank-seo'),
              __('Check file size limits (PHP upload_max_filesize)', 'prorank-seo'),
              __('Ensure URLs are properly formatted', 'prorank-seo'),
              __('Remove special characters from URLs', 'prorank-seo'),
            ],
          },
        ].map((item, index) => (
          <div key={index} className="pr-p-4 pr-bg-gray-50 pr-rounded-lg pr-border pr-border-gray-200">
            <h4 className="pr-text-lg pr-font-semibold pr-text-gray-900 pr-mb-2">{item.issue}</h4>
            <p className="pr-text-sm pr-text-gray-600 pr-mb-3">
              <span className="pr-font-medium">{__('Problem:', 'prorank-seo')}</span> {item.problem}
            </p>
            <div className="pr-text-sm pr-text-gray-700">
              <span className="pr-font-medium pr-block pr-mb-2">{__('Solutions:', 'prorank-seo')}</span>
              <ol className="pr-list-decimal pr-list-inside pr-space-y-1 pr-pl-2">
                {item.solutions.map((solution, sIndex) => (
                  <li key={sIndex}>{solution}</li>
                ))}
              </ol>
            </div>
          </div>
        ))}
      </div>
    </Card>

    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">{__('Debug Mode', 'prorank-seo')}</h3>
      <p className="pr-text-sm pr-text-gray-600 pr-mb-4">
        {__('Enable debug logging for detailed troubleshooting:', 'prorank-seo')}
      </p>

      <div className="pr-bg-gray-900 pr-rounded-lg pr-p-4">
        <pre className="pr-text-white pr-text-sm pr-overflow-x-auto">
          <code>
            {`// Add to wp-config.php
define('PRORANK_REDIRECT_DEBUG', true);
define('WP_DEBUG', true);
define('WP_DEBUG_LOG', true);`}
          </code>
        </pre>
      </div>

      <Alert variant="warning" className="pr-mt-4">
        {__('Remember to disable debug mode on production sites for security and performance.', 'prorank-seo')}
      </Alert>
    </Card>

    <Card>
      <h3 className="pr-text-xl pr-font-semibold pr-text-gray-900 pr-mb-4">{__('Get Help', 'prorank-seo')}</h3>

      <div className="pr-flex pr-flex-wrap pr-gap-3">
        <a
          href="https://prorank.io/support"
          target="_blank"
          rel="noopener noreferrer"
          className="pr-inline-flex pr-items-center pr-px-4 pr-py-2 pr-bg-white pr-text-gray-700 pr-border pr-border-gray-300 pr-rounded-md pr-font-medium hover:pr-bg-gray-50 pr-transition-colors"
        >
          {__('Contact Support', 'prorank-seo')}
        </a>
        <a
          href="https://prorank.io/docs"
          target="_blank"
          rel="noopener noreferrer"
          className="pr-inline-flex pr-items-center pr-px-4 pr-py-2 pr-bg-white pr-text-gray-700 pr-border pr-border-gray-300 pr-rounded-md pr-font-medium hover:pr-bg-gray-50 pr-transition-colors"
        >
          {__('Online Documentation', 'prorank-seo')}
        </a>
        <a
          href="https://prorank.io/community"
          target="_blank"
          rel="noopener noreferrer"
          className="pr-inline-flex pr-items-center pr-px-4 pr-py-2 pr-bg-white pr-text-gray-700 pr-border pr-border-gray-300 pr-rounded-md pr-font-medium hover:pr-bg-gray-50 pr-transition-colors"
        >
          {__('Community Forum', 'prorank-seo')}
        </a>
      </div>
    </Card>
  </div>
);

export default RedirectsDocumentation;
