import {  useState, useEffect  } from '@wordpress/element';
import {
  TextControl,
  TextareaControl,
  SelectControl,
  RadioControl,
  Spinner
} from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import ProrankButton from '../../../components/ProrankButton';
import { useForm, Controller } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import apiFetch from '@wordpress/api-fetch';
import { FeedbackFormSkeleton } from '../../../components/Skeleton/Skeleton';
import type { 
  TabProps, 
  FeedbackFormData, 
  FeedbackResponse,
  FeedbackListResponse,
  FeedbackType,
  FeedbackPriority 
} from '../../../types/support';
import '@admin/styles/index.css';
import * as React from 'react';

// Validation schema
const feedbackSchema = z.object({
  type: z.enum(['feature', 'bug', 'general', 'support']),
  subject: z.string().min(3, __('Subject must be at least 3 characters', 'prorank-seo')),
  message: z.string().min(10, __('Message must be at least 10 characters', 'prorank-seo')),
  priority: z.enum(['low', 'medium', 'high']),
  email: z.string().email(__('Please enter a valid email', 'prorank-seo')).optional().or(z.literal('')),
});

const FeedbackTab: React.FC<TabProps> = ({ addNotice }) => {
  const queryClient = useQueryClient();
  
  const {
    control,
    handleSubmit,
    reset,
    watch,
    setValue,
    formState: { errors, isSubmitting }
  } = useForm<FeedbackFormData>({
    resolver: zodResolver(feedbackSchema),
    defaultValues: {
      type: 'feature',
      subject: '',
      message: '',
      priority: 'medium',
      email: '',
    }
  });

  const feedbackType = watch('type');

  // Query for recent feedback
  const { 
    data: feedbackData, 
    isLoading: isLoadingFeedback,
    error: feedbackError,
    refetch: refetchFeedback 
  } = useQuery<FeedbackListResponse>({
    queryKey: ['feedback', 'recent'],
    queryFn: async () => {
      const response = await apiFetch<FeedbackListResponse>({
        path: '/prorank-seo/v1/feedback/list?limit=5',
        method: 'GET',
      });
      return response;
    },
    retry: 3,
    retryDelay: (attemptIndex) => Math.min(1000 * 2 ** attemptIndex, 30000),
    staleTime: 5 * 60 * 1000,
    gcTime: 10 * 60 * 1000,
  });

  // Mutation for submitting feedback
  const submitFeedback = useMutation<FeedbackResponse, Error, FeedbackFormData>({
    mutationFn: async (data) => {
      const response = await apiFetch<FeedbackResponse>({
        path: '/prorank-seo/v1/feedback/submit',
        method: 'POST',
        data: {
          ...data,
          siteUrl: window.location.origin,
          pluginVersion: window.prorankSeoAdmin?.pluginVersion || 'unknown',
        },
      });
      return response;
    },
    onSuccess: (response) => {
      if (response.success) {
        addNotice(
          'success',
          __("Thank you for your feedback! We'll review it soon.", 'prorank-seo')
        );
        reset();
        queryClient.invalidateQueries({ queryKey: ['feedback'] });
      } else {
        addNotice('error', response.message || __('Failed to submit feedback', 'prorank-seo'));
      }
    },
    onError: (error) => {
      const errorMessage = error.message || __('Failed to submit feedback', 'prorank-seo');
      addNotice('error', errorMessage);
    },
    retry: 2,
    retryDelay: (attemptIndex) => Math.min(1000 * 2 ** attemptIndex, 10000),
  });

  const feedbackTypes = [
    { label: __('Feature Request', 'prorank-seo'), value: 'feature' },
    { label: __('Bug Report', 'prorank-seo'), value: 'bug' },
    { label: __('General Feedback', 'prorank-seo'), value: 'general' },
    { label: __('Support Request', 'prorank-seo'), value: 'support' },
  ];

  const priorities = [
    { label: __('Low', 'prorank-seo'), value: 'low' },
    { label: __('Medium', 'prorank-seo'), value: 'medium' },
    { label: __('High', 'prorank-seo'), value: 'high' },
  ];

  const getTemplateMessage = (type: FeedbackType): string => {
    switch (type) {
      case 'bug':
        return __(
          'Steps to reproduce:\n1. \n2. \n3. \n\nExpected behavior:\n\nActual behavior:\n\nBrowser/Device:',
          'prorank-seo'
        );
      case 'feature':
        return __('Feature description:\n\nUse case:\n\nExpected benefit:', 'prorank-seo');
      default:
        return '';
    }
  };

  // Apply template when feedback type changes
  useEffect(() => {
    const template = getTemplateMessage(feedbackType);
    if (template && !watch('message')) {
      setValue('message', template);
    }
  }, [feedbackType, setValue, watch]);

  const onSubmit = (data: FeedbackFormData) => {
    submitFeedback.mutate(data);
  };

  return (
    <>
      <div className="prorank-settings-section">
        <div className="prorank-card">
          <div className="prorank-card-header">
            <h3 className="prorank-card-title">{__('Submit Feedback', 'prorank-seo')}</h3>
            <p className="prorank-card-description">
              {__(
                'We value your feedback! Help us improve ProRank SEO by sharing your thoughts, reporting bugs, or requesting new features.',
                'prorank-seo'
              )}
            </p>
          </div>

          <form 
            onSubmit={handleSubmit(onSubmit)} 
            className="prorank-form prorank-card-body"
          >
            <div className="prorank-form-field">
              <Controller
                name="type"
                control={control}
                render={({ field }) => (
                  <div className="prorank-radio-group">
                    <RadioControl
                      label={__('Feedback Type', 'prorank-seo')}
                      selected={field.value}
                      options={feedbackTypes}
                      onChange={(value) => field.onChange(value as FeedbackType)}
                    />
                  </div>
                )}
              />
            </div>

            <div className="prorank-form-field">
              <Controller
                name="subject"
                control={control}
                render={({ field }) => (
                  <div>
                    <TextControl
                      label={__('Subject *', 'prorank-seo')}
                      value={field.value}
                      onChange={field.onChange}
                      placeholder={__('Brief description of your feedback', 'prorank-seo')}
                      __nextHasNoMarginBottom
                      __next40pxDefaultSize
                    />
                    {errors.subject && (
                      <span className="prorank-error-message">{errors.subject.message}</span>
                    )}
                  </div>
                )}
              />
            </div>

            <div className="prorank-form-field">
              <Controller
                name="message"
                control={control}
                render={({ field }) => (
                  <div>
                    <TextareaControl
                      label={__('Message *', 'prorank-seo')}
                      value={field.value}
                      onChange={field.onChange}
                      rows={8}
                      help={__('Please be as detailed as possible', 'prorank-seo')}
                      __nextHasNoMarginBottom
                    />
                    {errors.message && (
                      <span className="prorank-error-message">{errors.message.message}</span>
                    )}
                  </div>
                )}
              />
            </div>

            {feedbackType === 'bug' && (
              <div className="prorank-form-field">
                <Controller
                  name="priority"
                  control={control}
                  render={({ field }) => (
                    <SelectControl
                      label={__('Priority', 'prorank-seo')}
                      value={field.value}
                      options={priorities}
                      onChange={(value) => field.onChange(value as FeedbackPriority)}
                    />
                  )}
                />
              </div>
            )}

            <div className="prorank-form-field">
              <Controller
                name="email"
                control={control}
                render={({ field }) => (
                  <div>
                    <TextControl
                      label={__('Email (optional)', 'prorank-seo')}
                      value={field.value}
                      onChange={field.onChange}
                      type="email"
                      help={__("If you'd like us to follow up with you", 'prorank-seo')}
                      __nextHasNoMarginBottom
                      __next40pxDefaultSize
                    />
                    {errors.email && (
                      <span className="prorank-error-message">{errors.email.message}</span>
                    )}
                  </div>
                )}
              />
            </div>

            <div className="prorank-form-actions">
              <ProrankButton
                variant="primary"
                type="submit"
                disabled={isSubmitting || submitFeedback.isPending}
                aria-busy={isSubmitting || submitFeedback.isPending}
              >
                {(isSubmitting || submitFeedback.isPending) ? (
                  <React.Fragment>
                    <Spinner />
                    <span aria-live="polite">{__('Submitting…', 'prorank-seo')}</span>
                  </React.Fragment>
                ) : (
                  <>📤 {__('Submit Feedback', 'prorank-seo')}</>
                )}
              </ProrankButton>
              
              {Object.keys(errors).length > 0 && (
                <span className="prorank-form-error-summary">
                  ⚠️ {__('Please fix the errors above', 'prorank-seo')}
                </span>
              )}
            </div>
          </form>
        </div>
      </div>

      {/* Recent feedback */}
      {!feedbackError && !isLoadingFeedback && feedbackData && feedbackData.feedback && feedbackData.feedback.length > 0 && (
        <div className="prorank-card">
          <div className="prorank-card-header">
            <h3 className="prorank-card-title">{__('Your Recent Feedback', 'prorank-seo')}</h3>
            <p className="prorank-card-description">
              {__('Track the status of your submitted feedback and feature requests', 'prorank-seo')}
            </p>
          </div>
          <div className="prorank-card-body prorank-feedback-list">
            {feedbackData.feedback.map((item) => (
              <div key={item.id} className="prorank-feedback-item">
                <div className="prorank-feedback-header">
                  <div className="prorank-feedback-badges">
                    <span className={`prorank-feedback-type-badge prorank-feedback-type-badge--${item.type}`}>
                      {item.type}
                    </span>
                    {item.priority && (
                      <span className={`prorank-feedback-priority-badge prorank-feedback-priority-badge--${item.priority}`}>
                        {item.priority === 'high' ? '🔴' : item.priority === 'medium' ? '🟡' : '⚪'} {item.priority}
                      </span>
                    )}
                  </div>
                  <span className="prorank-feedback-date">
                    {item.createdAt}
                  </span>
                </div>
                
                <h4 className="prorank-feedback-title">
                  {item.subject}
                </h4>
                
                {item.message && (
                  <p className="prorank-feedback-message">
                    {item.message}
                  </p>
                )}
                
                <div className="prorank-feedback-footer">
                  <span className={`prorank-feedback-status-badge prorank-feedback-status-badge--${item.status}`}>
                    {item.status === 'resolved' ? '✅ ' : 
                     item.status === 'in_progress' ? '⚡ ' :
                     item.status === 'reviewed' ? '👀 ' : '⏳ '}
                    {item.status === 'resolved' ? __('Resolved', 'prorank-seo') :
                     item.status === 'in_progress' ? __('In Progress', 'prorank-seo') :
                     item.status === 'reviewed' ? __('Reviewed', 'prorank-seo') : __('Pending', 'prorank-seo')}
                  </span>
                  
                  {item.response && (
                    <a href="#" className="prorank-feedback-response-link">
                      {__('View Response', 'prorank-seo')}
                    </a>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </>
  );
};

export default FeedbackTab;