import {  useState, useEffect  } from '@wordpress/element';
import {
  TextControl,
  TextareaControl,
  SelectControl,
  RadioControl,
  Spinner
} from '@wordpress/components';
import { __ } from '@wordpress/i18n';
import ProrankButton from '../../../components/ProrankButton';
import { useForm, Controller } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { z } from 'zod';
import { useQuery, useMutation, useQueryClient } from '@tanstack/react-query';
import apiFetch from '@wordpress/api-fetch';
import { FeedbackFormSkeleton, Skeleton } from '../../../components/Skeleton/Skeleton';
import type { 
  TabProps, 
  FeedbackFormData, 
  FeedbackResponse,
  FeedbackListResponse,
  FeedbackType,
  FeedbackPriority 
} from '../../../types/support';
import * as React from 'react';

// Validation schema
const feedbackSchema = z.object({
  type: z.enum(['feature', 'bug', 'general', 'support']),
  subject: z.string().min(3, __('Subject must be at least 3 characters', 'prorank-seo')),
  message: z.string().min(10, __('Message must be at least 10 characters', 'prorank-seo')),
  priority: z.enum(['low', 'medium', 'high']),
  email: z.string().email(__('Please enter a valid email', 'prorank-seo')).optional().or(z.literal('')),
});

const FeedbackTab: React.FC<TabProps> = ({ addNotice }) => {
  const queryClient = useQueryClient();
  
  const {
    control,
    handleSubmit,
    reset,
    watch,
    setValue,
    formState: { errors, isSubmitting }
  } = useForm<FeedbackFormData>({
    resolver: zodResolver(feedbackSchema),
    defaultValues: {
      type: 'feature',
      subject: '',
      message: '',
      priority: 'medium',
      email: '',
    }
  });

  const feedbackType = watch('type');

  // Query for recent feedback with enhanced error handling
  const { 
    data: feedbackData, 
    isLoading: isLoadingFeedback,
    error: feedbackError,
    refetch: refetchFeedback 
  } = useQuery<FeedbackListResponse>({
    queryKey: ['feedback', 'recent'],
    queryFn: async () => {
      const response = await apiFetch<FeedbackListResponse>({
        path: '/prorank-seo/v1/feedback/list?limit=5',
        method: 'GET',
      });
      // Ensure response has expected structure
      if (!response || typeof response !== 'object') {
        throw new Error('Invalid response format');
      }
      return response;
    },
    retry: 3,
    retryDelay: (attemptIndex) => Math.min(1000 * 2 ** attemptIndex, 30000),
    staleTime: 5 * 60 * 1000, // 5 minutes
    gcTime: 10 * 60 * 1000, // 10 minutes cache time
  });

  // Mutation for submitting feedback
  const submitFeedback = useMutation<FeedbackResponse, Error, FeedbackFormData>({
    mutationFn: async (data) => {
      const response = await apiFetch<FeedbackResponse>({
        path: '/prorank-seo/v1/feedback/submit',
        method: 'POST',
        data: {
          ...data,
          siteUrl: window.location.origin,
          pluginVersion: window.prorankSeoAdmin?.pluginVersion || 'unknown',
        },
      });
      return response;
    },
    onSuccess: (response) => {
      if (response.success) {
        addNotice(
          'success',
          __("Thank you for your feedback! We'll review it soon.", 'prorank-seo')
        );
        reset();
        queryClient.invalidateQueries({ queryKey: ['feedback'] });
      } else {
        addNotice('error', response.message || __('Failed to submit feedback', 'prorank-seo'));
      }
    },
    onError: (error) => {
      const errorMessage = error.message || __('Failed to submit feedback', 'prorank-seo');
      addNotice('error', errorMessage);
    },
    retry: 2,
    retryDelay: (attemptIndex) => Math.min(1000 * 2 ** attemptIndex, 10000),
  });

  const feedbackTypes = [
    { label: __('Feature Request', 'prorank-seo'), value: 'feature' },
    { label: __('Bug Report', 'prorank-seo'), value: 'bug' },
    { label: __('General Feedback', 'prorank-seo'), value: 'general' },
    { label: __('Support Request', 'prorank-seo'), value: 'support' },
  ];

  const priorities = [
    { label: __('Low', 'prorank-seo'), value: 'low' },
    { label: __('Medium', 'prorank-seo'), value: 'medium' },
    { label: __('High', 'prorank-seo'), value: 'high' },
  ];

  const getTemplateMessage = (type: FeedbackType): string => {
    switch (type) {
      case 'bug':
        return __(
          'Steps to reproduce:\n1. \n2. \n3. \n\nExpected behavior:\n\nActual behavior:\n\nBrowser/Device:',
          'prorank-seo'
        );
      case 'feature':
        return __('Feature description:\n\nUse case:\n\nExpected benefit:', 'prorank-seo');
      default:
        return '';
    }
  };

  // Apply template when feedback type changes
  useEffect(() => {
    const template = getTemplateMessage(feedbackType);
    if (template && !watch('message')) {
      setValue('message', template);
    }
  }, [feedbackType, setValue, watch]);

  const onSubmit = (data: FeedbackFormData) => {
    submitFeedback.mutate(data);
  };

  return (
    <>
      <div className="prorank-settings-section prorank-feedback-section" role="region" aria-label={__('Feedback Section', 'prorank-seo')}>
        <div className="prorank-card" style={{ 
          padding: '0',
          overflow: 'hidden',
          border: '1px solid #e5e7eb',
          borderRadius: '12px',
          background: '#ffffff'
        }}>
        <div className="prorank-card-header" style={{
          padding: '24px 32px',
          background: 'linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%)',
          borderBottom: '1px solid #e5e7eb'
        }}>
          <h3 className="prorank-card-title" id="feedback-form-title" style={{
            fontSize: '20px',
            fontWeight: '600',
            color: '#1f2937',
            marginBottom: '8px'
          }}>{__('Submit Feedback', 'prorank-seo')}</h3>
          <p className="prorank-card-description" id="feedback-form-description" style={{
            fontSize: '14px',
            color: '#6b7280',
            lineHeight: '1.5',
            margin: '0'
          }}>
            {__(
              'We value your feedback! Help us improve ProRank SEO by sharing your thoughts, reporting bugs, or requesting new features.',
              'prorank-seo'
            )}
          </p>
        </div>

        <form 
          onSubmit={handleSubmit(onSubmit)} 
          className="prorank-form prorank-feedback-form"
          aria-labelledby="feedback-form-title"
          aria-describedby="feedback-form-description"
          style={{
            padding: '32px'
          }}
        >
          <div className="prorank-form-field" style={{ marginBottom: '24px' }}>
            <Controller
              name="type"
              control={control}
              render={({ field }) => (
                <div style={{
                  padding: '16px',
                  background: '#f9fafb',
                  borderRadius: '8px',
                  border: '1px solid #e5e7eb'
                }}>
                  <RadioControl
                    label={
                      <span style={{ 
                        fontSize: '14px', 
                        fontWeight: '600',
                        color: '#374151',
                        marginBottom: '12px',
                        display: 'block'
                      }}>
                        {__('Feedback Type', 'prorank-seo')}
                      </span>
                    }
                    selected={field.value}
                    options={feedbackTypes}
                    onChange={(value) => field.onChange(value as FeedbackType)}
                  />
                </div>
              )}
            />
          </div>

          <div className="prorank-form-field" style={{ marginBottom: '24px' }}>
            <Controller
              name="subject"
              control={control}
              render={({ field }) => (
                <div>
                  <TextControl
                    label={
                      <span style={{ 
                        fontSize: '14px', 
                        fontWeight: '600',
                        color: '#374151'
                      }}>
                        {__('Subject *', 'prorank-seo')}
                      </span>
                    }
                    value={field.value}
                    onChange={field.onChange}
                    placeholder={__('Brief description of your feedback', 'prorank-seo')}
                    __nextHasNoMarginBottom
                    __next40pxDefaultSize
                    style={{
                      borderRadius: '6px'
                    }}
                  />
                  {errors.subject && (
                    <span className="prorank-error-message" style={{
                      color: '#ef4444',
                      fontSize: '12px',
                      marginTop: '4px',
                      display: 'block'
                    }}>{errors.subject.message}</span>
                  )}
                </div>
              )}
            />
          </div>

          <div className="prorank-form-field" style={{ marginBottom: '24px' }}>
            <Controller
              name="message"
              control={control}
              render={({ field }) => (
                <div>
                  <TextareaControl
                    label={
                      <span style={{ 
                        fontSize: '14px', 
                        fontWeight: '600',
                        color: '#374151'
                      }}>
                        {__('Message *', 'prorank-seo')}
                      </span>
                    }
                    value={field.value}
                    onChange={field.onChange}
                    rows={8}
                    help={
                      <span style={{ 
                        fontSize: '12px',
                        color: '#6b7280'
                      }}>
                        {__('Please be as detailed as possible', 'prorank-seo')}
                      </span>
                    }
                    __nextHasNoMarginBottom
                    style={{
                      borderRadius: '6px',
                      padding: '12px',
                      fontSize: '14px',
                      lineHeight: '1.5'
                    }}
                  />
                  {errors.message && (
                    <span className="prorank-error-message" style={{
                      color: '#ef4444',
                      fontSize: '12px',
                      marginTop: '4px',
                      display: 'block'
                    }}>{errors.message.message}</span>
                  )}
                </div>
              )}
            />
          </div>

          {feedbackType === 'bug' && (
            <div className="prorank-form-field" style={{ marginBottom: '24px' }}>
              <Controller
                name="priority"
                control={control}
                render={({ field }) => (
                  <SelectControl
                    label={
                      <span style={{ 
                        fontSize: '14px', 
                        fontWeight: '600',
                        color: '#374151'
                      }}>
                        {__('Priority', 'prorank-seo')}
                      </span>
                    }
                    value={field.value}
                    options={priorities}
                    onChange={(value) => field.onChange(value as FeedbackPriority)}
                    style={{
                      borderRadius: '6px'
                    }}
                  />
                )}
              />
            </div>
          )}

          <div className="prorank-form-field" style={{ marginBottom: '32px' }}>
            <Controller
              name="email"
              control={control}
              render={({ field }) => (
                <div>
                  <TextControl
                    label={
                      <span style={{ 
                        fontSize: '14px', 
                        fontWeight: '600',
                        color: '#374151'
                      }}>
                        {__('Email (optional)', 'prorank-seo')}
                      </span>
                    }
                    value={field.value}
                    onChange={field.onChange}
                    type="email"
                    help={
                      <span style={{ 
                        fontSize: '12px',
                        color: '#6b7280'
                      }}>
                        {__("If you'd like us to follow up with you", 'prorank-seo')}
                      </span>
                    }
                    __nextHasNoMarginBottom
                    __next40pxDefaultSize
                    style={{
                      borderRadius: '6px'
                    }}
                  />
                  {errors.email && (
                    <span className="prorank-error-message" style={{
                      color: '#ef4444',
                      fontSize: '12px',
                      marginTop: '4px',
                      display: 'block'
                    }}>{errors.email.message}</span>
                  )}
                </div>
              )}
            />
          </div>

          <div className="prorank-form-actions" style={{
            paddingTop: '8px',
            borderTop: '1px solid #e5e7eb',
            display: 'flex',
            justifyContent: 'flex-start',
            gap: '12px'
          }}>
            <ProrankButton
              variant="primary"
              type="submit"
              disabled={isSubmitting || submitFeedback.isPending}
              aria-busy={isSubmitting || submitFeedback.isPending}
              style={{ 
                minWidth: '160px',
                padding: '10px 24px',
                fontSize: '14px',
                fontWeight: '600',
                borderRadius: '6px'
              }}
            >
              {(isSubmitting || submitFeedback.isPending) ? (
                <React.Fragment>
                  <Spinner />
                  <span aria-live="polite" style={{ marginLeft: '8px' }}>{__('Submitting…', 'prorank-seo')}</span>
                </React.Fragment>
              ) : (
                <>📤 {__('Submit Feedback', 'prorank-seo')}</>
              )}
            </ProrankButton>
            
            {Object.keys(errors).length > 0 && (
              <span style={{
                display: 'flex',
                alignItems: 'center',
                color: '#ef4444',
                fontSize: '13px'
              }}>
                ⚠️ {__('Please fix the errors above', 'prorank-seo')}
              </span>
            )}
          </div>
        </form>
      </div>
    </div>

    {/* Error handling for feedback list */}
    {feedbackError && (
        <div className="prorank-card" style={{ marginTop: '24px' }}>
          <div className="prorank-alert prorank-alert-error">
            <p>{__('Failed to load recent feedback.', 'prorank-seo')}</p>
            <ProrankButton 
              variant="secondary" 
              onClick={() => refetchFeedback()}
              style={{ marginTop: '8px' }}
            >
              {__('Retry', 'prorank-seo')}
            </ProrankButton>
          </div>
        </div>
      )}

      {/* Loading state for recent feedback */}
      {isLoadingFeedback && (
        <div className="prorank-card" style={{ marginTop: '24px' }}>
          <div className="prorank-loading-state" style={{ display: 'flex', alignItems: 'center', gap: '12px' }}>
            <Spinner />
            <span className="prorank-loading-text">{__('Loading recent feedback...', 'prorank-seo')}</span>
          </div>
        </div>
      )}

      {/* Display recent feedback if available */}
      {!feedbackError && !isLoadingFeedback && feedbackData && feedbackData.feedback && feedbackData.feedback.length > 0 && (
        <div className="prorank-card" style={{ marginTop: '24px' }}>
          <div className="prorank-card-header">
            <h3 className="prorank-card-title" id="recent-feedback-title">{__('Your Recent Feedback', 'prorank-seo')}</h3>
            <p className="prorank-card-description">
              {__('Track the status of your submitted feedback and feature requests', 'prorank-seo')}
            </p>
          </div>
          <div 
            className="prorank-card-body prorank-feedback-list"
            role="list"
            aria-labelledby="recent-feedback-title"
          >
            {feedbackData.feedback.map((item) => (
              <div 
                key={item.id} 
                className="prorank-feedback-item" 
                role="listitem" 
                tabIndex={0}
                style={{
                  padding: '16px',
                  marginBottom: '12px',
                  background: '#f9fafb',
                  borderRadius: '8px',
                  border: '1px solid #e5e7eb',
                  transition: 'all 0.2s',
                  cursor: 'pointer'
                }}
                onMouseEnter={(e) => {
                  e.currentTarget.style.borderColor = '#2563eb';
                  e.currentTarget.style.boxShadow = '0 2px 4px rgba(0,0,0,0.05)';
                }}
                onMouseLeave={(e) => {
                  e.currentTarget.style.borderColor = '#e5e7eb';
                  e.currentTarget.style.boxShadow = 'none';
                }}
              >
                <div className="prorank-feedback-header" style={{ 
                  display: 'flex', 
                  justifyContent: 'space-between', 
                  alignItems: 'center',
                  marginBottom: '12px' 
                }}>
                  <div style={{ display: 'flex', gap: '8px', alignItems: 'center' }}>
                    <span style={{
                      padding: '4px 8px',
                      borderRadius: '4px',
                      fontSize: '11px',
                      fontWeight: '600',
                      textTransform: 'uppercase',
                      background: item.type === 'bug' ? '#fee2e2' : 
                                 item.type === 'feature' ? '#dbeafe' :
                                 item.type === 'support' ? '#fef3c7' : '#f3f4f6',
                      color: item.type === 'bug' ? '#dc2626' : 
                             item.type === 'feature' ? '#2563eb' :
                             item.type === 'support' ? '#d97706' : '#6b7280'
                    }}>
                      {item.type}
                    </span>
                    {item.priority && (
                      <span style={{
                        padding: '4px 8px',
                        borderRadius: '4px',
                        fontSize: '11px',
                        fontWeight: '500',
                        background: item.priority === 'high' ? '#fef2f2' :
                                   item.priority === 'medium' ? '#fff7ed' : '#f9fafb',
                        color: item.priority === 'high' ? '#ef4444' :
                               item.priority === 'medium' ? '#f97316' : '#6b7280'
                      }}>
                        {item.priority === 'high' ? '🔴' : item.priority === 'medium' ? '🟡' : '⚪'} {item.priority}
                      </span>
                    )}
                  </div>
                  <span style={{ 
                    fontSize: '12px', 
                    color: '#9ca3af' 
                  }}>
                    {item.createdAt}
                  </span>
                </div>
                
                <h4 style={{ 
                  fontSize: '15px',
                  fontWeight: '600',
                  color: '#1f2937',
                  marginBottom: '8px'
                }}>
                  {item.subject}
                </h4>
                
                {item.message && (
                  <p style={{
                    fontSize: '13px',
                    color: '#6b7280',
                    marginBottom: '12px',
                    lineHeight: '1.5',
                    maxHeight: '40px',
                    overflow: 'hidden',
                    textOverflow: 'ellipsis',
                    display: '-webkit-box',
                    WebkitLineClamp: 2,
                    WebkitBoxOrient: 'vertical'
                  }}>
                    {item.message}
                  </p>
                )}
                
                <div style={{ display: 'flex', justifyContent: 'space-between', alignItems: 'center' }}>
                  <span style={{
                    padding: '4px 12px',
                    borderRadius: '12px',
                    fontSize: '12px',
                    fontWeight: '500',
                    background: item.status === 'resolved' ? '#dcfce7' :
                               item.status === 'in_progress' ? '#fef3c7' :
                               item.status === 'reviewed' ? '#dbeafe' : '#f3f4f6',
                    color: item.status === 'resolved' ? '#16a34a' :
                           item.status === 'in_progress' ? '#d97706' :
                           item.status === 'reviewed' ? '#2563eb' : '#6b7280'
                  }}>
                    {item.status === 'resolved' ? '✅ ' : 
                     item.status === 'in_progress' ? '⚡ ' :
                     item.status === 'reviewed' ? '👀 ' : '⏳ '}
                    {item.status === 'resolved' ? __('Resolved', 'prorank-seo') :
                     item.status === 'in_progress' ? __('In Progress', 'prorank-seo') :
                     item.status === 'reviewed' ? __('Reviewed', 'prorank-seo') : __('Pending', 'prorank-seo')}
                  </span>
                  
                  {item.response && (
                    <span style={{
                      fontSize: '12px',
                      color: '#2563eb',
                      cursor: 'pointer',
                      textDecoration: 'underline'
                    }}>
                      {__('View Response', 'prorank-seo')}
                    </span>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </>
  );
};

export default FeedbackTab;