/* global wp */
import React, { useState, useEffect } from 'react';
import { Card, Button, Input, Select } from '../../../components/ui';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { useNotification } from '../../../contexts/NotificationContext';

const FeedbackTab = () => {
  const { showNotification } = useNotification();
  const [feedbackType, setFeedbackType] = useState('feature');
  const [subject, setSubject] = useState('');
  const [message, setMessage] = useState('');
  const [priority, setPriority] = useState('medium');
  const [email, setEmail] = useState('');
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [recentFeedback, setRecentFeedback] = useState([]);
  const [isLoadingFeedback, setIsLoadingFeedback] = useState(false);
  const [feedbackStats, setFeedbackStats] = useState([]);

  const feedbackTypes = [
    { label: __('Feature Request', 'prorank-seo'), value: 'feature' },
    { label: __('Bug Report', 'prorank-seo'), value: 'bug' },
    { label: __('General Feedback', 'prorank-seo'), value: 'general' },
    { label: __('Support Request', 'prorank-seo'), value: 'support' },
  ];

  const priorities = [
    { label: __('Low', 'prorank-seo'), value: 'low' },
    { label: __('Medium', 'prorank-seo'), value: 'medium' },
    { label: __('High', 'prorank-seo'), value: 'high' },
  ];

  const handleSubmit = async () => {
    if (!subject.trim() || !message.trim()) {
      showNotification(__('Please fill in all required fields', 'prorank-seo'), 'error');
      return;
    }

    setIsSubmitting(true);

    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/feedback/submit',
        method: 'POST',
        data: {
          type: feedbackType,
          subject,
          message,
          priority,
          email,
        },
      });

      if (response.success) {
        showNotification(__("Thank you for your feedback! We'll review it soon.", 'prorank-seo'), 'success');
        // Reset form
        setSubject('');
        setMessage('');
        setPriority('medium');
        setEmail('');
        // Refresh feedback list
        fetchRecentFeedback();
      } else {
        showNotification(response.message || __('Failed to submit feedback', 'prorank-seo'), 'error');
      }
    } catch (error) {
      showNotification(error.message || __('Failed to submit feedback', 'prorank-seo'), 'error');
    } finally {
      setIsSubmitting(false);
    }
  };

  const getTemplateMessage = (type) => {
    switch (type) {
      case 'bug':
        return __(
          'Steps to reproduce:\n1. \n2. \n3. \n\nExpected behavior:\n\nActual behavior:\n\nBrowser/Device:',
          'prorank-seo'
        );
      case 'feature':
        return __('Feature description:\n\nUse case:\n\nExpected benefit:', 'prorank-seo');
      default:
        return '';
    }
  };

  React.useEffect(() => {
    const template = getTemplateMessage(feedbackType);
    if (template && !message) {
      setMessage(template);
    }
  }, [feedbackType]);

  // Fetch recent feedback on component mount
  useEffect(() => {
    fetchRecentFeedback();
  }, []);

  const fetchRecentFeedback = async () => {
    setIsLoadingFeedback(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/feedback/list?type=feature&limit=5',
        method: 'GET',
      });

      if (response.data) {
        setRecentFeedback(response.data.feedback || []);
        setFeedbackStats(response.data.stats || []);
      }
    } catch (error) {
      // Silently handle error - feedback list is optional
    } finally {
      setIsLoadingFeedback(false);
    }
  };

  return (
    <div className="prorank-feedback-tab">
      <div className="prorank-section-header">
        <h2 className="prorank-section-title">
          {__('Feedback & Feature Requests', 'prorank-seo')}
        </h2>
        <p className="prorank-section-description">
          {__(
            'Help shape the future of ProRank SEO by sharing your ideas and reporting issues',
            'prorank-seo'
          )}
        </p>
      </div>

      <div className="prorank-support-grid">
        <Card className="prorank-card--modern">
          <div className="prorank-card-header">
            <h3 className="prorank-card-title">
              {__('Submit Feedback', 'prorank-seo')}
            </h3>
          </div>
          <div className="prorank-card-body">
            <p className="prorank-card-description">
              {__(
                'We value your feedback! Help us improve ProRank SEO by sharing your thoughts, reporting bugs, or requesting new features.',
                'prorank-seo'
              )}
            </p>

            <Select
              label={__('Feedback Type', 'prorank-seo')}
              value={feedbackType}
              options={feedbackTypes}
              onChange={(value) => setFeedbackType(value)}
            />

            <Input
              label={__('Subject *', 'prorank-seo')}
              value={subject}
              onChange={(e) => setSubject(e.target.value)}
              placeholder={__('Brief description of your feedback', 'prorank-seo')}
            />

            <div className="form-group">
              <label>{__('Message *', 'prorank-seo')}</label>
              <textarea
                value={message}
                onChange={(e) => setMessage(e.target.value)}
                rows={8}
                className="prorank-textarea"
              />
              <p className="help-text">{__('Please be as detailed as possible', 'prorank-seo')}</p>
            </div>

            {feedbackType === 'bug' && (
              <Select
                label={__('Priority', 'prorank-seo')}
                value={priority}
                options={priorities}
                onChange={(value) => setPriority(value)}
              />
            )}

            <Input
              label={__('Email (optional)', 'prorank-seo')}
              value={email}
              onChange={(e) => setEmail(e.target.value)}
              type="email"
              help={__("If you'd like us to follow up with you", 'prorank-seo')}
            />

            <div className="prorank-settings-actions">
              <Button
                variant="primary"
                onClick={handleSubmit}
                disabled={isSubmitting || !subject.trim() || !message.trim()}
              >
                {isSubmitting ? __('Submitting…', 'prorank-seo') : __('Submit Feedback', 'prorank-seo')}
              </Button>
            </div>
          </div>
        </Card>

      </div>

    </div>
  );
};

export default FeedbackTab;
