/**
 * Issues Page Component
 * 
 * Organized, filterable list of all SEO issues
 */

import React, { useState, useMemo } from 'react';
import { __, sprintf } from '@wordpress/i18n';
import {
  Card,
  CardBody,
  CardHeader,
  Button,
  Spinner,
  TextControl,
  SelectControl,
  Notice,
  Icon,
  __experimentalHeading as Heading,
  __experimentalText as Text,
  Flex,
  FlexItem,
  FlexBlock,
} from '@wordpress/components';
import { check, warning, info, closeSmall, arrowRight, external } from '@wordpress/icons';

const IssuesPage = ({
  issues = [],
  isLoading = false,
  auditStatus = null,
  onFixIssue,
  onIgnoreIssue,
}) => {
  const [searchTerm, setSearchTerm] = useState('');
  const [severityFilter, setSeverityFilter] = useState('all');
  const [categoryFilter, setCategoryFilter] = useState('all');
  const [expandedIssue, setExpandedIssue] = useState(null);
  const [sortBy, setSortBy] = useState('severity');

  // Severity order for sorting
  const severityOrder = { critical: 0, high: 1, medium: 2, low: 3, warning: 4 };

  // Get unique categories from issues
  const categories = useMemo(() => {
    const cats = new Set(issues.map(i => i.display_category || i.category || 'other'));
    return Array.from(cats);
  }, [issues]);

  // Filter and sort issues
  const filteredIssues = useMemo(() => {
    let filtered = [...issues];

    // Search filter
    if (searchTerm) {
      const term = searchTerm.toLowerCase();
      filtered = filtered.filter(issue =>
        issue.title?.toLowerCase().includes(term) ||
        issue.description?.toLowerCase().includes(term) ||
        issue.url?.toLowerCase().includes(term)
      );
    }

    // Severity filter
    if (severityFilter !== 'all') {
      filtered = filtered.filter(issue => issue.severity === severityFilter);
    }

    // Category filter
    if (categoryFilter !== 'all') {
      filtered = filtered.filter(issue => 
        (issue.display_category || issue.category) === categoryFilter
      );
    }

    // Sort
    filtered.sort((a, b) => {
      if (sortBy === 'severity') {
        return (severityOrder[a.severity] || 5) - (severityOrder[b.severity] || 5);
      }
      if (sortBy === 'category') {
        return (a.display_category || '').localeCompare(b.display_category || '');
      }
      return 0;
    });

    return filtered;
  }, [issues, searchTerm, severityFilter, categoryFilter, sortBy]);

  // Group issues by type for summary
  const issueSummary = useMemo(() => {
    const summary = { critical: 0, high: 0, medium: 0, low: 0, warning: 0 };
    issues.forEach(issue => {
      if (summary[issue.severity] !== undefined) {
        summary[issue.severity]++;
      }
    });
    return summary;
  }, [issues]);

  // Group filtered issues by category
  const groupedIssues = useMemo(() => {
    const groups = {};
    filteredIssues.forEach(issue => {
      const cat = issue.display_category || issue.category || 'other';
      if (!groups[cat]) {
        groups[cat] = [];
      }
      groups[cat].push(issue);
    });
    return groups;
  }, [filteredIssues]);

  const getSeverityColor = (severity) => {
    const colors = {
      critical: '#dc2626',
      high: '#ea580c',
      medium: '#d97706',
      low: '#2563eb',
      warning: '#6b7280',
    };
    return colors[severity] || '#6b7280';
  };

  const getSeverityBgColor = (severity) => {
    const colors = {
      critical: '#fef2f2',
      high: '#fff7ed',
      medium: '#fffbeb',
      low: '#eff6ff',
      warning: '#f9fafb',
    };
    return colors[severity] || '#f9fafb';
  };

  const getCategoryLabel = (cat) => {
    const labels = {
      technical_seo: __('Technical SEO', 'prorank-seo'),
      on_page_seo: __('On-Page SEO', 'prorank-seo'),
      performance: __('Performance', 'prorank-seo'),
      security: __('Security', 'prorank-seo'),
      content: __('Content', 'prorank-seo'),
    };
    return labels[cat] || cat;
  };

  if (auditStatus?.state !== 'completed') {
    return (
      <Card>
        <CardBody>
          <div style={{ textAlign: 'center', padding: '40px 20px' }}>
            <Icon icon={info} size={48} style={{ color: '#6b7280', marginBottom: '16px' }} />
            <Heading level={3} style={{ marginBottom: '8px' }}>
              {__('No Audit Data', 'prorank-seo')}
            </Heading>
            <Text>{__('Run an audit to see issues.', 'prorank-seo')}</Text>
          </div>
        </CardBody>
      </Card>
    );
  }

  return (
    <div className="prorank-issues-page">
      {/* Summary Cards */}
      <div style={{ display: 'grid', gridTemplateColumns: 'repeat(5, 1fr)', gap: '16px', marginBottom: '24px' }}>
        {[
          { key: 'critical', label: __('Critical', 'prorank-seo'), color: '#dc2626', bg: '#fef2f2' },
          { key: 'high', label: __('High', 'prorank-seo'), color: '#ea580c', bg: '#fff7ed' },
          { key: 'medium', label: __('Medium', 'prorank-seo'), color: '#d97706', bg: '#fffbeb' },
          { key: 'low', label: __('Low', 'prorank-seo'), color: '#2563eb', bg: '#eff6ff' },
          { key: 'warning', label: __('Warnings', 'prorank-seo'), color: '#6b7280', bg: '#f9fafb' },
        ].map(({ key, label, color, bg }) => (
          <Card 
            key={key} 
            style={{ 
              backgroundColor: bg, 
              border: `1px solid ${color}20`,
              cursor: 'pointer',
              transition: 'transform 0.2s, box-shadow 0.2s',
            }}
            onClick={() => setSeverityFilter(severityFilter === key ? 'all' : key)}
          >
            <CardBody style={{ padding: '16px', textAlign: 'center' }}>
              <div style={{ fontSize: '32px', fontWeight: '700', color }}>
                {issueSummary[key]}
              </div>
              <div style={{ fontSize: '14px', color: '#4b5563', marginTop: '4px' }}>
                {label}
              </div>
              {severityFilter === key && (
                <div style={{ marginTop: '8px', fontSize: '12px', color }}>
                  ✓ {__('Filtered', 'prorank-seo')}
                </div>
              )}
            </CardBody>
          </Card>
        ))}
      </div>

      {/* Filters Bar */}
      <Card style={{ marginBottom: '24px' }}>
        <CardBody>
          <Flex style={{ gap: '16px', flexWrap: 'wrap', alignItems: 'flex-end' }}>
            <FlexBlock>
              <TextControl
                label={__('Search Issues', 'prorank-seo')}
                value={searchTerm}
                onChange={setSearchTerm}
                placeholder={__('Search by title, description, or URL...', 'prorank-seo')}
                __next40pxDefaultSize
                __nextHasNoMarginBottom
              />
            </FlexBlock>
            <FlexItem>
              <SelectControl
                label={__('Severity', 'prorank-seo')}
                value={severityFilter}
                options={[
                  { label: __('All Severities', 'prorank-seo'), value: 'all' },
                  { label: __('Critical', 'prorank-seo'), value: 'critical' },
                  { label: __('High', 'prorank-seo'), value: 'high' },
                  { label: __('Medium', 'prorank-seo'), value: 'medium' },
                  { label: __('Low', 'prorank-seo'), value: 'low' },
                  { label: __('Warning', 'prorank-seo'), value: 'warning' },
                ]}
                onChange={setSeverityFilter}
                __next40pxDefaultSize
                __nextHasNoMarginBottom
              />
            </FlexItem>
            <FlexItem>
              <SelectControl
                label={__('Category', 'prorank-seo')}
                value={categoryFilter}
                options={[
                  { label: __('All Categories', 'prorank-seo'), value: 'all' },
                  ...categories.map(cat => ({ 
                    label: getCategoryLabel(cat), 
                    value: cat 
                  }))
                ]}
                onChange={setCategoryFilter}
                __next40pxDefaultSize
                __nextHasNoMarginBottom
              />
            </FlexItem>
            <FlexItem>
              <SelectControl
                label={__('Sort By', 'prorank-seo')}
                value={sortBy}
                options={[
                  { label: __('Severity', 'prorank-seo'), value: 'severity' },
                  { label: __('Category', 'prorank-seo'), value: 'category' },
                ]}
                onChange={setSortBy}
                __next40pxDefaultSize
                __nextHasNoMarginBottom
              />
            </FlexItem>
          </Flex>
        </CardBody>
      </Card>

      {/* Results Count */}
      <div style={{ marginBottom: '16px', color: '#6b7280' }}>
        {sprintf(__('Showing %d of %d issues', 'prorank-seo'), filteredIssues.length, issues.length)}
      </div>

      {/* Loading State */}
      {isLoading && (
        <Card>
          <CardBody style={{ textAlign: 'center', padding: '40px' }}>
            <Spinner />
            <Text style={{ marginTop: '16px' }}>{__('Loading issues...', 'prorank-seo')}</Text>
          </CardBody>
        </Card>
      )}

      {/* No Results */}
      {!isLoading && filteredIssues.length === 0 && (
        <Card>
          <CardBody>
            <Notice status="success" isDismissible={false}>
              <p>
                {issues.length === 0 
                  ? __('No issues found! Your site is in great shape.', 'prorank-seo')
                  : __('No issues match your filters.', 'prorank-seo')
                }
              </p>
            </Notice>
          </CardBody>
        </Card>
      )}

      {/* Grouped Issues */}
      {!isLoading && Object.entries(groupedIssues).map(([category, categoryIssues]) => (
        <Card key={category} style={{ marginBottom: '16px' }}>
          <CardHeader style={{ 
            backgroundColor: '#f8fafc', 
            borderBottom: '1px solid #e2e8f0',
            padding: '12px 16px',
          }}>
            <Flex justify="space-between" align="center">
              <Heading level={4} style={{ margin: 0 }}>
                {getCategoryLabel(category)}
              </Heading>
              <span style={{ 
                backgroundColor: '#e2e8f0', 
                padding: '4px 12px', 
                borderRadius: '12px',
                fontSize: '13px',
                fontWeight: '500',
              }}>
                {categoryIssues.length} {__('issues', 'prorank-seo')}
              </span>
            </Flex>
          </CardHeader>
          <CardBody style={{ padding: 0 }}>
            {categoryIssues.map((issue, index) => (
              <div 
                key={issue.id || index}
                style={{ 
                  borderBottom: index < categoryIssues.length - 1 ? '1px solid #e2e8f0' : 'none',
                }}
              >
                {/* Issue Row */}
                <div 
                  style={{ 
                    padding: '16px',
                    cursor: 'pointer',
                    backgroundColor: expandedIssue === issue.id ? '#f8fafc' : 'transparent',
                    transition: 'background-color 0.2s',
                  }}
                  onClick={() => setExpandedIssue(expandedIssue === issue.id ? null : issue.id)}
                >
                  <Flex align="center" gap={3}>
                    {/* Severity Badge */}
                    <div 
                      style={{ 
                        padding: '4px 10px',
                        borderRadius: '4px',
                        backgroundColor: getSeverityBgColor(issue.severity),
                        color: getSeverityColor(issue.severity),
                        fontSize: '12px',
                        fontWeight: '600',
                        textTransform: 'uppercase',
                        minWidth: '70px',
                        textAlign: 'center',
                      }}
                    >
                      {issue.severity}
                    </div>
                    
                    {/* Issue Title & URL */}
                    <FlexBlock>
                      <div style={{ fontWeight: '500', marginBottom: '4px' }}>
                        {issue.title || issue.message || issue.type}
                      </div>
                      {issue.url && (
                        <div style={{ fontSize: '13px', color: '#6b7280' }}>
                          <a 
                            href={issue.url} 
                            target="_blank" 
                            rel="noopener noreferrer"
                            onClick={(e) => e.stopPropagation()}
                            style={{ color: '#2563eb', textDecoration: 'none' }}
                          >
                            {issue.url.length > 60 ? issue.url.substring(0, 60) + '...' : issue.url}
                            <Icon icon={external} size={12} style={{ marginLeft: '4px' }} />
                          </a>
                        </div>
                      )}
                    </FlexBlock>

                    {/* Expand Icon */}
                    <div style={{ 
                      transform: expandedIssue === issue.id ? 'rotate(90deg)' : 'rotate(0deg)',
                      transition: 'transform 0.2s',
                      color: '#6b7280',
                    }}>
                      <Icon icon={arrowRight} size={20} />
                    </div>
                  </Flex>
                </div>

                {/* Expanded Details */}
                {expandedIssue === issue.id && (
                  <div style={{ 
                    padding: '0 16px 16px 16px',
                    backgroundColor: '#f8fafc',
                  }}>
                    <div style={{ 
                      padding: '16px',
                      backgroundColor: 'white',
                      borderRadius: '8px',
                      border: '1px solid #e2e8f0',
                    }}>
                      {/* Description */}
                      {issue.description && (
                        <div style={{ marginBottom: '16px' }}>
                          <strong style={{ display: 'block', marginBottom: '4px' }}>
                            {__('Description', 'prorank-seo')}
                          </strong>
                          <p style={{ margin: 0, color: '#4b5563' }}>{issue.description}</p>
                        </div>
                      )}

                      {/* Current vs Recommended */}
                      {(issue.current_value || issue.recommended_value) && (
                        <div style={{ 
                          display: 'grid', 
                          gridTemplateColumns: '1fr 1fr', 
                          gap: '16px',
                          marginBottom: '16px',
                        }}>
                          {issue.current_value && (
                            <div style={{ 
                              padding: '12px', 
                              backgroundColor: '#fef2f2', 
                              borderRadius: '6px',
                              border: '1px solid #fecaca',
                            }}>
                              <strong style={{ display: 'block', marginBottom: '4px', color: '#dc2626' }}>
                                {__('Current', 'prorank-seo')}
                              </strong>
                              <code style={{ fontSize: '13px', wordBreak: 'break-all' }}>
                                {issue.current_value}
                              </code>
                            </div>
                          )}
                          {issue.recommended_value && (
                            <div style={{ 
                              padding: '12px', 
                              backgroundColor: '#f0fdf4', 
                              borderRadius: '6px',
                              border: '1px solid #bbf7d0',
                            }}>
                              <strong style={{ display: 'block', marginBottom: '4px', color: '#16a34a' }}>
                                {__('Recommended', 'prorank-seo')}
                              </strong>
                              <code style={{ fontSize: '13px', wordBreak: 'break-all' }}>
                                {issue.recommended_value}
                              </code>
                            </div>
                          )}
                        </div>
                      )}

                      {/* How to Fix */}
                      {issue.how_to_fix && (
                        <div style={{ 
                          padding: '12px 16px',
                          backgroundColor: '#eff6ff',
                          borderRadius: '6px',
                          border: '1px solid #bfdbfe',
                          marginBottom: '16px',
                        }}>
                          <strong style={{ display: 'block', marginBottom: '8px', color: '#1d4ed8' }}>
                            {__('How to Fix', 'prorank-seo')}
                          </strong>
                          <p style={{ margin: 0, color: '#1e40af' }}>{issue.how_to_fix}</p>
                        </div>
                      )}

                      {/* Actions */}
                      <Flex gap={2}>
                        {issue.fix_available && onFixIssue && (
                          <Button 
                            variant="primary"
                            size="small"
                            onClick={() => onFixIssue(issue)}
                          >
                            {__('Apply Fix', 'prorank-seo')}
                          </Button>
                        )}
                        {onIgnoreIssue && (
                          <Button 
                            variant="secondary"
                            size="small"
                            onClick={() => onIgnoreIssue(issue)}
                          >
                            {__('Ignore', 'prorank-seo')}
                          </Button>
                        )}
                        {issue.reference && (
                          <Button 
                            variant="tertiary"
                            size="small"
                            href={issue.reference}
                            target="_blank"
                          >
                            {__('Learn More', 'prorank-seo')}
                            <Icon icon={external} size={14} style={{ marginLeft: '4px' }} />
                          </Button>
                        )}
                      </Flex>
                    </div>
                  </div>
                )}
              </div>
            ))}
          </CardBody>
        </Card>
      ))}
    </div>
  );
};

export default IssuesPage;
