import React, { useState, useCallback } from 'react';
import { CheckCircleIcon, XCircleIcon, EyeSlashIcon, ArrowDownTrayIcon, FunnelIcon } from '@heroicons/react/24/outline';
import { apiClient } from '../../../utils/api';
import { toast } from 'react-toastify';

const IssueManagement = ({ issues = [], onIssuesUpdate, auditId }) => {
    const [selectedIssues, setSelectedIssues] = useState([]);
    const [isLoading, setIsLoading] = useState(false);
    const [filterSeverity, setFilterSeverity] = useState('all');
    const [filterType, setFilterType] = useState('all');
    const [showIgnored, setShowIgnored] = useState(false);
    const [sortBy, setSortBy] = useState('priority');

    // Handle issue selection
    const handleSelectIssue = (issueId) => {
        setSelectedIssues(prev => 
            prev.includes(issueId) 
                ? prev.filter(id => id !== issueId)
                : [...prev, issueId]
        );
    };

    // Select all issues
    const handleSelectAll = () => {
        if (selectedIssues.length === getFilteredIssues().length) {
            setSelectedIssues([]);
        } else {
            setSelectedIssues(getFilteredIssues().map(issue => issue.id));
        }
    };

    // Ignore selected issues
    const handleIgnoreIssues = async (reason = '') => {
        if (selectedIssues.length === 0) {
            toast.warning('Please select issues to ignore');
            return;
        }

        setIsLoading(true);
        try {
            const response = await apiClient.post('/audit/issues/ignore', {
                issue_ids: selectedIssues,
                ignore_reason: reason,
                audit_id: auditId
            });

            if (response.data.success) {
                toast.success(`${response.data.ignored_count} issues marked as ignored`);
                setSelectedIssues([]);
                if (onIssuesUpdate) {
                    onIssuesUpdate();
                }
            }
        } catch (error) {
            toast.error('Failed to ignore issues: ' + error.message);
        } finally {
            setIsLoading(false);
        }
    };

    // Restore ignored issues
    const handleRestoreIssues = async () => {
        if (selectedIssues.length === 0) {
            toast.warning('Please select issues to restore');
            return;
        }

        setIsLoading(true);
        try {
            const response = await apiClient.post('/audit/issues/restore', {
                issue_ids: selectedIssues,
                audit_id: auditId
            });

            if (response.data.success) {
                toast.success(`${response.data.restored_count} issues restored`);
                setSelectedIssues([]);
                if (onIssuesUpdate) {
                    onIssuesUpdate();
                }
            }
        } catch (error) {
            toast.error('Failed to restore issues: ' + error.message);
        } finally {
            setIsLoading(false);
        }
    };

    // Export audit report
    const handleExport = async (format = 'csv') => {
        setIsLoading(true);
        try {
            const response = await apiClient.get('/audit/export', {
                params: { format, audit_id: auditId },
                responseType: format === 'csv' ? 'blob' : 'json'
            });

            if (format === 'csv') {
                // Download CSV file
                const url = window.URL.createObjectURL(new Blob([response.data]));
                const link = document.createElement('a');
                link.href = url;
                link.setAttribute('download', `audit-report-${new Date().toISOString().split('T')[0]}.csv`);
                document.body.appendChild(link);
                link.click();
                link.remove();
                toast.success('CSV report downloaded');
            } else if (format === 'pdf') {
                // Open PDF URL in new tab
                if (response.data.pdf_url) {
                    window.open(response.data.pdf_url, '_blank');
                    toast.success('PDF report is being generated');
                }
            }
        } catch (error) {
            toast.error('Failed to export report: ' + error.message);
        } finally {
            setIsLoading(false);
        }
    };

    // Filter and sort issues
    const getFilteredIssues = useCallback(() => {
        let filtered = [...issues];

        // Filter by ignored status
        if (!showIgnored) {
            filtered = filtered.filter(issue => !issue.is_ignored);
        }

        // Filter by severity
        if (filterSeverity !== 'all') {
            filtered = filtered.filter(issue => issue.severity === filterSeverity);
        }

        // Filter by type
        if (filterType !== 'all') {
            filtered = filtered.filter(issue => issue.type === filterType);
        }

        // Sort issues
        filtered.sort((a, b) => {
            switch (sortBy) {
                case 'priority':
                    return (b.priority_score || 0) - (a.priority_score || 0);
                case 'severity':
                    const severityOrder = { critical: 5, high: 4, medium: 3, low: 2, warning: 1, notice: 0 };
                    return severityOrder[b.severity] - severityOrder[a.severity];
                case 'type':
                    return a.type.localeCompare(b.type);
                default:
                    return 0;
            }
        });

        return filtered;
    }, [issues, showIgnored, filterSeverity, filterType, sortBy]);

    // Get unique issue types for filter
    const issueTypes = [...new Set(issues.map(issue => issue.type))];

    // Get severity badge color
    const getSeverityColor = (severity) => {
        const colors = {
            critical: 'bg-red-100 text-red-800',
            high: 'bg-orange-100 text-orange-800',
            medium: 'bg-yellow-100 text-yellow-800',
            low: 'bg-blue-100 text-blue-800',
            warning: 'bg-gray-100 text-gray-800',
            notice: 'bg-green-100 text-green-800'
        };
        return colors[severity] || 'bg-gray-100 text-gray-800';
    };

    const filteredIssues = getFilteredIssues();

    return (
        <div className="issue-management">
            {/* Toolbar */}
            <div className="bg-white shadow-sm rounded-lg p-4 mb-4">
                <div className="flex flex-wrap items-center justify-between gap-4">
                    {/* Filters */}
                    <div className="flex items-center gap-3">
                        <FunnelIcon className="h-5 w-5 text-gray-400" />
                        
                        <select
                            value={filterSeverity}
                            onChange={(e) => setFilterSeverity(e.target.value)}
                            className="rounded-md border-gray-300 text-sm"
                        >
                            <option value="all">All Severities</option>
                            <option value="critical">Critical</option>
                            <option value="high">High</option>
                            <option value="medium">Medium</option>
                            <option value="low">Low</option>
                            <option value="warning">Warning</option>
                        </select>

                        <select
                            value={filterType}
                            onChange={(e) => setFilterType(e.target.value)}
                            className="rounded-md border-gray-300 text-sm"
                        >
                            <option value="all">All Types</option>
                            {issueTypes.map(type => (
                                <option key={type} value={type}>
                                    {type.replace(/_/g, ' ').charAt(0).toUpperCase() + type.slice(1)}
                                </option>
                            ))}
                        </select>

                        <select
                            value={sortBy}
                            onChange={(e) => setSortBy(e.target.value)}
                            className="rounded-md border-gray-300 text-sm"
                        >
                            <option value="priority">Sort by Priority</option>
                            <option value="severity">Sort by Severity</option>
                            <option value="type">Sort by Type</option>
                        </select>

                        <label className="flex items-center gap-2 text-sm">
                            <input
                                type="checkbox"
                                checked={showIgnored}
                                onChange={(value) => {
                                    const isChecked = typeof value === 'boolean' ? value : value?.target?.checked;
                                    setShowIgnored(Boolean(isChecked));
                                }}
                                className="rounded border-gray-300"
                            />
                            Show Ignored
                        </label>
                    </div>

                    {/* Actions */}
                    <div className="flex items-center gap-2">
                        {selectedIssues.length > 0 && (
                            <>
                                <span className="text-sm text-gray-500">
                                    {selectedIssues.length} selected
                                </span>
                                
                                {!showIgnored && (
                                    <button
                                        onClick={() => handleIgnoreIssues()}
                                        disabled={isLoading}
                                        className="inline-flex items-center px-3 py-1.5 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                                    >
                                        <EyeSlashIcon className="h-4 w-4 mr-1" />
                                        Ignore
                                    </button>
                                )}
                                
                                {showIgnored && (
                                    <button
                                        onClick={handleRestoreIssues}
                                        disabled={isLoading}
                                        className="inline-flex items-center px-3 py-1.5 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                                    >
                                        <CheckCircleIcon className="h-4 w-4 mr-1" />
                                        Restore
                                    </button>
                                )}
                            </>
                        )}

                        <div className="relative group">
                            <button
                                className="inline-flex items-center px-3 py-1.5 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                            >
                                <ArrowDownTrayIcon className="h-4 w-4 mr-1" />
                                Export
                            </button>
                            <div className="absolute right-0 mt-2 w-48 rounded-md shadow-lg bg-white ring-1 ring-black ring-opacity-5 hidden group-hover:block z-10">
                                <div className="py-1">
                                    <button
                                        onClick={() => handleExport('csv')}
                                        className="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100"
                                    >
                                        Export as CSV
                                    </button>
                                    <button
                                        onClick={() => handleExport('pdf')}
                                        className="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100"
                                    >
                                        Export as PDF
                                    </button>
                                    <button
                                        onClick={() => handleExport('json')}
                                        className="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100"
                                    >
                                        Export as JSON
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            {/* Issues List */}
            <div className="bg-white shadow-sm rounded-lg overflow-hidden">
                <table className="min-w-full divide-y divide-gray-200">
                    <thead className="bg-gray-50">
                        <tr>
                            <th className="px-6 py-3 text-left">
                                <input
                                    type="checkbox"
                                    checked={selectedIssues.length === filteredIssues.length && filteredIssues.length > 0}
                                    onChange={handleSelectAll}
                                    className="rounded border-gray-300"
                                />
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Severity
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Issue
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Type
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Priority
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Status
                            </th>
                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                Actions
                            </th>
                        </tr>
                    </thead>
                    <tbody className="bg-white divide-y divide-gray-200">
                        {filteredIssues.map((issue) => (
                            <tr key={issue.id} className={issue.is_ignored ? 'opacity-60' : ''}>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <input
                                        type="checkbox"
                                        checked={selectedIssues.includes(issue.id)}
                                        onChange={() => handleSelectIssue(issue.id)}
                                        className="rounded border-gray-300"
                                    />
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <span className={`px-2 py-1 text-xs font-medium rounded-full ${getSeverityColor(issue.severity)}`}>
                                        {issue.severity}
                                    </span>
                                </td>
                                <td className="px-6 py-4">
                                    <div className="text-sm text-gray-900">{issue.title}</div>
                                    <div className="text-sm text-gray-500">{issue.description}</div>
                                    {issue.affected_url && (
                                        <div className="text-xs text-gray-400 mt-1">
                                            URL: {issue.affected_url}
                                        </div>
                                    )}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                                    {issue.type.replace(/_/g, ' ')}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    <div className="text-sm text-gray-900">
                                        {issue.priority_score || 0}
                                    </div>
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap">
                                    {issue.is_ignored ? (
                                        <span className="text-xs text-gray-500">Ignored</span>
                                    ) : issue.is_fixed ? (
                                        <span className="text-xs text-green-600">Fixed</span>
                                    ) : (
                                        <span className="text-xs text-orange-600">Active</span>
                                    )}
                                </td>
                                <td className="px-6 py-4 whitespace-nowrap text-sm">
                                    {issue.fix_available && !issue.is_fixed && !issue.is_ignored && (
                                        <button
                                            className="text-indigo-600 hover:text-indigo-900"
                                            onClick={() => {/* Apply fix handler to be implemented */}}
                                        >
                                            Fix
                                        </button>
                                    )}
                                    {issue.how_to_fix && (
                                        <button
                                            className="ml-2 text-gray-600 hover:text-gray-900"
                                            onClick={() => alert(issue.how_to_fix)}
                                        >
                                            Help
                                        </button>
                                    )}
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
                
                {filteredIssues.length === 0 && (
                    <div className="text-center py-8 text-gray-500">
                        No issues found matching your filters
                    </div>
                )}
            </div>
        </div>
    );
};

export default IssueManagement;
