/**
 * Issue Detail Row Component
 *
 * Expandable row displaying issue details with fix options
 */

import React, { useState } from "react";
import { __, sprintf } from "@wordpress/i18n";
import { Button } from "../../../components/ui";
import {
  ChevronDownIcon,
  ChevronRightIcon,
  ExclamationCircleIcon,
  ExclamationTriangleIcon,
  InformationCircleIcon,
  WrenchScrewdriverIcon,
  EyeSlashIcon,
  ArrowTopRightOnSquareIcon,
} from "@heroicons/react/24/outline";

const IssueDetailRow = ({ issue, onApplyFix, onIgnore, isApplyingFix }) => {
  const [isExpanded, setIsExpanded] = useState(false);

  const getSeverityConfig = (severity) => {
    const configs = {
      critical: {
        icon: ExclamationCircleIcon,
        bgColor: "pr-bg-red-100",
        textColor: "pr-text-red-800",
        borderColor: "pr-border-red-200",
        label: __("Critical", "prorank-seo"),
      },
      high: {
        icon: ExclamationTriangleIcon,
        bgColor: "pr-bg-orange-100",
        textColor: "pr-text-orange-800",
        borderColor: "pr-border-orange-200",
        label: __("High", "prorank-seo"),
      },
      medium: {
        icon: ExclamationTriangleIcon,
        bgColor: "pr-bg-yellow-100",
        textColor: "pr-text-yellow-800",
        borderColor: "pr-border-yellow-200",
        label: __("Medium", "prorank-seo"),
      },
      low: {
        icon: InformationCircleIcon,
        bgColor: "pr-bg-blue-100",
        textColor: "pr-text-blue-800",
        borderColor: "pr-border-blue-200",
        label: __("Low", "prorank-seo"),
      },
      warning: {
        icon: InformationCircleIcon,
        bgColor: "pr-bg-gray-100",
        textColor: "pr-text-gray-800",
        borderColor: "pr-border-gray-200",
        label: __("Warning", "prorank-seo"),
      },
    };
    return configs[severity] || configs.low;
  };

  const severityConfig = getSeverityConfig(issue.severity);
  const SeverityIcon = severityConfig.icon;

  const truncateUrl = (url, maxLength = 50) => {
    if (!url) return "";
    if (url.length <= maxLength) return url;
    return url.substring(0, maxLength) + "...";
  };

  return (
    <div
      className={`issue-detail-row pr-border-b pr-border-gray-200 pr-transition-all pr-duration-200 ${
        isExpanded ? "pr-bg-gray-50" : "pr-bg-white hover:pr-bg-gray-50"
      }`}
    >
      {/* Collapsed Row */}
      <div
        className="pr-flex pr-items-center pr-gap-3 pr-p-4 pr-cursor-pointer"
        onClick={() => setIsExpanded(!isExpanded)}
      >
        {/* Expand/Collapse Icon */}
        <div className="pr-flex-shrink-0">
          {isExpanded ? (
            <ChevronDownIcon className="pr-w-5 pr-h-5 pr-text-gray-400" />
          ) : (
            <ChevronRightIcon className="pr-w-5 pr-h-5 pr-text-gray-400" />
          )}
        </div>

        {/* Severity Badge */}
        <span
          className={`pr-inline-flex pr-items-center pr-gap-1 pr-px-2 pr-py-1 pr-rounded-full pr-text-xs pr-font-medium ${severityConfig.bgColor} ${severityConfig.textColor}`}
        >
          <SeverityIcon className="pr-w-3 pr-h-3" />
          {severityConfig.label}
        </span>

        {/* Issue Title */}
        <div className="pr-flex-1 pr-min-w-0">
          <p className="pr-text-sm pr-font-medium pr-text-gray-900 pr-truncate">
            {issue.title}
          </p>
          {issue.url && (
            <p className="pr-text-xs pr-text-gray-500 pr-truncate">
              {truncateUrl(issue.url)}
            </p>
          )}
        </div>

        {/* Fix Available Indicator */}
        {issue.fix_available && (
          <span className="pr-inline-flex pr-items-center pr-gap-1 pr-px-2 pr-py-1 pr-bg-green-100 pr-text-green-700 pr-rounded pr-text-xs">
            <WrenchScrewdriverIcon className="pr-w-3 pr-h-3" />
            {__("Fix Available", "prorank-seo")}
          </span>
        )}
      </div>

      {/* Expanded Content */}
      {isExpanded && (
        <div className="pr-px-4 pr-pb-4 pr-pl-12 pr-space-y-4">
          {/* Description */}
          {issue.description && (
            <div>
              <h4 className="pr-text-xs pr-font-semibold pr-text-gray-500 pr-uppercase pr-tracking-wide pr-mb-1">
                {__("Description", "prorank-seo")}
              </h4>
              <p className="pr-text-sm pr-text-gray-700">{issue.description}</p>
            </div>
          )}

          {/* Affected URL */}
          {issue.url && (
            <div>
              <h4 className="pr-text-xs pr-font-semibold pr-text-gray-500 pr-uppercase pr-tracking-wide pr-mb-1">
                {__("Affected URL", "prorank-seo")}
              </h4>
              <a
                href={issue.url}
                target="_blank"
                rel="noopener noreferrer"
                className="pr-text-sm pr-text-blue-600 hover:pr-text-blue-800 pr-inline-flex pr-items-center pr-gap-1"
              >
                {issue.url}
                <ArrowTopRightOnSquareIcon className="pr-w-4 pr-h-4" />
              </a>
            </div>
          )}

          {/* Current vs Recommended Value */}
          {(issue.current_value || issue.recommended_value) && (
            <div className="pr-grid pr-grid-cols-2 pr-gap-4">
              {issue.current_value && (
                <div>
                  <h4 className="pr-text-xs pr-font-semibold pr-text-gray-500 pr-uppercase pr-tracking-wide pr-mb-1">
                    {__("Current Value", "prorank-seo")}
                  </h4>
                  <p className="pr-text-sm pr-text-red-600 pr-bg-red-50 pr-p-2 pr-rounded pr-break-words">
                    {issue.current_value}
                  </p>
                </div>
              )}
              {issue.recommended_value && (
                <div>
                  <h4 className="pr-text-xs pr-font-semibold pr-text-gray-500 pr-uppercase pr-tracking-wide pr-mb-1">
                    {__("Recommended", "prorank-seo")}
                  </h4>
                  <p className="pr-text-sm pr-text-green-600 pr-bg-green-50 pr-p-2 pr-rounded pr-break-words">
                    {issue.recommended_value}
                  </p>
                </div>
              )}
            </div>
          )}

          {/* How to Fix */}
          {issue.how_to_fix && (
            <div>
              <h4 className="pr-text-xs pr-font-semibold pr-text-gray-500 pr-uppercase pr-tracking-wide pr-mb-1">
                {__("How to Fix", "prorank-seo")}
              </h4>
              <p className="pr-text-sm pr-text-gray-700 pr-bg-blue-50 pr-p-3 pr-rounded pr-border pr-border-blue-200">
                {issue.how_to_fix}
              </p>
            </div>
          )}

          {/* Impact */}
          {issue.impact && (
            <div>
              <h4 className="pr-text-xs pr-font-semibold pr-text-gray-500 pr-uppercase pr-tracking-wide pr-mb-1">
                {__("Impact", "prorank-seo")}
              </h4>
              <p className="pr-text-sm pr-text-gray-600">{issue.impact}</p>
            </div>
          )}

          {/* Action Buttons */}
          <div className="pr-flex pr-gap-2 pr-pt-2">
            {issue.fix_available && onApplyFix && (
              <Button
                variant="primary"
                size="sm"
                onClick={(e) => {
                  e.stopPropagation();
                  onApplyFix(issue);
                }}
                disabled={isApplyingFix}
                className="pr-inline-flex pr-items-center pr-gap-1"
              >
                <WrenchScrewdriverIcon className="pr-w-4 pr-h-4" />
                {isApplyingFix
                  ? __("Applying...", "prorank-seo")
                  : __("Apply Fix", "prorank-seo")}
              </Button>
            )}
            {onIgnore && (
              <Button
                variant="secondary"
                size="sm"
                onClick={(e) => {
                  e.stopPropagation();
                  onIgnore(issue);
                }}
                className="pr-inline-flex pr-items-center pr-gap-1"
              >
                <EyeSlashIcon className="pr-w-4 pr-h-4" />
                {__("Ignore", "prorank-seo")}
              </Button>
            )}
            {issue.reference && (
              <a
                href={issue.reference}
                target="_blank"
                rel="noopener noreferrer"
                className="pr-inline-flex pr-items-center pr-gap-1 pr-px-3 pr-py-1.5 pr-text-sm pr-text-gray-600 hover:pr-text-gray-800 pr-border pr-border-gray-300 pr-rounded hover:pr-bg-gray-50"
                onClick={(e) => e.stopPropagation()}
              >
                <ArrowTopRightOnSquareIcon className="pr-w-4 pr-h-4" />
                {__("Learn More", "prorank-seo")}
              </a>
            )}
          </div>
        </div>
      )}
    </div>
  );
};

export default IssueDetailRow;
