/**
 * Enhanced Core Web Vitals Component with 2025 Features
 * 
 * Features:
 * - Real-time monitoring with web-vitals library
 * - Historical charts with Chart.js
 * - CrUX field data integration
 * - Performance budgets and alerts
 * - Multi-page monitoring
 * - Export functionality
 * - Automated recommendations
 * 
 * @package ProRank\SEO
 * @since   2.0.0
 */

import { useState, useEffect, useRef } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Line, Bar } from 'react-chartjs-2';
import {
  Chart as ChartJS,
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  BarElement,
  Title,
  Tooltip,
  Legend,
  Filler
} from 'chart.js';
import ProrankButton from '../../../components/ProrankButton';
import { ProrankCard, ProrankCardBody, ProrankCardHeader } from '../../../components/ProrankCard';
import { ProrankSpinner } from '../../../components/ProrankSpinner';
import { ProrankProgressBar } from '../../../components/ProrankProgressBar';
import { ProrankNotice } from '../../../components/ProrankNotice';
import ProrankToggleSlider from '../../../components/ProrankToggleSlider';
import { Flex, FlexItem, FlexBlock } from '../../../components/design-system/Flex';
import { useNotification } from '../../../contexts/NotificationContext';
import * as React from 'react';

// Register Chart.js components
ChartJS.register(
  CategoryScale,
  LinearScale,
  PointElement,
  LineElement,
  BarElement,
  Title,
  Tooltip,
  Legend,
  Filler
);

interface WebVitalMetric {
  name: string;
  value: number;
  rating: 'good' | 'needs-improvement' | 'poor';
  delta?: number;
  entries?: any[];
  id?: string;
  navigationType?: string;
}

interface CoreWebVitalsData {
  lcp: number | null;
  inp: number | null;
  cls: number | null;
  fcp: number | null;
  ttfb: number | null;
  score?: number;
  timestamp: string;
  url: string;
  device?: string;
}

interface VitalThresholds {
  good: number;
  poor: number;
}

interface PerformanceBudget {
  metric: string;
  threshold: number;
  enabled: boolean;
}

interface CrUXData {
  available: boolean;
  metrics?: {
    [key: string]: {
      p75: number;
      good: number;
      needs_improvement: number;
      poor: number;
    };
  };
}

const VITAL_THRESHOLDS: { [key: string]: VitalThresholds } = {
  lcp: { good: 2500, poor: 4000 },
  inp: { good: 200, poor: 500 },
  cls: { good: 0.1, poor: 0.25 },
  fcp: { good: 1800, poor: 3000 },
  ttfb: { good: 800, poor: 1800 }
};

const CoreWebVitalsEnhanced: React.FC = () => {
  const [vitalsData, setVitalsData] = useState<CoreWebVitalsData | null>(null);
  const [historicalData, setHistoricalData] = useState<CoreWebVitalsData[]>([]);
  const [cruxData, setCruxData] = useState<CrUXData | null>(null);
  const [loading, setLoading] = useState<boolean>(true);
  const [measuring, setMeasuring] = useState<boolean>(false);
  const [selectedMetric, setSelectedMetric] = useState<string>('lcp');
  const [selectedPeriod, setSelectedPeriod] = useState<number>(30);
  const [performanceBudgets, setPerformanceBudgets] = useState<PerformanceBudget[]>([]);
  const [budgetAlerts, setBudgetAlerts] = useState<string[]>([]);
  const [recommendations, setRecommendations] = useState<any[]>([]);
  const [autoMonitoring, setAutoMonitoring] = useState<boolean>(false);
  const [multiPageData, setMultiPageData] = useState<any[]>([]);
  const [showExport, setShowExport] = useState<boolean>(false);
  const { showNotification } = useNotification();
  const chartRef = useRef<any>(null);

  useEffect(() => {
    loadVitalsData();
    loadPerformanceBudgets();
    measureCurrentPageVitals();
    
    // Set up auto-monitoring interval if enabled
    if (autoMonitoring) {
      const interval = setInterval(() => {
        measureCurrentPageVitals();
      }, 60000); // Every minute
      
      return () => clearInterval(interval);
    }
  }, [autoMonitoring]);

  const loadVitalsData = async () => {
    try {
      const [vitalsResponse, historyResponse, cruxResponse, budgetResponse] = await Promise.all([
        apiFetch({ path: '/prorank-seo/v1/core-web-vitals' }),
        apiFetch({ path: `/prorank-seo/v1/core-web-vitals/history?days=${selectedPeriod}` }),
        apiFetch({ path: '/prorank-seo/v1/core-web-vitals/crux' }),
        apiFetch({ path: '/prorank-seo/v1/core-web-vitals/budget' })
      ]);

      setVitalsData(vitalsResponse as any);
      setHistoricalData((historyResponse as any).data || []);
      setCruxData(cruxResponse as any);
      
      const budgetData = budgetResponse as any;
      if (budgetData.alerts && budgetData.alerts.length > 0) {
        setBudgetAlerts(budgetData.alerts);
      }
      
      // Load recommendations
      if ((vitalsResponse as any).recommendations) {
        setRecommendations((vitalsResponse as any).recommendations);
      }
    } catch (error) {
      console.error('Failed to load Core Web Vitals data:', error);
      showNotification(__('Failed to load Core Web Vitals data', 'prorank-seo'), 'error');
    } finally {
      setLoading(false);
    }
  };

  const loadPerformanceBudgets = () => {
    // Load saved budgets from localStorage or API
    const savedBudgets = localStorage.getItem('prorank_performance_budgets');
    if (savedBudgets) {
      setPerformanceBudgets(JSON.parse(savedBudgets));
    } else {
      // Default budgets
      setPerformanceBudgets([
        { metric: 'lcp', threshold: 2500, enabled: true },
        { metric: 'inp', threshold: 200, enabled: true },
        { metric: 'cls', threshold: 0.1, enabled: true },
        { metric: 'fcp', threshold: 1800, enabled: false },
        { metric: 'ttfb', threshold: 800, enabled: false }
      ]);
    }
  };

  const measureCurrentPageVitals = () => {
    if (typeof window !== 'undefined' && 'PerformanceObserver' in window) {
      try {
        import(/* webpackMode: "eager" */ 'web-vitals').then(({ onLCP, onINP, onCLS, onFCP, onTTFB }) => {
          const vitals: Partial<CoreWebVitalsData> = {
            url: window.location.href,
            timestamp: new Date().toISOString(),
          };

          onLCP((metric) => {
            vitals.lcp = metric.value;
            updateVitalsDisplay(vitals);
            checkBudget('lcp', metric.value);
          });

          onINP((metric) => {
            vitals.inp = metric.value;
            updateVitalsDisplay(vitals);
            checkBudget('inp', metric.value);
          });

          onCLS((metric) => {
            vitals.cls = metric.value;
            updateVitalsDisplay(vitals);
            checkBudget('cls', metric.value);
          });

          onFCP((metric) => {
            vitals.fcp = metric.value;
            updateVitalsDisplay(vitals);
          });

          onTTFB((metric) => {
            vitals.ttfb = metric.value;
            updateVitalsDisplay(vitals);
          });
          
          // Save measurement to backend
          setTimeout(() => {
            if (vitals.lcp || vitals.inp || vitals.cls) {
              saveVitalsMeasurement(vitals);
            }
          }, 5000);
        }).catch(error => {
          console.warn('Web Vitals library not available:', error);
        });
      } catch (error) {
        console.error('Error measuring Core Web Vitals:', error);
      }
    }
  };

  const updateVitalsDisplay = (vitals: Partial<CoreWebVitalsData>) => {
    setVitalsData(prev => ({
      ...prev,
      ...vitals,
    } as CoreWebVitalsData));
  };

  const saveVitalsMeasurement = async (vitals: Partial<CoreWebVitalsData>) => {
    try {
      await apiFetch({
        path: '/prorank-seo/v1/core-web-vitals/save',
        method: 'POST',
        data: vitals
      });
    } catch (error) {
      console.error('Failed to save vitals measurement:', error);
    }
  };

  const checkBudget = (metric: string, value: number) => {
    const budget = performanceBudgets.find(b => b.metric === metric && b.enabled);
    if (budget && value > budget.threshold) {
      const alert = sprintf(
        __('%s exceeded budget: %s (threshold: %s)', 'prorank-seo'),
        metric.toUpperCase(),
        formatMetricValue(metric, value),
        formatMetricValue(metric, budget.threshold)
      );
      
      setBudgetAlerts(prev => [...prev, alert]);
      showNotification(alert, 'warning');
    }
  };

  const runFullAudit = async () => {
    setMeasuring(true);
    showNotification(__('Starting comprehensive Core Web Vitals audit...', 'prorank-seo'), 'info');

    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/core-web-vitals/audit',
        method: 'POST',
        data: {
          url: window.location.origin,
          pages: 10,
        }
      });

      const result = response as any;
      showNotification(
        sprintf(__('Audit completed! Analyzed %d pages', 'prorank-seo'), result.audited),
        'success'
      );
      
      // Reload data
      await loadVitalsData();
      
      // Load multi-page results
      if (result.results) {
        setMultiPageData(result.results);
      }
    } catch (error) {
      showNotification(__('Failed to run Core Web Vitals audit', 'prorank-seo'), 'error');
    } finally {
      setMeasuring(false);
    }
  };

  const exportData = (format: 'csv' | 'pdf' | 'json') => {
    const data = {
      current: vitalsData,
      history: historicalData,
      budgets: performanceBudgets,
      recommendations: recommendations,
      crux: cruxData
    };

    switch (format) {
      case 'json':
        downloadJSON(data);
        break;
      case 'csv':
        downloadCSV(historicalData);
        break;
      case 'pdf':
        generatePDFReport(data);
        break;
    }
    
    showNotification(sprintf(__('Data exported as %s', 'prorank-seo'), format.toUpperCase()), 'success');
  };

  const downloadJSON = (data: any) => {
    const blob = new Blob([JSON.stringify(data, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `core-web-vitals-${new Date().toISOString().split('T')[0]}.json`;
    a.click();
  };

  const downloadCSV = (data: CoreWebVitalsData[]) => {
    const headers = ['Timestamp', 'URL', 'LCP', 'INP', 'CLS', 'FCP', 'TTFB', 'Score'];
    const rows = data.map(row => [
      row.timestamp,
      row.url,
      row.lcp,
      row.inp,
      row.cls,
      row.fcp,
      row.ttfb,
      row.score || ''
    ]);
    
    const csv = [headers, ...rows].map(row => row.join(',')).join('\n');
    const blob = new Blob([csv], { type: 'text/csv' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = `core-web-vitals-${new Date().toISOString().split('T')[0]}.csv`;
    a.click();
  };

  const generatePDFReport = async (data: any) => {
    // This would typically use a library like jsPDF
    // For now, we'll create a simple HTML report that can be printed to PDF
    const reportWindow = window.open('', '_blank');
    if (reportWindow) {
      reportWindow.document.write(`
        <html>
          <head>
            <title>Core Web Vitals Report</title>
            <style>
              body { font-family: Arial, sans-serif; padding: 20px; }
              h1 { color: #2563eb; }
              .metric { margin: 20px 0; padding: 15px; border: 1px solid #e5e7eb; }
              .good { color: #10b981; }
              .needs-improvement { color: #f59e0b; }
              .poor { color: #ef4444; }
            </style>
          </head>
          <body>
            <h1>Core Web Vitals Report</h1>
            <p>Generated: ${new Date().toLocaleString()}</p>
            <div class="metric">
              <h2>Current Metrics</h2>
              <p>LCP: ${data.current?.lcp || 'N/A'}</p>
              <p>INP: ${data.current?.inp || 'N/A'}</p>
              <p>CLS: ${data.current?.cls || 'N/A'}</p>
            </div>
            <div class="metric">
              <h2>Recommendations</h2>
              ${data.recommendations.map((r: any) => `<p>• ${r.message || r}</p>`).join('')}
            </div>
          </body>
        </html>
      `);
      reportWindow.document.close();
      reportWindow.print();
    }
  };

  const getRating = (metric: string, value: number): 'good' | 'needs-improvement' | 'poor' => {
    const thresholds = VITAL_THRESHOLDS[metric];
    if (!thresholds) return 'poor';
    
    if (value <= thresholds.good) return 'good';
    if (value <= thresholds.poor) return 'needs-improvement';
    return 'poor';
  };

  const formatMetricValue = (metric: string, value: number | null | undefined): string => {
    if (value === null || value === undefined || Number.isNaN(value)) {
      return '—';
    }
    if (metric === 'cls') {
      return value.toFixed(3);
    }
    if (metric === 'lcp' || metric === 'inp' || metric === 'fcp' || metric === 'ttfb') {
      if (value >= 1000) {
        return `${(value / 1000).toFixed(2)}s`;
      }
      return `${Math.round(value)}ms`;
    }
    return value.toString();
  };

  const getChartData = () => {
    const labels = historicalData.map(d => 
      new Date(d.timestamp).toLocaleDateString()
    );
    
    const datasets = [
      {
        label: selectedMetric.toUpperCase(),
        data: historicalData.map(d => (d as any)[selectedMetric] || 0),
        borderColor: '#2563eb',
        backgroundColor: 'rgba(37, 99, 235, 0.1)',
        tension: 0.4,
        fill: true
      }
    ];
    
    // Add threshold lines
    const threshold = VITAL_THRESHOLDS[selectedMetric];
    if (threshold) {
      datasets.push({
        label: __('Good Threshold', 'prorank-seo'),
        data: new Array(labels.length).fill(threshold.good),
        borderColor: '#10b981',
        borderDash: [5, 5],
        pointRadius: 0,
        fill: false
      } as any);
      
      datasets.push({
        label: __('Poor Threshold', 'prorank-seo'),
        data: new Array(labels.length).fill(threshold.poor),
        borderColor: '#ef4444',
        borderDash: [5, 5],
        pointRadius: 0,
        fill: false
      } as any);
    }
    
    return { labels, datasets };
  };

  const chartOptions = {
    responsive: true,
    maintainAspectRatio: false,
    plugins: {
      legend: {
        position: 'top' as const,
      },
      title: {
        display: true,
        text: sprintf(__('%s Trend', 'prorank-seo'), selectedMetric.toUpperCase()),
      },
      tooltip: {
        callbacks: {
          label: (context: any) => {
            const value = context.parsed.y;
            return `${context.dataset.label}: ${formatMetricValue(selectedMetric, value)}`;
          }
        }
      }
    },
    scales: {
      y: {
        beginAtZero: true,
        title: {
          display: true,
          text: selectedMetric === 'cls' ? 'Score' : 'Time (ms)'
        }
      }
    }
  };

  if (loading) {
    return (
      <div className="prorank-loading-container">
        <ProrankSpinner size="large" />
        <p>{__('Loading Core Web Vitals...', 'prorank-seo')}</p>
      </div>
    );
  }

  return (
    <div className="prorank-page prorank-core-web-vitals-enhanced">
      {/* Performance Budget Alerts */}
      {budgetAlerts.length > 0 && (
        <ProrankNotice status="warning" isDismissible onDismiss={() => setBudgetAlerts([])}>
          <strong>{__('Performance Budget Alerts', 'prorank-seo')}</strong>
          <ul>
            {budgetAlerts.map((alert, index) => (
              <li key={index}>{alert}</li>
            ))}
          </ul>
        </ProrankNotice>
      )}

      {/* Main Dashboard Card */}
      <ProrankCard>
        <ProrankCardHeader>
          <Flex align="center" justify="space-between">
            <FlexItem>
              <h3>{__('Core Web Vitals Dashboard', 'prorank-seo')}</h3>
              <p className="prorank-text-secondary">
                {__('Real-time monitoring with INP (2024) and field data from CrUX', 'prorank-seo')}
              </p>
            </FlexItem>
            <FlexItem>
              <Flex gap={2}>
                <div className="prorank-toggle-row" style={{ display: 'inline-flex', alignItems: 'center', gap: '8px' }}>
                  <ProrankToggleSlider
                    checked={autoMonitoring}
                    onChange={setAutoMonitoring}
                  />
                  <label>{__('Auto Monitor', 'prorank-seo')}</label>
                </div>
                <ProrankButton
                  variant="secondary"
                  onClick={() => setShowExport(!showExport)}
                >
                  {__('Export', 'prorank-seo')}
                </ProrankButton>
                <ProrankButton
                  variant="primary"
                  onClick={runFullAudit}
                  disabled={measuring}
                >
                  {measuring && <ProrankSpinner size="small" />}
                  {__('Run Audit', 'prorank-seo')}
                </ProrankButton>
              </Flex>
            </FlexItem>
          </Flex>
        </ProrankCardHeader>
        <ProrankCardBody>
          {/* Export Options */}
          {showExport && (
            <div className="prorank-export-options prorank-mb-4">
              <Flex gap={2}>
                <ProrankButton size="small" onClick={() => exportData('json')}>
                  {__('Export JSON', 'prorank-seo')}
                </ProrankButton>
                <ProrankButton size="small" onClick={() => exportData('csv')}>
                  {__('Export CSV', 'prorank-seo')}
                </ProrankButton>
                <ProrankButton size="small" onClick={() => exportData('pdf')}>
                  {__('Generate PDF Report', 'prorank-seo')}
                </ProrankButton>
              </Flex>
            </div>
          )}

          {/* Current Metrics Grid */}
          <div className="prorank-vitals-grid prorank-grid-auto">
            {/* LCP Card */}
            <div className="prorank-vital-card prorank-metric-card">
              <h4>{__('LCP', 'prorank-seo')}</h4>
              <p className="prorank-text-secondary prorank-text-sm">
                {__('Largest Contentful Paint', 'prorank-seo')}
              </p>
              <div className="prorank-metric-value">
                {vitalsData?.lcp ? formatMetricValue('lcp', vitalsData.lcp) : '--'}
              </div>
              {vitalsData?.lcp && (
                <span className={`prorank-badge prorank-badge-${getRating('lcp', vitalsData.lcp)}`}>
                  {getRating('lcp', vitalsData.lcp)}
                </span>
              )}
            </div>

            {/* INP Card */}
            <div className="prorank-vital-card prorank-metric-card prorank-metric-card-highlighted">
              <h4>
                {__('INP', 'prorank-seo')}
                <span className="prorank-badge prorank-badge-new">
                  {__('2024', 'prorank-seo')}
                </span>
              </h4>
              <p className="prorank-text-secondary prorank-text-sm">
                {__('Interaction to Next Paint', 'prorank-seo')}
              </p>
              <div className="prorank-metric-value">
                {vitalsData?.inp ? formatMetricValue('inp', vitalsData.inp) : '--'}
              </div>
              {vitalsData?.inp && (
                <span className={`prorank-badge prorank-badge-${getRating('inp', vitalsData.inp)}`}>
                  {getRating('inp', vitalsData.inp)}
                </span>
              )}
            </div>

            {/* CLS Card */}
            <div className="prorank-vital-card prorank-metric-card">
              <h4>{__('CLS', 'prorank-seo')}</h4>
              <p className="prorank-text-secondary prorank-text-sm">
                {__('Cumulative Layout Shift', 'prorank-seo')}
              </p>
              <div className="prorank-metric-value">
                {vitalsData?.cls ? formatMetricValue('cls', vitalsData.cls) : '--'}
              </div>
              {vitalsData?.cls && (
                <span className={`prorank-badge prorank-badge-${getRating('cls', vitalsData.cls)}`}>
                  {getRating('cls', vitalsData.cls)}
                </span>
              )}
            </div>

            {/* Performance Score */}
            {vitalsData?.score && (
              <div className="prorank-vital-card prorank-metric-card">
                <h4>{__('Performance Score', 'prorank-seo')}</h4>
                <p className="prorank-text-secondary prorank-text-sm">
                  {__('Overall PageSpeed Score', 'prorank-seo')}
                </p>
                <div className="prorank-metric-value">
                  {vitalsData.score}
                </div>
                <ProrankProgressBar 
                  value={vitalsData.score} 
                  max={100}
                  variant={vitalsData.score >= 90 ? 'success' : vitalsData.score >= 50 ? 'warning' : 'error'}
                />
              </div>
            )}
          </div>

          {/* Historical Chart */}
          {historicalData.length > 0 && (
            <ProrankCard className="prorank-mt-4">
              <ProrankCardHeader>
                <Flex align="center" justify="space-between">
                  <h4>{__('Historical Trends', 'prorank-seo')}</h4>
                  <Flex gap={2}>
                    <select 
                      value={selectedMetric}
                      onChange={(e) => setSelectedMetric(e.target.value)}
                      className="prorank-select"
                    >
                      <option value="lcp">LCP</option>
                      <option value="inp">INP</option>
                      <option value="cls">CLS</option>
                      <option value="fcp">FCP</option>
                      <option value="ttfb">TTFB</option>
                    </select>
                    <select
                      value={selectedPeriod}
                      onChange={(e) => setSelectedPeriod(parseInt(e.target.value))}
                      className="prorank-select"
                    >
                      <option value={7}>{__('Last 7 days', 'prorank-seo')}</option>
                      <option value={30}>{__('Last 30 days', 'prorank-seo')}</option>
                      <option value={90}>{__('Last 90 days', 'prorank-seo')}</option>
                    </select>
                  </Flex>
                </Flex>
              </ProrankCardHeader>
              <ProrankCardBody>
                <div style={{ height: '300px' }}>
                  <Line ref={chartRef} data={getChartData()} options={chartOptions} />
                </div>
              </ProrankCardBody>
            </ProrankCard>
          )}

          {/* CrUX Field Data */}
          {cruxData?.available && (
            <ProrankCard className="prorank-mt-4">
              <ProrankCardHeader>
                <h4>{__('Chrome User Experience Report (Field Data)', 'prorank-seo')}</h4>
              </ProrankCardHeader>
              <ProrankCardBody>
                <div className="prorank-grid-3">
                  {Object.entries(cruxData.metrics || {}).map(([metric, data]) => (
                    <div key={metric} className="prorank-crux-metric">
                      <h5>{metric.toUpperCase()}</h5>
                      <div className="prorank-crux-bars">
                        <div className="prorank-crux-bar good" style={{ width: `${data.good * 100}%` }}>
                          {Math.round(data.good * 100)}%
                        </div>
                        <div className="prorank-crux-bar needs-improvement" style={{ width: `${data.needs_improvement * 100}%` }}>
                          {Math.round(data.needs_improvement * 100)}%
                        </div>
                        <div className="prorank-crux-bar poor" style={{ width: `${data.poor * 100}%` }}>
                          {Math.round(data.poor * 100)}%
                        </div>
                      </div>
                      <p className="prorank-text-sm prorank-text-secondary">
                        P75: {formatMetricValue(metric, data.p75)}
                      </p>
                    </div>
                  ))}
                </div>
              </ProrankCardBody>
            </ProrankCard>
          )}

          {/* Multi-Page Analysis */}
          {multiPageData.length > 0 && (
            <ProrankCard className="prorank-mt-4">
              <ProrankCardHeader>
                <h4>{__('Multi-Page Analysis', 'prorank-seo')}</h4>
              </ProrankCardHeader>
              <ProrankCardBody>
                <table className="prorank-table">
                  <thead>
                    <tr>
                      <th>{__('Page', 'prorank-seo')}</th>
                      <th>LCP</th>
                      <th>INP</th>
                      <th>CLS</th>
                      <th>{__('Score', 'prorank-seo')}</th>
                    </tr>
                  </thead>
                  <tbody>
                    {multiPageData.map((page, index) => (
                      <tr key={index}>
                        <td>{new URL(page.url).pathname}</td>
                        <td className={`prorank-badge-${getRating('lcp', page.vitals?.lab_data?.lcp?.value || 0)}`}>
                          {formatMetricValue('lcp', page.vitals?.lab_data?.lcp?.value || 0)}
                        </td>
                        <td className={`prorank-badge-${getRating('inp', page.vitals?.lab_data?.inp?.value || 0)}`}>
                          {formatMetricValue('inp', page.vitals?.lab_data?.inp?.value || 0)}
                        </td>
                        <td className={`prorank-badge-${getRating('cls', page.vitals?.lab_data?.cls?.value || 0)}`}>
                          {formatMetricValue('cls', page.vitals?.lab_data?.cls?.value || 0)}
                        </td>
                        <td>{page.vitals?.score ? Math.round(page.vitals.score * 100) : '--'}</td>
                      </tr>
                    ))}
                  </tbody>
                </table>
              </ProrankCardBody>
            </ProrankCard>
          )}

          {/* Recommendations */}
          {recommendations.length > 0 && (
            <ProrankCard className="prorank-mt-4">
              <ProrankCardHeader>
                <h4>{__('Automated Recommendations', 'prorank-seo')}</h4>
              </ProrankCardHeader>
              <ProrankCardBody>
                {recommendations.map((rec, index) => (
                  <ProrankNotice 
                    key={index}
                    status={rec.severity === 'high' ? 'error' : rec.severity === 'medium' ? 'warning' : 'info'}
                    isDismissible={false}
                  >
                    <strong>{rec.metric}</strong>: {rec.message}
                  </ProrankNotice>
                ))}
              </ProrankCardBody>
            </ProrankCard>
          )}

          {/* Performance Budgets */}
          <ProrankCard className="prorank-mt-4">
            <ProrankCardHeader>
              <h4>{__('Performance Budgets', 'prorank-seo')}</h4>
            </ProrankCardHeader>
            <ProrankCardBody>
              <div className="prorank-budgets-grid">
                {performanceBudgets.map((budget, index) => (
                  <div key={index} className="prorank-budget-item">
                    <Flex align="center" justify="space-between">
                      <FlexBlock>
                        <strong>{budget.metric.toUpperCase()}</strong>
                        <input
                          type="number"
                          value={budget.threshold}
                          onChange={(e) => {
                            const newBudgets = [...performanceBudgets];
                            newBudgets[index].threshold = parseFloat(e.target.value);
                            setPerformanceBudgets(newBudgets);
                            localStorage.setItem('prorank_performance_budgets', JSON.stringify(newBudgets));
                          }}
                          className="prorank-input prorank-ml-2"
                          style={{ width: '100px' }}
                        />
                      </FlexBlock>
                      <ProrankToggleSlider
                        checked={budget.enabled}
                        onChange={(checked) => {
                          const newBudgets = [...performanceBudgets];
                          newBudgets[index].enabled = checked;
                          setPerformanceBudgets(newBudgets);
                          localStorage.setItem('prorank_performance_budgets', JSON.stringify(newBudgets));
                        }}
                      />
                    </Flex>
                  </div>
                ))}
              </div>
            </ProrankCardBody>
          </ProrankCard>
        </ProrankCardBody>
      </ProrankCard>
    </div>
  );
};

export default CoreWebVitalsEnhanced;
