/**
 * Audit Progress Component
 * 
 * Visual representation of audit progress with animations
 */

import React from 'react';
import { __, sprintf } from '@wordpress/i18n';
import { ProgressBar } from '../../../components/ui';
import { GlobeAltIcon, MagnifyingGlassIcon, ShieldCheckIcon, ChartBarIcon, CheckIcon } from '@heroicons/react/24/outline';

const AuditProgress = ({ 
  state, 
  progress, 
  elapsedTime, 
  totalUrls, 
  checkedUrls,
  estimatedTime 
}) => {
  
  const getStateIcon = () => {
    switch (state) {
      case 'crawling':
        return GlobeAltIcon;
      case 'analyzing':
        return MagnifyingGlassIcon;
      case 'checking':
        return ShieldCheckIcon;
      case 'completed':
        return CheckIcon;
      default:
        return ChartBarIcon;
    }
  };
  
  const getStateMessage = () => {
    switch (state) {
      case 'crawling':
        return __('Discovering pages on your site...', 'prorank-seo');
      case 'analyzing':
        return __('Analyzing SEO issues...', 'prorank-seo');
      case 'checking':
        return __('Running comprehensive checks...', 'prorank-seo');
      case 'completed':
        return __('Audit completed successfully!', 'prorank-seo');
      default:
        return __('Preparing audit...', 'prorank-seo');
    }
  };
  
  const formatTime = (seconds) => {
    if (seconds < 60) {
      return `${seconds}s`;
    }
    const minutes = Math.floor(seconds / 60);
    const remainingSeconds = seconds % 60;
    return `${minutes}m ${remainingSeconds}s`;
  };
  
  const calculateEstimatedRemaining = () => {
    if (progress > 0 && elapsedTime > 0) {
      const totalEstimated = (elapsedTime / progress) * 100;
      const remaining = Math.round(totalEstimated - elapsedTime);
      return remaining > 0 ? remaining : 0;
    }
    return estimatedTime || 0;
  };
  
  const IconComponent = getStateIcon();

  return (
    <div className="audit-progress">
      <div className="audit-progress__header pr-transition-all pr-duration-300">
        <div className="audit-progress__icon-wrapper">
          <IconComponent
            className="audit-progress__icon pr-w-8 pr-h-8 pr-animate-spin"
          />
        </div>
        <div className="audit-progress__details">
          <h3 className="audit-progress__message pr-text-lg pr-font-semibold">
            {getStateMessage()}
          </h3>
          <div className="audit-progress__stats pr-flex pr-gap-4 pr-text-sm pr-text-gray-600 pr-mt-2">
            {checkedUrls > 0 && (
              <span className="audit-progress__stat">
                {sprintf(
                  __('%d of %d pages checked', 'prorank-seo'),
                  checkedUrls,
                  totalUrls || checkedUrls
                )}
              </span>
            )}
            {elapsedTime > 0 && (
              <span className="audit-progress__stat">
                {__('Elapsed:', 'prorank-seo')} {formatTime(elapsedTime)}
              </span>
            )}
            {state !== 'completed' && calculateEstimatedRemaining() > 0 && (
              <span className="audit-progress__stat">
                {__('Remaining:', 'prorank-seo')} ~{formatTime(calculateEstimatedRemaining())}
              </span>
            )}
          </div>
        </div>
      </div>

      <div className="audit-progress__bar-wrapper pr-mt-4">
        <ProgressBar
          value={progress || 0}
          max={100}
          className="audit-progress__bar"
        />
        <span className="audit-progress__percentage pr-text-sm pr-text-gray-600 pr-mt-1 pr-block">
          {Math.round(progress || 0)}%
        </span>
      </div>

      {state === 'crawling' && (
        <div className="audit-progress__discovery pr-mt-4 pr-transition-all pr-duration-300">
          <div className="audit-progress__discovery-animation pr-flex pr-gap-2 pr-justify-center">
            <span className="dot pr-w-2 pr-h-2 pr-bg-primary-500 pr-rounded-full pr-animate-pulse"></span>
            <span className="dot pr-w-2 pr-h-2 pr-bg-primary-500 pr-rounded-full pr-animate-pulse pr-transition-all pr-duration-300" style={{animationDelay: '0.2s'}}></span>
            <span className="dot pr-w-2 pr-h-2 pr-bg-primary-500 pr-rounded-full pr-animate-pulse pr-transition-all pr-duration-300" style={{animationDelay: '0.4s'}}></span>
          </div>
          <p className="audit-progress__discovery-text pr-text-center pr-text-sm pr-text-gray-600 pr-mt-2">
            {__('Discovering new URLs in real-time...', 'prorank-seo')}
          </p>
        </div>
      )}
    </div>
  );
};

export default AuditProgress;