/**
 * Audit Health Score Component
 *
 * Displays the overall site health score with visual indicators
 */

import React from 'react';
import { __ } from '@wordpress/i18n';
import { Icon, trendingUp, trendingDown } from '@wordpress/icons';

const AuditHealthScore = ({ score = 0, previousScore = 0, lastUpdated }) => {
  const getScoreColor = (score) => {
    if (score >= 80) return '#10b981'; // Green
    if (score >= 60) return '#f59e0b'; // Yellow
    return '#ef4444'; // Red
  };

  const getScoreLabel = (score) => {
    if (score >= 80) return __('Excellent', 'prorank-seo');
    if (score >= 60) return __('Good', 'prorank-seo');
    if (score >= 40) return __('Fair', 'prorank-seo');
    return __('Poor', 'prorank-seo');
  };

  const scoreDiff = score - previousScore;
  const hasImproved = scoreDiff > 0;
  
  // Calculate circle progress
  const radius = 45;
  const circumference = 2 * Math.PI * radius;
  const strokeDashoffset = circumference - (score / 100) * circumference;

  return (
    <div className="prorank-health-score">
      <div className="score-visualization">
        <div style={{ position: 'relative', width: '240px', height: '240px' }} className="pr-transition-all pr-duration-300">
          <svg
            width="240"
            height="240"
            style={{ transform: 'rotate(-90deg)' }}
            className="pr-transition-all pr-duration-300"
          >
            {/* Background circle */}
            <circle
              cx="120"
              cy="120"
              r={radius}
              stroke="#f3f4f6"
              strokeWidth="12"
              fill="none"
            />
            {/* Progress circle */}
            <circle
              cx="120"
              cy="120"
              r={radius}
              stroke={getScoreColor(score)}
              strokeWidth="12"
              fill="none"
              strokeDasharray={circumference}
              strokeDashoffset={strokeDashoffset}
              strokeLinecap="round"
              style={{
                transition: 'stroke-dashoffset 0.5s ease-in-out',
              }}
            />
          </svg>
          <div
            className="score-center pr-transition-all pr-duration-300"
            style={{
              position: 'absolute',
              top: '50%',
              left: '50%',
              transform: 'translate(-50%, -50%)',
              textAlign: 'center',
            }}
          >
            <span
              className="score-value pr-transition-all pr-duration-300"
              style={{
                display: 'block',
                fontSize: '3rem',
                fontWeight: 'bold',
                color: getScoreColor(score),
              }}
            >
              {score}
            </span>
            <span
              className="score-label pr-transition-all pr-duration-300"
              style={{
                display: 'block',
                fontSize: '0.875rem',
                color: '#6b7280',
                marginTop: '4px',
              }}
            >
              {getScoreLabel(score)}
            </span>
          </div>
        </div>
      </div>

      <div className="score-details">
        <h3>{__('Site Health Score', 'prorank-seo')}</h3>

        {scoreDiff !== 0 && (
          <div className={`score-trend ${hasImproved ? 'improved' : 'declined'}`}>
            <Icon icon={hasImproved ? trendingUp : trendingDown} />
            <span>
              {hasImproved ? '+' : ''}
              {scoreDiff} {__('from last audit', 'prorank-seo')}
            </span>
          </div>
        )}

        {lastUpdated && (
          <p className="last-updated">
            {__('Last updated:', 'prorank-seo')}{' '}
            {(() => {
              try {
                const date = new Date(lastUpdated);
                return isNaN(date.getTime()) ? __('Unknown', 'prorank-seo') : date.toLocaleString();
              } catch (e) {
                return __('Unknown', 'prorank-seo');
              }
            })()}
          </p>
        )}
      </div>
    </div>
  );
};

export default AuditHealthScore;