/**
 * Audit Category Card Component
 *
 * Displays audit results for a specific category
 */

import React from 'react';
import { __, sprintf } from '@wordpress/i18n';
import { Card, Button, ProgressBar } from '../../../components/ui';

const AuditCategoryCard = ({ category, results, onClick }) => {
  if (!results) {
    results = {
      total: category.checks?.length || 0,
      passed: 0,
      warnings: 0,
      critical: 0,
      loading: true,
    };
  }

  const score = results.total > 0 ? Math.round((results.passed / results.total) * 100) : 0;
  
  const getScoreColor = (score) => {
    if (score >= 80) return '#10b981';
    if (score >= 60) return '#f59e0b';
    return '#ef4444';
  };

  return (
    <Card className="audit-category-card pr-transition-all pr-duration-300">
      <div className="pr-p-6">
        <div className="category-header pr-flex pr-justify-between pr-items-center pr-mb-4">
          <h3 className="pr-text-lg pr-font-semibold">{category.title}</h3>
          <span className="category-score pr-text-2xl pr-font-bold pr-transition-all pr-duration-300" style={{ color: getScoreColor(score) }}>
            {score}%
          </span>
        </div>

        <p className="category-description pr-text-gray-600 pr-mb-6">{category.description}</p>

        {results.loading ? (
          <div className="category-loading pr-space-y-4">
            <ProgressBar value={0} max={100} />
            <p className="pr-text-center pr-text-gray-500">{__('Analyzing...', 'prorank-seo')}</p>
          </div>
        ) : (
          <>
            <div className="category-stats">
              <div className="stat-grid pr-grid pr-grid-cols-3 pr-gap-4 pr-mb-4 pr-transition-all pr-duration-300">
                <div className="stat-item pr-text-center">
                  <span className="stat-value pr-text-2xl pr-font-bold pr-text-success-500 pr-transition-all pr-duration-300">
                    {results.passed}
                  </span>
                  <span className="stat-label pr-block pr-text-xs pr-text-gray-600 pr-mt-1">
                    {__('Passed', 'prorank-seo')}
                  </span>
                </div>
                <div className="stat-item pr-text-center">
                  <span className="stat-value pr-text-2xl pr-font-bold pr-text-warning-500 pr-transition-all pr-duration-300">
                    {results.warnings}
                  </span>
                  <span className="stat-label pr-block pr-text-xs pr-text-gray-600 pr-mt-1">
                    {__('Warnings', 'prorank-seo')}
                  </span>
                </div>
                <div className="stat-item pr-text-center">
                  <span className="stat-value pr-text-2xl pr-font-bold pr-text-error-500 pr-transition-all pr-duration-300">
                    {results.critical}
                  </span>
                  <span className="stat-label pr-block pr-text-xs pr-text-gray-600 pr-mt-1">
                    {__('Critical', 'prorank-seo')}
                  </span>
                </div>
              </div>

              {/* Progress bar visualization */}
              <div className="progress-visualization pr-mt-4 pr-transition-all pr-duration-300">
                <div className="pr-flex pr-h-2 pr-rounded pr-overflow-hidden pr-bg-gray-200 pr-transition-all pr-duration-300">
                  {results.passed > 0 && (
                    <div
                      className="pr-bg-success-500 pr-transition-all pr-duration-300"
                      style={{ width: `${(results.passed / results.total) * 100}%` }}
                    />
                  )}
                  {results.warnings > 0 && (
                    <div
                      className="pr-bg-warning-500 pr-transition-all pr-duration-300"
                      style={{ width: `${(results.warnings / results.total) * 100}%` }}
                    />
                  )}
                  {results.critical > 0 && (
                    <div
                      className="pr-bg-error-500 pr-transition-all pr-duration-300"
                      style={{ width: `${(results.critical / results.total) * 100}%` }}
                    />
                  )}
                </div>
              </div>
            </div>

            {category.checks && category.checks.length > 0 && (
              <div className="category-checks pr-mt-6">
                <h4 className="pr-font-semibold pr-mb-2">{__('Checks performed:', 'prorank-seo')}</h4>
                <ul className="pr-list-disc pr-list-inside pr-space-y-1 pr-text-sm pr-text-gray-600">
                  {category.checks.slice(0, 3).map((check, index) => (
                    <li key={index}>{check}</li>
                  ))}
                  {category.checks.length > 3 && (
                    <li>
                      {sprintf(
                        __('...and %d more', 'prorank-seo'),
                        category.checks.length - 3
                      )}
                    </li>
                  )}
                </ul>
              </div>
            )}

            <div className="pr-mt-6">
              <Button variant="secondary" onClick={() => onClick && onClick(category)}>
                {__('View Details', 'prorank-seo')}
              </Button>
            </div>
          </>
        )}
      </div>
    </Card>
  );
};

export default AuditCategoryCard;