/* global sprintf, wp */
import { useState, useCallback, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import ModuleNav from '../../components/ModuleNav';
import { useNotification } from '../../contexts/NotificationContext';
import PageHeader from '../../components/PageHeader';
import headerBadges from '../../utils/headerBadges';
import CachingDeliverySettings from '../../settings/Performance/CachingDelivery/CachingDelivery';
import ImageOptimizationSettings from '../../settings/Performance/ImageOptimization/ImageOptimizationSettings';
import AssetOptimizationSettings from '../../settings/Performance/AssetOptimization/AssetOptimizationSettings';
import * as React from 'react';

// TypeScript Interfaces
interface IndividualModule {
  slug: string;
  name: string;
  description?: string;
  enabled: boolean;
  available: boolean;
}

interface PerformanceModule {
  slug: string;
  name: string;
  description: string;
  icon: string;
  enabled: boolean;
  available: boolean;
  modules?: IndividualModule[];
}

interface ModuleGroupConfig {
  modules: string[];
  name: string;
  description: string;
  icon: string;
}

interface ModuleGroupMap {
  [key: string]: ModuleGroupConfig;
}

interface ApiModulesResponse {
  modules: IndividualModule[];
}

interface ApiBetaResponse {
  value: boolean;
}

interface PerformanceProps {}

const Performance: React.FC<PerformanceProps> = () => {
  const [activeTab, setActiveTab] = useState<string>('caching-delivery');
  const [modules, setModules] = useState<PerformanceModule[]>([]);
  const [loading, setLoading] = useState<boolean>(true);
  const [updating, setUpdating] = useState<Record<string, boolean>>({});
  const [betaEnabled, setBetaEnabled] = useState<boolean>(true);
  
  const { showNotification } = useNotification();
  
  const loadModules = useCallback(async () => {
    setLoading(true);
    try {
      // Load individual modules from API
      const response = await apiFetch<ApiModulesResponse>({
        path: '/prorank-seo/v1/modules/by-group/performance',
      });
      
      // Store individual modules for reference
      const individualModules = response.modules || [];
      
      // Create grouped module cards based on loaded modules
      const moduleGroups: ModuleGroupMap = {
        'caching-delivery': {
          modules: ['browser_cache', 'js_minify', 'resource_hints'],
          name: __('Caching & Delivery', 'prorank-seo'),
          description: __(
            'Browser caching, JavaScript minify, and resource hints for faster delivery',
            'prorank-seo'
          ),
          icon: 'dashicons-performance',
        },
        'asset-optimisation': {
          modules: ['font_optimization'],
          name: __('Asset Optimisation', 'prorank-seo'),
          description: __(
            'Font loading and display tweaks for faster text rendering',
            'prorank-seo'
          ),
          icon: 'dashicons-editor-code',
        },
        'image-optimisation': {
          modules: ['image_optimization', 'lazyload'],
          name: __('Image Optimisation', 'prorank-seo'),
          description: __(
            'Lazy loading, WebP/AVIF conversion and optimized delivery',
            'prorank-seo'
          ),
          icon: 'dashicons-format-image',
        },
      };
      
      // Build grouped cards with aggregated data
      const performanceModules: PerformanceModule[] = Object.entries(moduleGroups).map(
        ([groupSlug, groupConfig]) => {
          // Find modules that belong to this group
          const groupModules = individualModules.filter((m) =>
            groupConfig.modules.includes(m.slug)
          );
          
          const availableGroupModules = groupModules.filter((m) => m.available !== false);
          const anyEnabled = availableGroupModules.some((m) => m.enabled);
          const anyAvailable = availableGroupModules.length > 0;
          
          return {
            slug: groupSlug,
            name: groupConfig.name,
            description: groupConfig.description,
            icon: groupConfig.icon,
            enabled: anyEnabled,
            available: anyAvailable,
            modules: groupModules, // Store individual modules for reference
          };
        }
      );

      const nonEmptyModules = performanceModules.filter((m) => (m.modules?.length || 0) > 0);

      setModules(nonEmptyModules);
      setActiveTab((current) =>
        nonEmptyModules.some((m) => m.slug === current)
          ? current
          : (nonEmptyModules[0]?.slug ?? current)
      );
      
      // Get beta toggle state
      let betaStatus = false;
      try {
        const betaResponse = await apiFetch<ApiBetaResponse>({
          path: '/prorank-seo/v1/performance-beta',
        });
        betaStatus = betaResponse.value || false;
        setBetaEnabled(betaStatus);
      } catch (error) {
        // Default to false if error
      }
      
    } catch (error) {
      // Show error state instead of using fallback data
      showNotification(
        __('Failed to load modules. Please refresh the page.', 'prorank-seo'),
        'error'
      );
      setModules([]);
    } finally {
      setLoading(false);
    }
  }, []);
  
  // Set page title
  useEffect(() => {
    document.title =
      __('Performance', 'prorank-seo') + ' ‹ ' + __('ProRank SEO', 'prorank-seo') + ' — WordPress';
  }, []);
  
  // Load modules from API
  useEffect(() => {
    loadModules();
  }, [loadModules]);
  
  const toggleModule = async (module: PerformanceModule) => {
    const slug = module.slug;
    const newState = !module.enabled;
    setUpdating((prev) => ({ ...prev, [slug]: true }));
    
    try {
      // For grouped modules, toggle all individual modules
      if (module.modules && module.modules.length > 0) {
        // Toggle all modules in the group
        const togglePromises = module.modules
          .filter((individualModule) => individualModule.available !== false)
          .map((individualModule) =>
          apiFetch({
            path: `/prorank-seo/v1/modules/${individualModule.slug}/toggle`,
            method: 'POST',
            data: { enabled: newState },
          })
        );
        
        if (togglePromises.length > 0) {
          await Promise.all(togglePromises);
        }
      } else {
        // Fallback for individual modules
        await apiFetch({
          path: `/prorank-seo/v1/modules/${slug}/toggle`,
          method: 'POST',
          data: { enabled: newState },
        });
      }
      
      // Update module state
      setModules((prev) =>
        prev.map((m) => (m.slug === slug ? { ...m, enabled: newState } : m))
      );
      
      showNotification(
        newState
          ? __('Module group enabled successfully', 'prorank-seo')
          : __('Module group disabled successfully', 'prorank-seo'),
        'success'
      );
    } catch (error) {
      // Show error and don't update UI if API fails
      showNotification(
        error.message || __('Failed to toggle module', 'prorank-seo'),
        'error'
      );
    } finally {
      setUpdating((prev) => ({ ...prev, [slug]: false }));
    }
  };
  
  
  const renderTabContent = () => {
    const activeModule = modules.find((m) => m.slug === activeTab);
    if (!activeModule) {
      return null;
    }

    const isEnabled = activeModule.enabled;

    // If module is disabled, show enable message (matches Technical SEO)
    if (!isEnabled) {
      return (
        <div className="prorank-module-disabled">
          <div className="prorank-disabled-content">
            <span
              className={`dashicons ${activeModule.icon}`}
              style={{ fontSize: '48px', marginBottom: '16px', opacity: 0.5 }}
            ></span>
            <h3>{activeModule.name}</h3>
            <p>
              {__(
                'This module is currently disabled. Enable it to access its settings and features.',
                'prorank-seo'
              )}
            </p>
          </div>
        </div>
      );
    }

    // Render the appropriate settings component (only when enabled - matches Technical SEO)
    const renderSettings = () => {
      switch (activeModule.slug) {
        case 'caching-delivery':
          return <CachingDeliverySettings onClose={() => {}} betaEnabled={betaEnabled} />;
        case 'image-optimisation':
          return <ImageOptimizationSettings onClose={() => {}} />;
        case 'asset-optimisation':
          return <AssetOptimizationSettings onClose={() => {}} />;
        default:
          return (
            <div className="prorank-settings-card" style={{ textAlign: 'center', padding: '60px' }}>
              <h3>{__('Module Not Found', 'prorank-seo')}</h3>
              <p>{__('The selected module does not have settings available.', 'prorank-seo')}</p>
            </div>
          );
      }
    };

    return <div className="prorank-settings-wrapper">{renderSettings()}</div>;
  };
  
  if (loading) {
    return (
      <div className="prorank-page prorank-performance-page">
        <div className="prorank-loading">
          <div className="prorank-loading-spinner"></div>
          <p>{__('Loading modules…', 'prorank-seo')}</p>
        </div>
      </div>
    );
  }
  
  return (
    <div className="prorank-page prorank-performance-page">
      <PageHeader
        title={__('Performance', 'prorank-seo')}
        subtitle={__('Optimize your site speed with caching, image optimization, and performance tools', 'prorank-seo')}
        badges={headerBadges.performance}
      />

      {/* Premium Features Banner - dismissible, WP.org compliant */}

      {/* Main Content */}
      <div className="prorank-content prorank-performance-content">
        <ModuleNav
          modules={modules.map((module) => {
            return {
              id: module.slug,
              label: module.name,
              badge: null,
              badgeVariant: null,
              hasToggle: true,
              toggleValue: module.enabled,
              toggleDisabled: updating[module.slug] || module.available === false,
            };
          })}
          activeModule={activeTab}
          onModuleChange={setActiveTab}
          onToggleChange={(moduleId, checked) => {
            const module = modules.find((item) => item.slug === moduleId);
            if (!module || module.enabled === checked) return;
            toggleModule(module);
          }}
        />
        
        <div className="pr-mt-6">{renderTabContent()}</div>
      </div>
    </div>
  );
};

export default Performance;
