import { useState, useCallback, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import ModuleNav from '../../components/ModuleNav';
import PageHeader from '../../components/PageHeader';
import headerBadges from '../../utils/headerBadges';
import { useNotification } from '../../contexts/NotificationContext';
import SiteBasicsSettings from '../../settings/OnPageSEO/SiteBasics/SiteBasicsSettings';
import TitleMetaSettings from '../../settings/OnPageSEO/TitlesMeta/TitleMetaSettings';
import BreadcrumbsSettings from '../../settings/OnPageSEO/Breadcrumbs/BreadcrumbsSettings';
import PodcastSeoSettings from '../../settings/OnPageSEO/PodcastSeo/PodcastSeoSettings';
import ContentAnalysisSettings from '../../settings/OnPageSEO/ContentAnalysis/ContentAnalysisSettings';
import InternalLinkingSettings from '../../settings/OnPageSEO/InternalLinking/InternalLinkingSettings';
import { Alert, Button, Card, Input, Select, Toggle } from '../../components/ui';
import LoadingState from '../../components/LoadingState';

const SchemaMarkupSettings = () => {
  const defaultSettings = {
    disable_all_schema: false,
    enable_website_schema: true,
    enable_organization_schema: true,
    enable_article_schema: true,
    enable_breadcrumb_schema: true,
    organization_name: '',
    organization_logo: '',
    organization_type: 'Organization',
  };

  const [settings, setSettings] = useState(defaultSettings);

  const [loading, setLoading] = useState(true);
  const [saving, setSaving] = useState(false);
  const [notice, setNotice] = useState(null);
  const { showNotification } = useNotification();

  useEffect(() => {
    loadSettings();
  }, []);

  const loadSettings = async () => {
    setLoading(true);
    setNotice(null);

    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/settings/basic-schema',
        method: 'GET',
      });

      if (response && typeof response === 'object') {
        const normalized = { ...defaultSettings, ...response };
        if (normalized.disable_all_schema) {
          normalized.enable_website_schema = false;
          normalized.enable_organization_schema = false;
          normalized.enable_article_schema = false;
          normalized.enable_breadcrumb_schema = false;
        }
        setSettings(normalized);
      }
    } catch (error) {
      setNotice({ type: 'error', message: __('Failed to load schema settings', 'prorank-seo') });
    } finally {
      setLoading(false);
    }
  };

  const saveSettings = async () => {
    setSaving(true);
    setNotice(null);

    try {
      await apiFetch({
        path: '/prorank-seo/v1/settings/basic-schema',
        method: 'POST',
        data: settings,
      });

      setNotice({ type: 'success', message: __('Settings saved successfully', 'prorank-seo') });
      showNotification(__('Schema settings saved', 'prorank-seo'), 'success');
    } catch (error) {
      setNotice({ type: 'error', message: __('Failed to save schema settings', 'prorank-seo') });
      showNotification(__('Failed to save schema settings', 'prorank-seo'), 'error');
    } finally {
      setSaving(false);
    }
  };

  const updateSetting = (key, value) => {
    setSettings((prev) => {
      if (key === 'disable_all_schema') {
        const next = { ...prev, disable_all_schema: value };
        if (value) {
          next.enable_website_schema = false;
          next.enable_organization_schema = false;
          next.enable_article_schema = false;
          next.enable_breadcrumb_schema = false;
        }
        return next;
      }
      return { ...prev, [key]: value };
    });
  };

  if (loading) {
    return <LoadingState message={__('Loading schema settings…', 'prorank-seo')} />;
  }

  return (
    <div className="pr-space-y-6 pr-max-w-4xl">
      <div className="pr-mb-6">
        <h2 className="pr-text-2xl pr-font-semibold pr-text-gray-900 pr-mb-2">
          {__('Schema Markup', 'prorank-seo')}
        </h2>
        <p className="pr-text-sm pr-text-gray-600">
          {__(
            'Enable basic structured data (WebSite, Organization, Article, Breadcrumbs) to help search engines understand your site.',
            'prorank-seo'
          )}
        </p>
      </div>

      {notice && (
        <Alert variant={notice.type === 'success' ? 'success' : 'error'} title={notice.message} />
      )}

      <Card>
        <div className="pr-p-6 pr-space-y-5">
          <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">{__('Schemas', 'prorank-seo')}</h3>

          <Toggle
            label={__('Disable all schema output', 'prorank-seo')}
            checked={Boolean(settings.disable_all_schema)}
            onChange={(value) => updateSetting('disable_all_schema', value)}
            help={__('Turns off all JSON-LD output from ProRank SEO and disables the toggles below.', 'prorank-seo')}
          />

          <Toggle
            label={__('WebSite schema', 'prorank-seo')}
            checked={Boolean(settings.enable_website_schema)}
            onChange={(value) => updateSetting('enable_website_schema', value)}
            help={__('Outputs WebSite JSON-LD (with search action) and WebPage schema for pages.', 'prorank-seo')}
            disabled={Boolean(settings.disable_all_schema)}
          />

          <Toggle
            label={__('Organization schema', 'prorank-seo')}
            checked={Boolean(settings.enable_organization_schema)}
            onChange={(value) => updateSetting('enable_organization_schema', value)}
            help={__('Outputs Organization JSON-LD on the homepage.', 'prorank-seo')}
            disabled={Boolean(settings.disable_all_schema)}
          />

          <Toggle
            label={__('Article schema (posts)', 'prorank-seo')}
            checked={Boolean(settings.enable_article_schema)}
            onChange={(value) => updateSetting('enable_article_schema', value)}
            help={__('Outputs Article JSON-LD on single posts.', 'prorank-seo')}
            disabled={Boolean(settings.disable_all_schema)}
          />

          <Toggle
            label={__('Breadcrumb schema', 'prorank-seo')}
            checked={Boolean(settings.enable_breadcrumb_schema)}
            onChange={(value) => updateSetting('enable_breadcrumb_schema', value)}
            help={__('Outputs BreadcrumbList JSON-LD on non-home pages.', 'prorank-seo')}
            disabled={Boolean(settings.disable_all_schema)}
          />
        </div>
      </Card>

      <Card>
        <div className="pr-p-6 pr-space-y-5">
          <h3 className="pr-text-lg pr-font-semibold pr-text-gray-900">
            {__('Organization Details', 'prorank-seo')}
          </h3>

          <Input
            label={__('Organization name', 'prorank-seo')}
            value={settings.organization_name || ''}
            onChange={(value) => updateSetting('organization_name', value)}
            placeholder={__('Your business or site name', 'prorank-seo')}
          />

          <Select
            label={__('Organization type', 'prorank-seo')}
            value={settings.organization_type || 'Organization'}
            onChange={(value) => updateSetting('organization_type', value)}
            options={[
              { label: __('Organization', 'prorank-seo'), value: 'Organization' },
              { label: __('LocalBusiness', 'prorank-seo'), value: 'LocalBusiness' },
              { label: __('Person', 'prorank-seo'), value: 'Person' },
            ]}
          />

          <Input
            label={__('Organization logo URL', 'prorank-seo')}
            value={settings.organization_logo || ''}
            onChange={(value) => updateSetting('organization_logo', value)}
            placeholder={__('https://example.com/logo.png', 'prorank-seo')}
            help={__('Optional. Used in Organization schema when enabled.', 'prorank-seo')}
          />
        </div>
      </Card>

      <div className="pr-flex pr-gap-3">
        <Button variant="primary" onClick={saveSettings} loading={saving} disabled={saving}>
          {saving ? __('Saving…', 'prorank-seo') : __('Save Settings', 'prorank-seo')}
        </Button>
        <Button variant="secondary" onClick={loadSettings} disabled={saving}>
          {__('Reload', 'prorank-seo')}
        </Button>
      </div>
    </div>
  );
};

const OnPageSEO = () => {
  const [activeTab, setActiveTab] = useState('site-basics');
  const [modules, setModules] = useState([]);
  const [loading, setLoading] = useState(true);
  const [updating, setUpdating] = useState({});
  const { showNotification } = useNotification();
  const loadModules = useCallback(async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/modules/by-group/on-page-seo',
        method: 'GET',
      });

      if (response && response.modules) {
        const nextModules = response.modules || [];
        // Only show main modules as tabs - filter out advanced/sub modules
        // Note: readability-analysis is a sub-feature of content-analysis, not a separate tab
        const mainModuleSlugs = [
          'site-basics',
          'titles-meta',
          'breadcrumbs',
          'schema-markup',
          'content-analysis',
          'internal-linking',
          'podcast-seo',
        ];

        // Filter to only main modules (not advanced variants like advanced-schema,
        // advanced-content-analysis, content-safeguard, visual-link-map, etc.)
        const mainModules = nextModules.filter((m) => mainModuleSlugs.includes(m.slug));

        const orderIndex = new Map(mainModuleSlugs.map((slug, idx) => [slug, idx]));

        setModules(
          mainModules.slice().sort((a, b) => {
            const ai = orderIndex.has(a.slug) ? orderIndex.get(a.slug) : 999;
            const bi = orderIndex.has(b.slug) ? orderIndex.get(b.slug) : 999;
            if (ai !== bi) return ai - bi;
            return String(a.name || '').localeCompare(String(b.name || ''));
          })
        );
      }
    } catch (error) {
      console.error('Failed to load modules:', error);
      showNotification(
        __('Failed to load modules. Please refresh the page.', 'prorank-seo'),
        'error'
      );
    } finally {
      setLoading(false);
    }
  }, []);
  // Set page title
  useEffect(() => {
    document.title =
      __('On-Page SEO', 'prorank-seo') + ' ‹ ' + __('ProRank SEO', 'prorank-seo') + ' — WordPress';
  }, []);
  // Load modules from API once on mount
  useEffect(() => {
    loadModules();
  }, []);
  const toggleModule = async (module) => {
    const slug = module.slug;
    const newState = !module.enabled;
    setUpdating((prev) => ({ ...prev, [slug]: true }));
    try {
      // Toggle module via REST API
      await apiFetch({
        path: `/prorank-seo/v1/modules/${slug}/toggle`,
        method: 'POST',
        data: { enabled: newState },
      });

      // Update module state locally
      setModules((prev) => prev.map((m) => (m.slug === slug ? { ...m, enabled: newState } : m)));
      showNotification(
        newState
          ? __('Module enabled successfully', 'prorank-seo')
          : __('Module disabled successfully', 'prorank-seo'),
        'success'
      );
    } catch (error) {
      showNotification(error.message || __('Failed to toggle module', 'prorank-seo'), 'error');
    } finally {
      setUpdating((prev) => ({ ...prev, [slug]: false }));
    }
  };
  const renderTabContent = () => {
    const activeModule = modules.find((m) => m.slug === activeTab);
    if (!activeModule) {
      return null;
    }
    const isEnabled = activeModule.enabled;
    // If module is disabled, just show the control
    if (!isEnabled) {
      return (
        <div className="prorank-module-disabled">
          <div className="prorank-disabled-content">
            <span
              className={`dashicons ${activeModule.icon}`}
              style={{ fontSize: '48px', marginBottom: '16px', opacity: 0.5 }}
            ></span>
            <h3>{activeModule.name}</h3>
            <p>
              {__(
                'This module is currently disabled. Enable it to access its settings and features.',
                'prorank-seo'
              )}
            </p>
          </div>
        </div>
      );
    }
    // Render the appropriate settings component
    const renderSettings = () => {
      switch (activeModule.slug) {
        case 'site-basics':
          return <SiteBasicsSettings />;
        case 'titles-meta':
          return <TitleMetaSettings />;
        case 'breadcrumbs':
          return <BreadcrumbsSettings />;
        case 'schema-markup':
          return <SchemaMarkupSettings />;
        case 'content-analysis':
          return <ContentAnalysisSettings />;
        case 'internal-linking':
          return <InternalLinkingSettings />;
        case 'podcast-seo':
          return <PodcastSeoSettings moduleSlug={activeModule.slug} />;
        default:
          return null;
      }
    };
    return <div className="prorank-settings-wrapper">{renderSettings()}</div>;
  };
  if (loading) {
    return (
      <div className="prorank-page prorank-onpage-seo-page">
        <div className="prorank-loading">
          <div className="prorank-loading-spinner"></div>
          <p>{__('Loading modules…', 'prorank-seo')}</p>
        </div>
      </div>
    );
  }
  return (
    <div className="prorank-page prorank-onpage-seo-page">
      <PageHeader
        title={__('On-Page SEO', 'prorank-seo')}
        subtitle={__('Optimize your on-page elements for better search engine visibility', 'prorank-seo')}
        badges={headerBadges['on-page-seo']}
      />
      
      {/* Main Content */}
      <div className="prorank-content prorank-onpage-seo-content">
        <ModuleNav
          modules={modules.map((module) => {
            return {
              id: module.slug,
              label: module.name,
              badge: null,
              badgeVariant: null,
              hasToggle: true,
              toggleValue: module.enabled,
              toggleDisabled: updating[module.slug] || module.available === false,
            };
          })}
          activeModule={activeTab}
          onModuleChange={setActiveTab}
          onToggleChange={(moduleId, checked) => {
            const module = modules.find((item) => item.slug === moduleId);
            if (!module || module.enabled === checked) return;
            toggleModule(module);
          }}
        />
        <div className="pr-mt-6">{renderTabContent()}</div>
      </div>
    </div>
  );
};
export default OnPageSEO;
