/**
 * Internal Linking Pages Hub (Free)
 *
 * @package
 * @since   1.0.0
 */

import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import ModuleNav from '../../components/ModuleNav';
import PageHeader from '../../components/PageHeader';
import headerBadges from '../../utils/headerBadges';
import LinkingDashboard from './LinkingDashboard';
import InternalLinkingSettings from '../../settings/OnPageSEO/InternalLinking/InternalLinkingSettings';

const InternalLinkingPages = () => {
  const tabs = [
    {
      id: 'dashboard',
      label: __('Dashboard', 'prorank-seo'),
    },
    {
      id: 'settings',
      label: __('Settings', 'prorank-seo'),
    },
  ];

  // Check for tab in URL hash
  const getInitialTab = () => {
    const hash = window.location.hash;
    const tabMatch = hash.match(/tab=([^&]+)/);
    if (tabMatch) {
      const tabName = tabMatch[1];
      if (tabs.some((tab) => tab.id === tabName)) {
        return tabName;
      }
    }
    return 'dashboard';
  };

  const [activeTab, setActiveTab] = useState(getInitialTab());

  // Update URL when tab changes
  const handleTabChange = (tabName) => {
    setActiveTab(tabName);
    const currentHash = window.location.hash;
    const baseHash = currentHash.split('?')[0];
    window.location.hash = `${baseHash}?tab=${tabName}`;
  };

  // Listen for hash changes
  useEffect(() => {
    const handleHashChange = () => {
      const newTab = getInitialTab();
      if (newTab !== activeTab) {
        setActiveTab(newTab);
      }
    };

    window.addEventListener('hashchange', handleHashChange);
    return () => window.removeEventListener('hashchange', handleHashChange);
  }, [activeTab]);

  const renderTabContent = () => {
    switch (activeTab) {
      case 'dashboard':
        return <LinkingDashboard />;
      case 'settings':
        return <InternalLinkingSettings />;
      default:
        return <LinkingDashboard />;
    }
  };

  return (
    <div className="prorank-page">
      <PageHeader
        title={__('Internal Linking', 'prorank-seo')}
        subtitle={__('Manage internal link suggestions and settings.', 'prorank-seo')}
        badges={headerBadges['internal-linking']}
      />

      <div className="prorank-content">
        <ModuleNav
          modules={tabs.map((tab) => ({
            id: tab.id,
            label: tab.label,
            badge: null,
            badgeVariant: null,
            hasToggle: false,
          }))}
          activeModule={activeTab}
          onModuleChange={handleTabChange}
        />

        <div className="pr-mt-6">{renderTabContent()}</div>
      </div>
    </div>
  );
};

export default InternalLinkingPages;
