/**
 * Internal Linking Dashboard
 * 
 * Enhanced dashboard with graphs, metrics, and comprehensive link analysis
 */

import { useState, useEffect } from '@wordpress/element';
import { __ } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { Card, Button, Alert } from '../../components/ui';
import LoadingState from '../../components/LoadingState';

const LinkingDashboard = () => {
  const [loading, setLoading] = useState(true);
  const [scanning, setScanning] = useState(false);
  const [scanProgress, setScanProgress] = useState(0);
  const [lastScan, setLastScan] = useState(null);
  const [stats, setStats] = useState({
    posts_crawled: 0,
    link_clicks_tracked: 0,
    link_clicks_change: 0,
    orphaned_posts: 0,
    broken_links: 0,
    link_coverage: 0,
    link_quality_score: 0,
    external_site_focus: 0,
    anchor_length_score: 0,
    most_linked_domains: [],
    link_health_details: {
      total_internal: 0,
      total_external: 0,
      avg_links_per_post: 0,
      posts_without_links: 0,
    },
    link_distribution: [],
    recent_changes: [],
  });

  useEffect(() => {
    loadDashboardData();
    checkLastScan();
  }, []);

  const loadDashboardData = async () => {
    setLoading(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/linking/dashboard',
      });
      
      if (response && response.data) {
        // Process and enhance the data
        const enhancedStats = {
          posts_crawled: response.data.total_posts || 0,
          link_clicks_tracked: response.data.link_clicks || 0,
          link_clicks_change: response.data.link_clicks_change || -100,
          orphaned_posts: response.data.orphan_count || 0,
          broken_links: response.data.broken_links || 0,
          link_coverage: response.data.link_coverage || 0,
          link_quality_score: calculateQualityScore(response.data),
          external_site_focus: response.data.external_percentage || 60,
          anchor_length_score: response.data.anchor_score || 38,
          most_linked_domains: response.data.top_domains || [],
          link_health_details: {
            total_internal: response.data.total_internal_links || 0,
            total_external: response.data.total_external_links || 0,
            avg_links_per_post: response.data.avg_links || 0,
            posts_without_links: response.data.posts_without_links || 0,
          },
          link_distribution: response.data.link_distribution || [],
          recent_changes: response.data.recent_changes || [],
        };
        setStats(enhancedStats);
      }
    } catch (error) {
      // Error loading dashboard data
    } finally {
      setLoading(false);
    }
  };

  const calculateQualityScore = (data) => {
    // Calculate quality score based on multiple factors
    let score = 0;
    
    // Coverage contributes 40%
    if (data.link_coverage) {
      score += (data.link_coverage / 100) * 40;
    }
    
    // Orphan rate contributes 30% (inverse)
    const orphanRate = data.total_posts > 0 ? 
      (data.orphan_count / data.total_posts) : 1;
    score += (1 - orphanRate) * 30;
    
    // Broken links contribute 30% (inverse)
    const brokenRate = data.total_internal_links > 0 ?
      (data.broken_links / data.total_internal_links) : 0;
    score += (1 - Math.min(brokenRate, 1)) * 30;
    
    return Math.round(score);
  };

  const checkLastScan = async () => {
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/linking/scan-info',
      });
      if (response && response.last_scan) {
        setLastScan(new Date(response.last_scan));
      }
    } catch (error) {
      // Error checking scan info
    }
  };

  const runLinkScan = async () => {
    setScanning(true);
    setScanProgress(0);

    try {
      // Start the scan
      await apiFetch({
        path: '/prorank-seo/v1/linking/start-scan',
        method: 'POST',
      });

      // Poll for real progress with exponential backoff on errors
      let pollDelay = 1000;
      let consecutiveErrors = 0;
      const maxErrors = 5;

      const pollProgress = async () => {
        if (consecutiveErrors >= maxErrors) {
          setScanning(false);
          setScanProgress(100);
          loadDashboardData();
          return;
        }

        try {
          const response = await apiFetch({
            path: '/prorank-seo/v1/linking/scan-progress',
          });

          // Reset error count on success
          consecutiveErrors = 0;
          pollDelay = 1000;

          if (response) {
            const progress = response.progress || 0;
            setScanProgress(progress);

            if (response.status === 'complete' || progress >= 100) {
              setScanning(false);
              loadDashboardData();
              checkLastScan();
              return;
            }

            // Continue polling
            setTimeout(pollProgress, pollDelay);
          } else {
            // No response, continue polling
            setTimeout(pollProgress, pollDelay);
          }
        } catch (error) {
          // On error, use exponential backoff
          consecutiveErrors++;
          pollDelay = Math.min(pollDelay * 2, 10000); // Max 10 seconds
          setTimeout(pollProgress, pollDelay);
        }
      };

      // Start polling after a short delay
      setTimeout(pollProgress, 500);

    } catch (error) {
      setScanning(false);
    }
  };

  const getHealthScoreColor = (score) => {
    if (score >= 70) return '#10b981';
    if (score >= 40) return '#f59e0b';
    return '#ef4444';
  };

  const getChangeIndicator = (value) => {
    if (value > 0) {
      return <span className="pr-text-green-500 pr-text-[11px] pr-transition-all pr-duration-300">+{value}%</span>;
    } else if (value < 0) {
      return <span className="pr-text-red-500 pr-text-[11px] pr-transition-all pr-duration-300">{value}%</span>;
    }
    return <span className="pr-text-gray-500 pr-text-[11px] pr-transition-all pr-duration-300">0%</span>;
  };

  const safeNumber = (value, fallback = 0) => {
    const numeric = Number(value);
    return Number.isFinite(numeric) ? numeric : fallback;
  };

  const formatNumber = (value, decimals = 0) => safeNumber(value).toFixed(decimals);

  const renderProgressCircle = (value, max = 100, color = '#3b82f6') => {
    const percentage = (value / max) * 100;
    const radius = 30;
    const circumference = 2 * Math.PI * radius;
    const offset = circumference - (percentage / 100) * circumference;
    
    return (
      <div className="pr-relative pr-w-[70px] pr-h-[70px]">
        <svg width="70" height="70" className="pr-transform pr--rotate-90">
          <circle
            cx="35"
            cy="35"
            r={radius}
            stroke="#e5e7eb"
            strokeWidth="4"
            fill="none"
          />
          <circle
            cx="35"
            cy="35"
            r={radius}
            stroke={color}
            strokeWidth="4"
            fill="none"
            strokeDasharray={circumference}
            strokeDashoffset={offset}
            className="pr-transition-all pr-duration-500"
          />
        </svg>
        <div className="pr-absolute pr-top-1/2 pr-left-1/2 pr-transform pr--translate-x-1/2 pr--translate-y-1/2 pr-text-sm pr-font-semibold pr-text-gray-900">
          {Math.round(percentage)}%
        </div>
      </div>
    );
  };

  if (loading) {
    return (
      <Card className="pr-p-8">
        <LoadingState message={__('Loading dashboard...', 'prorank-seo')} />
      </Card>
    );
  }

  return (
    <div className="prorank-linking-dashboard">
      {/* Scan Section */}
      <Card className="pr-mb-5 pr-transition-all pr-duration-300 pr-bg-gradient-to-br pr-from-gray-100 pr-to-white">
        <div className="pr-p-6">
          <div className="pr-flex pr-items-center pr-justify-between">
            <div>
              <h3 className="pr-text-base pr-font-semibold pr-mb-2">
                {__('Link Scan', 'prorank-seo')}
              </h3>
              {lastScan && (
                <span className="pr-text-sm pr-text-gray-500">
                  {__('Last scan:', 'prorank-seo')} {lastScan.toLocaleDateString()} {lastScan.toLocaleTimeString()}
                </span>
              )}
            </div>
            <Button
              variant="primary"
              onClick={runLinkScan}
              disabled={scanning}
              className="pr-transition-all pr-duration-300 pr-border-none pr-text-white pr-px-5 pr-py-2 pr-text-sm pr-font-medium pr-rounded-md pr-shadow-sm"
              style={{
                background: scanning ? '#9ca3af' : 'linear-gradient(135deg, #3b82f6 0%, #2563eb 100%)',
                transition: 'all 0.3s ease',
              }}
            >
              {scanning ? __('Scanning...', 'prorank-seo') : __('Run Link Scan', 'prorank-seo')}
            </Button>
          </div>
          {scanning && (
            <div className="pr-mt-4 pr-transition-all pr-duration-300">
              <div className="pr-w-full pr-bg-gray-200 pr-rounded-full pr-h-2 pr-overflow-hidden">
                <div
                  className="pr-bg-gradient-to-r pr-from-purple-600 pr-to-purple-800 pr-h-full pr-transition-all pr-duration-300"
                  style={{ width: `${scanProgress}%` }}
                />
              </div>
              <span className="pr-text-sm pr-text-gray-500 pr-mt-2 pr-block">
                {__('Scanning posts...', 'prorank-seo')} {scanProgress}%
              </span>
            </div>
          )}
        </div>
      </Card>

      {/* Link Insights */}
      <div className="pr-bg-white pr-rounded-lg pr-p-5 pr-mb-5 pr-shadow-sm pr-transition-all pr-duration-300">
        <h3 className="pr-text-base pr-font-semibold pr-mb-4">
          {__('Link Insights', 'prorank-seo')}
        </h3>

        <div className="pr-grid pr-grid-cols-4 pr-gap-5">
          <div
            className="pr-text-center pr-cursor-pointer pr-p-2.5 pr-rounded-md pr-transition-all pr-duration-200 hover:pr-bg-gray-100"
            onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=links-report'}
          >
            <div className="pr-text-2xl pr-font-semibold pr-text-gray-900">
              {stats.posts_crawled}
            </div>
            <div className="pr-text-xs pr-text-gray-500 pr-mt-1">
              {__('Posts Crawled', 'prorank-seo')}
            </div>
          </div>

          <div
            className="pr-text-center pr-cursor-pointer pr-p-2.5 pr-rounded-md pr-transition-all pr-duration-200 hover:pr-bg-gray-100"
            onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=links-report'}
          >
            <div className="pr-text-2xl pr-font-semibold pr-text-gray-900">
              {stats.link_clicks_tracked}
            </div>
            <div className="pr-text-xs pr-text-gray-500 pr-mt-1">
              {__('Link Clicks Tracked', 'prorank-seo')}
            </div>
            {getChangeIndicator(stats.link_clicks_change)}
          </div>

          <div
            className="pr-text-center pr-cursor-pointer pr-p-2.5 pr-rounded-md pr-transition-all pr-duration-200 hover:pr-bg-gray-100"
            onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=links-report'}
          >
            <div className="pr-text-2xl pr-font-semibold pr-text-red-500">
              {stats.orphaned_posts}
            </div>
            <div className="pr-text-xs pr-text-gray-500 pr-mt-1">
              {__('Orphaned Posts', 'prorank-seo')}
            </div>
          </div>

          <div
            className="pr-text-center pr-cursor-pointer pr-p-2.5 pr-rounded-md pr-transition-all pr-duration-200 hover:pr-bg-gray-100"
            onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=links-report'}
          >
            <div className="pr-text-2xl pr-font-semibold pr-text-amber-500">
              {stats.broken_links}
            </div>
            <div className="pr-text-xs pr-text-gray-500 pr-mt-1">
              {__('Broken Links', 'prorank-seo')}
            </div>
          </div>
        </div>
      </div>

      {/* Link Health Metrics */}
      <div className="pr-grid pr-grid-cols-2 pr-gap-5 pr-mb-5">
        {/* Link Health Scores */}
        <Card className="pr-transition-all pr-duration-300">
          <div className="pr-p-6">
            <h3 className="pr-text-base pr-font-semibold pr-mb-4">{__('Link Health', 'prorank-seo')}</h3>
            <div className="pr-grid pr-grid-cols-2 pr-gap-4">
              <div>
                <div className="pr-flex pr-items-center pr-justify-between pr-mb-3">
                  <span className="pr-text-sm">{__('Link Coverage', 'prorank-seo')}</span>
                  <span className="pr-text-sm pr-font-semibold">{formatNumber(stats.link_coverage, 2)}%</span>
                </div>
                <div className="pr-bg-gray-200 pr-h-2 pr-rounded pr-overflow-hidden">
                  <div
                    className="pr-h-full pr-transition-all pr-duration-300"
                    style={{
                      width: `${safeNumber(stats.link_coverage)}%`,
                      background: getHealthScoreColor(safeNumber(stats.link_coverage)),
                    }}
                  />
                </div>
              </div>

              <div>
                <div className="pr-flex pr-items-center pr-justify-between pr-mb-3">
                  <span className="pr-text-sm">{__('Link Quality Score', 'prorank-seo')}</span>
                  <span className="pr-text-sm pr-font-semibold">{safeNumber(stats.link_quality_score)}%</span>
                </div>
                <div className="pr-bg-gray-200 pr-h-2 pr-rounded pr-overflow-hidden">
                  <div
                    className="pr-h-full pr-transition-all pr-duration-300"
                    style={{
                      width: `${safeNumber(stats.link_quality_score)}%`,
                      background: getHealthScoreColor(safeNumber(stats.link_quality_score)),
                    }}
                  />
                </div>
              </div>

              <div>
                <div className="pr-flex pr-items-center pr-justify-between pr-mb-3">
                  <span className="pr-text-sm">{__('External Site Focus', 'prorank-seo')}</span>
                  <span className="pr-text-sm pr-font-semibold">{safeNumber(stats.external_site_focus)}%</span>
                </div>
                <div className="pr-bg-gray-200 pr-h-2 pr-rounded pr-overflow-hidden">
                  <div
                    className="pr-h-full pr-transition-all pr-duration-300"
                    style={{
                      width: `${safeNumber(stats.external_site_focus)}%`,
                      background: safeNumber(stats.external_site_focus) > 70
                        ? '#ef4444'
                        : getHealthScoreColor(100 - safeNumber(stats.external_site_focus)),
                    }}
                  />
                </div>
              </div>

              <div>
                <div className="pr-flex pr-items-center pr-justify-between pr-mb-3">
                  <span className="pr-text-sm">{__('Anchor Length Score', 'prorank-seo')}</span>
                  <span className="pr-text-sm pr-font-semibold">{safeNumber(stats.anchor_length_score)}%</span>
                  <button
                    className="pr-text-xs pr-text-purple-600 pr-underline pr-ml-2 pr-transition-all pr-duration-300 hover:pr-text-purple-700"
                    onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=links-report'}
                  >
                    {__('Fix', 'prorank-seo')}
                  </button>
                </div>
                <div className="pr-bg-gray-200 pr-h-2 pr-rounded pr-overflow-hidden">
                  <div
                    className="pr-h-full pr-transition-all pr-duration-300"
                    style={{
                      width: `${safeNumber(stats.anchor_length_score)}%`,
                      background: getHealthScoreColor(safeNumber(stats.anchor_length_score)),
                    }}
                  />
                </div>
              </div>
            </div>
          </div>
        </Card>

        {/* Most Linked To Domains */}
        <Card className="pr-transition-all pr-duration-300">
          <div className="pr-p-6">
            <h3 className="pr-text-base pr-font-semibold pr-mb-4">{__('Most Linked To Domains', 'prorank-seo')}</h3>
            {stats.most_linked_domains.length > 0 ? (
              <div className="pr-space-y-2">
                {stats.most_linked_domains.map((domain, index) => (
                  <div
                    key={index}
                    className="pr-flex pr-items-center pr-justify-between pr-w-full pr-p-3 pr-rounded pr-transition-all pr-duration-300"
                    style={{ background: index % 2 === 0 ? '#f9fafb' : '#fff' }}
                  >
                    <span className="pr-text-sm pr-font-mono">
                      {domain.domain || domain}
                    </span>
                    <span className="pr-text-sm pr-font-semibold pr-text-gray-500">
                      {domain.count || 0} links
                    </span>
                  </div>
                ))}
              </div>
            ) : (
              <Alert variant="info">
                {__('No external domains found. Run a scan to populate this data.', 'prorank-seo')}
              </Alert>
            )}
          </div>
        </Card>
      </div>

      {/* Link Distribution Chart */}
      <Card className="pr-mb-5 pr-transition-all pr-duration-300">
        <div className="pr-p-6">
          <h3 className="pr-text-base pr-font-semibold pr-mb-4">{__('Link Distribution', 'prorank-seo')}</h3>
          <div className="pr-flex pr-items-center pr-justify-around pr-py-5">
            <div className="pr-text-center">
              {renderProgressCircle(stats.link_health_details.total_internal,
                stats.link_health_details.total_internal + stats.link_health_details.total_external,
                '#10b981')}
              <span className="pr-text-sm pr-mt-2 pr-block">
                {__('Internal Links', 'prorank-seo')}
              </span>
              <span className="pr-text-sm pr-font-semibold pr-block">
                {stats.link_health_details.total_internal}
              </span>
            </div>

            <div className="pr-text-center">
              {renderProgressCircle(stats.link_health_details.total_external,
                stats.link_health_details.total_internal + stats.link_health_details.total_external,
                '#f59e0b')}
              <span className="pr-text-sm pr-mt-2 pr-block">
                {__('External Links', 'prorank-seo')}
              </span>
              <span className="pr-text-sm pr-font-semibold pr-block">
                {stats.link_health_details.total_external}
              </span>
            </div>

            <div className="pr-text-center">
              <div className="pr-w-[70px] pr-h-[70px] pr-rounded-full pr-flex pr-items-center pr-justify-center pr-text-xl pr-font-semibold pr-text-indigo-700 pr-transition-all pr-duration-300"
                style={{ background: 'linear-gradient(135deg, #e0e7ff 0%, #c7d2fe 100%)' }}>
                {formatNumber(stats.link_health_details?.avg_links_per_post, 1)}
              </div>
              <span className="pr-text-sm pr-mt-2 pr-block">
                {__('Avg Links/Post', 'prorank-seo')}
              </span>
            </div>
          </div>
        </div>
      </Card>

      {/* Quick Actions */}
      <div className="pr-grid pr-grid-cols-4 pr-gap-4">
        <button
          onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=orphans'}
          className="pr-p-3 pr-text-center pr-border pr-border-gray-200 pr-rounded-lg pr-bg-white pr-transition-all pr-duration-300 hover:pr-shadow-md hover:pr-border-purple-200"
        >
          <div className="pr-space-y-2">
            <svg className="pr-w-6 pr-h-6 pr-mx-auto pr-text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
            </svg>
            <span className="pr-text-sm">{__('Fix Orphans', 'prorank-seo')}</span>
          </div>
        </button>

        <button
          onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=broken'}
          className="pr-p-3 pr-text-center pr-border pr-border-gray-200 pr-rounded-lg pr-bg-white pr-transition-all pr-duration-300 hover:pr-shadow-md hover:pr-border-purple-200"
        >
          <div className="pr-space-y-2">
            <svg className="pr-w-6 pr-h-6 pr-mx-auto pr-text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z" />
            </svg>
            <span className="pr-text-sm">{__('Fix Broken', 'prorank-seo')}</span>
          </div>
        </button>

        <button
          onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=auto-linking'}
          className="pr-p-3 pr-text-center pr-border pr-border-gray-200 pr-rounded-lg pr-bg-white pr-transition-all pr-duration-300 hover:pr-shadow-md hover:pr-border-purple-200"
        >
          <div className="pr-space-y-2">
            <svg className="pr-w-6 pr-h-6 pr-mx-auto pr-text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
            </svg>
            <span className="pr-text-sm">{__('Auto Link', 'prorank-seo')}</span>
          </div>
        </button>

        <button
          onClick={() => window.location.hash = '#/on-page-seo/internal-linking?tab=analysis'}
          className="pr-p-3 pr-text-center pr-border pr-border-gray-200 pr-rounded-lg pr-bg-white pr-transition-all pr-duration-300 hover:pr-shadow-md hover:pr-border-purple-200"
        >
          <div className="pr-space-y-2">
            <svg className="pr-w-6 pr-h-6 pr-mx-auto pr-text-gray-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 002-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
            </svg>
            <span className="pr-text-sm">{__('Analyze', 'prorank-seo')}</span>
          </div>
        </button>
      </div>
    </div>
  );
};

export default LinkingDashboard;
