import { useState, useRef } from '@wordpress/element';
import { __, sprintf } from '@wordpress/i18n';
import apiFetch from '@wordpress/api-fetch';
import { download, upload, trash } from '@wordpress/icons';
import { useNotification } from '../../contexts/NotificationContext';
import PageHeader from '../../components/PageHeader';
import headerBadges from '../../utils/headerBadges';
import { Card, Button, Select, Alert, Modal } from '../../components/ui';

const ImportExport = () => {
  const [exportType, setExportType] = useState('all');
  const [importType, setImportType] = useState('all');
  const [importing, setImporting] = useState(false);
  const [exporting, setExporting] = useState(false);
  const [resetting, setResetting] = useState(false);
  const [showResetModal, setShowResetModal] = useState(false);
  const [importFile, setImportFile] = useState(null);
  const [importError, setImportError] = useState('');
  const fileInputRef = useRef(null);
  const { showNotification } = useNotification();

  // Export options
  const exportOptions = [
    { label: __('All Data', 'prorank-seo'), value: 'all' },
    { label: __('Settings Only', 'prorank-seo'), value: 'settings' },
    { label: __('Module States', 'prorank-seo'), value: 'modules' },
    { label: __('Redirects', 'prorank-seo'), value: 'redirects' },
    { label: __('SEO Meta Data', 'prorank-seo'), value: 'meta' },
  ];

  // Handle export
  const handleExport = async () => {
    setExporting(true);
    try {
      const response = await apiFetch({
        path: `/prorank-seo/v1/import-export/export?type=${exportType}`,
      });

      if (response.success) {
        // Create and download file
        const blob = new Blob([atob(response.data)], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        const link = document.createElement('a');
        link.href = url;
        link.download = response.filename || `prorank-seo-export-${exportType}-${Date.now()}.json`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(url);

        showNotification(__('Export completed successfully', 'prorank-seo'), 'success');
      }
    } catch (error) {
      showNotification(error.message || __('Export failed', 'prorank-seo'), 'error');
    } finally {
      setExporting(false);
    }
  };

  const handleFileSelect = (event) => {
    const file = event.target.files[0];
    if (file) {
      if (file.type !== 'application/json') {
        setImportError(__('Please select a valid JSON file', 'prorank-seo'));
        setImportFile(null);
        return;
      }
      setImportFile(file);
      setImportError('');
    }
  };

  const handleImport = async () => {
    if (!importFile) {
      setImportError(__('Please select a file to import', 'prorank-seo'));
      return;
    }

    setImporting(true);
    setImportError('');

    try {
      // Read file content
      const fileContent = await new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onload = (e) => resolve(e.target.result);
        reader.onerror = reject;
        reader.readAsText(importFile);
      });

      // Base64 encode for transport
      const encodedData = btoa(fileContent);

      const response = await apiFetch({
        path: '/prorank-seo/v1/import-export/import',
        method: 'POST',
        data: {
          data: encodedData,
          type: importType,
        },
      });

      if (response.success) {
        showNotification(
          response.message || __('Import completed successfully', 'prorank-seo'),
          'success'
        );

        // Reset file input
        setImportFile(null);
        if (fileInputRef.current) {
          fileInputRef.current.value = '';
        }
        
        // Reload page after successful import to reflect changes
        setTimeout(() => {
          window.location.reload();
        }, 2000);
      }
    } catch (error) {
      setImportError(error.message || __('Import failed', 'prorank-seo'));
    } finally {
      setImporting(false);
    }
  };

  const handleReset = async () => {
    setResetting(true);
    try {
      const response = await apiFetch({
        path: '/prorank-seo/v1/import-export/reset',
        method: 'DELETE',
        data: { confirm: true },
      });

      if (response.success) {
        showNotification(response.message || __('All settings have been reset', 'prorank-seo'), 'success');
        setShowResetModal(false);

        // Reload page after reset
        setTimeout(() => {
          window.location.reload();
        }, 2000);
      }
    } catch (error) {
      showNotification(__('Reset failed', 'prorank-seo'), 'error');
    } finally {
      setResetting(false);
    }
  };

  return (
    <div className="pr-min-h-screen pr-bg-gray-50">
      <PageHeader
        title={__('Import & Export', 'prorank-seo')}
        subtitle={__('Transfer settings, module states, and metadata between sites.', 'prorank-seo')}
        badges={headerBadges['import-export']}
      />
      <div className="pr-p-8 pr-space-y-6">
        <div className="pr-grid pr-grid-cols-1 lg:pr-grid-cols-2 pr-gap-6 pr-mb-6">
        {/* Export Section */}
        <Card className="pr-transition-all pr-duration-300">
          <div className="pr-px-6 pr-py-4 pr-border-b pr-border-gray-200 pr-bg-gray-50 pr-transition-all pr-duration-300">
            <h3 className="pr-text-base pr-font-semibold pr-text-gray-900">{__('Export Settings', 'prorank-seo')}</h3>
          </div>
          <div className="pr-p-6">
            <p>
              {__(
                'Export your ProRank SEO settings to a JSON file. This file can be imported into another WordPress site.',
                'prorank-seo'
              )}
            </p>

            <Select
              label={__('Export Type', 'prorank-seo')}
              value={exportType}
              options={exportOptions}
              onChange={setExportType}
              description={__('Choose what data to include in the export', 'prorank-seo')}
              className="pr-mb-4 pr-transition-all pr-duration-300"
            />

            <div className="pr-mb-4">
              {exportType === 'all' && (
                <Alert variant="info" className="pr-transition-all pr-duration-300">
                  <p>
                    {__(
                      'This will export all settings, module states, redirects, and SEO meta data.',
                      'prorank-seo'
                    )}
                  </p>
                </Alert>
              )}
              {exportType === 'settings' && (
                <p>
                  {__(
                    'Only plugin settings will be exported. Module states and content data will not be included.',
                    'prorank-seo'
                  )}
                </p>
              )}
              {exportType === 'modules' && (
                <p>
                  {__(
                    'Only module enabled/disabled states and their settings will be exported.',
                    'prorank-seo'
                  )}
                </p>
              )}
              {exportType === 'redirects' && (
                <p>{__('Only redirect rules will be exported.', 'prorank-seo')}</p>
              )}
              {exportType === 'meta' && (
                <p>
                  {__(
                    'Only SEO meta data (titles, descriptions, etc.) for posts and pages will be exported.',
                    'prorank-seo'
                  )}
                </p>
              )}
            </div>

            <Button
              variant="primary"
              onClick={handleExport}
              disabled={exporting === true}
              className="pr-transition-all pr-duration-300"
            >
              {exporting ? __('Exporting…', 'prorank-seo') : __('Export', 'prorank-seo')}
            </Button>
          </div>
        </Card>

        {/* Import Section */}
        <Card className="pr-transition-all pr-duration-300">
          <div className="pr-px-6 pr-py-4 pr-border-b pr-border-gray-200 pr-bg-gray-50 pr-transition-all pr-duration-300">
            <h3 className="pr-text-base pr-font-semibold pr-text-gray-900">{__('Import Settings', 'prorank-seo')}</h3>
          </div>
          <div className="pr-p-6">
            <p>
              {__(
                'Import ProRank SEO settings from a JSON file exported from another site.',
                'prorank-seo'
              )}
            </p>

            <div className="pr-flex pr-items-center pr-gap-3 pr-mb-4">
              <input
                ref={fileInputRef}
                type="file"
                accept=".json"
                onChange={handleFileSelect}
                style={{ display: 'none' }}
                id="prorank-import-file"
              />
              <label htmlFor="prorank-import-file">
                <Button
                  variant="secondary"
                  icon={upload}
                  onClick={() => fileInputRef.current?.click()}
                >
                  {__('Choose File', 'prorank-seo')}
                </Button>
              </label>
              {importFile && (
                <span className="pr-text-sm pr-text-gray-700 pr-font-medium">
                  {/* translators: %s: placeholder description */}
                  {sprintf(__('Selected: %s', 'prorank-seo'), importFile.name)}
                </span>
              )}
            </div>

            <Select
              label={__('Import Type', 'prorank-seo')}
              value={importType}
              options={exportOptions}
              onChange={setImportType}
              description={__('Choose what data to import from the file', 'prorank-seo')}
              className="pr-mb-4 pr-transition-all pr-duration-300"
            />

            {importError && (
              <Alert variant="error" className="pr-mb-4 pr-transition-all pr-duration-300">
                <p>{importError}</p>
              </Alert>
            )}

            <Alert variant="warning" className="pr-mb-4 pr-transition-all pr-duration-300">
              <p>
                {__(
                  'Warning: Importing will overwrite existing settings. This action cannot be undone.',
                  'prorank-seo'
                )}
              </p>
            </Alert>

            <Button
              variant="primary"
              onClick={handleImport}
              disabled={importing || !importFile}
              className="pr-transition-all pr-duration-300"
            >
              {importing ? __('Importing…', 'prorank-seo') : __('Import', 'prorank-seo')}
            </Button>
          </div>
        </Card>
      </div>

      {/* Reset Section - Full Width */}
      <Card className="pr-transition-all pr-duration-300">
          <div className="pr-px-6 pr-py-4 pr-border-b pr-border-gray-200 pr-bg-gray-50 pr-transition-all pr-duration-300">
            <h3 className="pr-text-base pr-font-semibold pr-text-gray-900">{__('Reset Settings', 'prorank-seo')}</h3>
          </div>
          <div className="pr-p-6">
            <Alert variant="error" className="pr-mb-4 pr-transition-all pr-duration-300">
              <strong>{__('Danger Zone:', 'prorank-seo')}</strong>{' '}
              {__('This action is irreversible!', 'prorank-seo')}
            </Alert>

            <p>
              {__(
                'Reset all ProRank SEO settings to their default values. This will remove all settings, module configurations, and custom data.',
                'prorank-seo'
              )}
            </p>

            <p>
              <strong>{__('This will permanently delete:', 'prorank-seo')}</strong>
            </p>
            <ul>
              <li>{__('All plugin settings', 'prorank-seo')}</li>
              <li>{__('Module configurations', 'prorank-seo')}</li>
              <li>{__('Redirect rules', 'prorank-seo')}</li>
              <li>{__('Custom SEO data', 'prorank-seo')}</li>
            </ul>

            <Button
              variant="danger"
              onClick={() => setShowResetModal(true)}
              className="pr-transition-all pr-duration-300"
            >
              {__('Reset All Settings', 'prorank-seo')}
            </Button>
          </div>
        </Card>

      {/* Reset Confirmation Modal */}
      {showResetModal && (
        <Modal
          isOpen={showResetModal}
          onClose={() => setShowResetModal(false)}
          title={__('Confirm Reset', 'prorank-seo')}
          className="pr-transition-all pr-duration-300"
        >
          <p>
            {__(
              'Are you absolutely sure you want to reset all ProRank SEO settings?',
              'prorank-seo'
            )}
          </p>
          <p>
            <strong>{__('This will permanently delete:', 'prorank-seo')}</strong>
          </p>
          <ul>
            <li>{__('All plugin settings', 'prorank-seo')}</li>
            <li>{__('Module configurations', 'prorank-seo')}</li>
            <li>{__('Redirect rules', 'prorank-seo')}</li>
            <li>{__('Custom SEO data', 'prorank-seo')}</li>
          </ul>
          <p>
            <strong>{__('This action cannot be undone!', 'prorank-seo')}</strong>
          </p>

          <div className="pr-flex pr-gap-2 pr-justify-end pr-mt-6">
            <Button variant="secondary" onClick={() => setShowResetModal(false)} className="pr-transition-all pr-duration-300">
              {__('Cancel', 'prorank-seo')}
            </Button>
            <Button
              variant="danger"
              onClick={handleReset}
              disabled={resetting === true}
              className="pr-transition-all pr-duration-300"
            >
              {resetting
                ? __('Resetting…', 'prorank-seo')
                : __('Yes, Reset Everything', 'prorank-seo')}
            </Button>
          </div>
        </Modal>
      )}
      </div>
    </div>
  );
};

export default ImportExport;
